/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.classifier.assessment;

import java.util.ArrayList;
import java.util.Collection;

import de.jstacs.DataType;
import de.jstacs.NonParsableException;
import de.jstacs.data.Sample.PartitionMethod;
import de.jstacs.parameters.EnumParameter;
import de.jstacs.parameters.SimpleParameter;
import de.jstacs.parameters.SimpleParameter.IllegalValueException;
import de.jstacs.parameters.validation.NumberValidator;
import de.jstacs.results.CategoricalResult;
import de.jstacs.results.NumericalResult;
import de.jstacs.results.Result;

/**
 * @author Jens Keilwagen
 */
public class Sampled_RepeatedHoldOutAssessParameterSet extends ClassifierAssessmentAssessParameterSet
{

	/**
	 * inherited from <code>ClassifierAssessmentAssessParameterSet</code>
	 */
	protected Sampled_RepeatedHoldOutAssessParameterSet( Class alternativeInstanceClass )
	{
		super( alternativeInstanceClass );
	}

	/**
	 * inherited from <code>ClassifierAssessmentAssessParameterSet</code>
	 */
	public Sampled_RepeatedHoldOutAssessParameterSet() throws UnsupportedOperationException
	{
		super();
	}

	/**
	 * inherited from <code>ClassifierAssessmentAssessParameterSet</code>
	 */
	public Sampled_RepeatedHoldOutAssessParameterSet( StringBuffer representation ) throws NonParsableException
	{
		super( representation );
	}

	/**
	 * 
	 * @param dataSplitMethod
	 *            defines the method used to split user-supplied data into
	 * 							<code>k</code> mutually exclusive, random-splits
	 * 							(available options are: {@link PartitionMethod#PARTITION_BY_NUMBER_OF_ELEMENTS} and {@link PartitionMethod#PARTITION_BY_NUMBER_OF_SYMBOLS}) See docu of <code>Sample</code>
	 * 							for further details.
	 * 
	 * @param elementLength
	 *            defines the length of elements (sequences) the classifiers to be assessed are able to classify
	 * 
	 * @param exceptionIfMPNotComputable
	 *            a <code>Sampled_RepeatedHoldOutParameterSet</code> is used in combination with an
	 *            <code>MeasureParameters</code>-object to call <code>assess</code>-methods of
	 *            <code>Sampled_RepeatedHoldOutExperiment</code>s. If <code>exceptionIfMPNotComputable=true</code>
	 *            then an expection is thrown in case of a selected measure-parameters that could not be computed.
	 * 
	 * @param repeats
	 *            the number of repeates of each iteration (mutually exclusive, randomly split data to obtain test- and
	 *            train-data-sets, train classifiers using train-data-sets and test them using test-data-sets) of that
	 *            <code>RepeatedHoldOutExperiment</code> this <code>Sampled_RepatedHoldOutAssessParameterSet</code>
	 *            is used with
	 * 
	 * @param referenceClass
	 *            the index of the class for which the complete data set is used
	 * 
	 * @param percentage
	 *            the percentage of the <code>referenceClass</code>-data that should be used as test-data in each
	 *            iteration
	 * 
	 * @param sameLength
	 *            if <code>true</code> then for test respectively train data set the sequences of the non-reference
	 *            classes have the same length as the corresponding sequence of the reference class
	 * 
	 * @throws IllegalValueException
	 *             is thrown in case of out-of-range or invalid given parameters
	 */
	public Sampled_RepeatedHoldOutAssessParameterSet( PartitionMethod dataSplitMethod, int elementLength,
			boolean exceptionIfMPNotComputable, int repeats, int referenceClass, double percentage, boolean sameLength )
			throws IllegalValueException
	{
		super( elementLength, exceptionIfMPNotComputable );

		this.parameters.get( 2 ).setValue( repeats );
		this.parameters.get( 3 ).setValue( referenceClass );
		this.parameters.get( 4 ).setValue( percentage );
		this.parameters.get( 5 ).setValue( dataSplitMethod );
		this.parameters.get( 6 ).setValue( sameLength );
	}

	@Override
	protected void initializeMyParametersArrayList()
	{
		initParameterList( 6 );
	}

	@Override
	protected void loadParameters() throws Exception
	{
		super.loadParameters();

		// 2-k
		this.parameters.add( new SimpleParameter( DataType.INT, "repeats",
				"Determines how often the procedure of train/test classifers using random created "
						+ "train- and test-data should be repeated.", true, new NumberValidator<Integer>( 1,
						Integer.MAX_VALUE ) ) );

		// 3-reference class
		this.parameters.add( new SimpleParameter( DataType.INT, "reference class",
				"the index of the class for which the complete data set is partitioned in an iteration", true,
				new NumberValidator<Integer>( 0, Integer.MAX_VALUE ) ) );

		// 4-percentage
		this.parameters.add( new SimpleParameter( DataType.DOUBLE, "testDataPercentage",
				"the percentage of of the foreground data set  that is used in each iterations as test-data-set", true,
				new NumberValidator<Double>( 0d, 1d ) ) );

		// 5-dataSplitMethod
		this.parameters.add( new EnumParameter(PartitionMethod.class,"The method used to compute the percentages of the partitions",true ) );

		// 6-sameLength
		this.parameters.add( new SimpleParameter( DataType.BOOLEAN, "sameLength",
				"if true then for test respectively train data set the sequences of the non-reference"
						+ "classes have the same length as the corresponding sequence of the reference class", true ) );
	}

	@Override
	public String getInstanceName()
	{
		return getClass().getSimpleName();
	}

	@Override
	public String getInstanceComment()
	{
		return "This " + getInstanceName() + " may be used to pass parameters into "
				+ RepeatedHoldOutExperiment.class.getSimpleName() + "'s method assess().";
	}

	/**
	 * @return the repeates defined by this <code>Sampled_RepatedHoldOutAssessParameterSet</code> (repeats defines how
	 *         many iterations (train and test classifiers) of that <code>Sampled_RepeatedHoldOutExperiment</code>
	 *         this <code>Sampled_RepeatedHoldOutAssessParameterSet</code> is used with are performed)
	 */
	public int getRepeats()
	{
		return (Integer) (this.getParameterAt( 2 ).getValue());
	}

	/**
	 * @return the index of the reference class
	 */
	public int getReferenceClass()
	{
		return (Integer) (this.getParameterAt( 3 ).getValue());
	}

	/**
	 * @return the percentage of user supplied-data that is used in each iterations as test-data-set
	 */
	public double getPercent()
	{
		return (Double) (this.getParameterAt( 4 ).getValue());
	}

	/**
	 * @return 	The {@link de.jstacs.data.Sample.PartitionMethod} defining how the mutually exclusive, random-splits
	 * 			of user-supplied data are generated. See class <code>Sample</code> for further details.
	 */
	public PartitionMethod getDataSplitMethod
	(){
	return (PartitionMethod) ((EnumParameter) getParameterAt(5)).getValue();
	}
	
	/**
	 * @return returns <code>true</code> if for test respectively train data set the sequences of the non-reference
	 *         classes have the same length as the corresponding sequence of the reference class
	 */
	public boolean sameLength()
	{
		return (Boolean) (getParameterAt( 6 ).getValue());
	}

	@Override
	public Collection<Result> getAnnotation()
	{
		ArrayList<Result> l = new ArrayList<Result>( 3 );
		l.add( new NumericalResult( "repeats", "The number of iterations", getRepeats() ) );
		l.add( new NumericalResult( "reference class",
				"The index of the class for which the complete data set is partitioned in an iteration",
				getReferenceClass() ) );
		l.add( new NumericalResult( "percentage",
				"The percentage of the entire data of the reference class, that was used in an assessment",
				getPercent() ) );
		l.add( new CategoricalResult( "dataSplitMethod",
				"Describes how data should be splitted in ClassifierAssessment.evaluateClassifier())",
				getDataSplitMethod().name() ) );
		l.add( new CategoricalResult( "sameLength",
				"if true then for test respectively train data set the sequences of the non-reference"
				+ "classes have the same length as the corresponding sequence of the reference class", sameLength() ) );
		return l;
	}

}
