/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.classifier.scoringFunctionBased;

import de.jstacs.DataType;
import de.jstacs.NonParsableException;
import de.jstacs.algorithms.optimization.Optimizer;
import de.jstacs.data.AlphabetContainer;
import de.jstacs.parameters.CollectionParameter;
import de.jstacs.parameters.InstanceParameterSet;
import de.jstacs.parameters.SimpleParameter;
import de.jstacs.parameters.validation.NumberValidator;

/**
 * The parameter set for any CL classifier.
 * 
 * @author Jens Keilwagen
 */
public class ScoreClassifierParameterSet extends InstanceParameterSet
{
	private static final String[] algorithmStrings = new String[]{ "steepest descent", "conjugate gradients (F., R.)",
			"conjugate gradients (P., R. positive)", "quasi newton (D., F., P.)", "quasi newton (B., F., G., S.)",
			"limited memory quasi newton (B., F., G., S.; n=3)", "limited memory quasi newton (B., F., G., S.; n=4)",
			"limited memory quasi newton (B., F., G., S.; n=5)", "limited memory quasi newton (B., F., G., S.; n=6)",
			"limited memory quasi newton (B., F., G., S.; n=7)", "limited memory quasi newton (B., F., G., S.; n=8)",
			"limited memory quasi newton (B., F., G., S.; n=9)", "limited memory quasi newton (B., F., G., S.; n=10)" };

	private static final Byte[] algorithms = new Byte[]{ Optimizer.STEEPEST_DESCENT, Optimizer.CONJUGATE_GRADIENTS_FR,
			Optimizer.CONJUGATE_GRADIENTS_PRP, Optimizer.QUASI_NEWTON_DFP, Optimizer.QUASI_NEWTON_BFGS, (byte) 3,
			(byte) 4, (byte) 5, (byte) 6, (byte) 7, (byte) 8, (byte) 9, (byte) 10 };

	/**
	 * The default constructor.
	 * 
	 * @param simple whether the AlphabetContainer shall be simple.
	 * @param variableLength whether the corresponding classifier can handle sequences of arbitrary length
	 */
	public ScoreClassifierParameterSet( boolean simple, boolean variableLength )
	{
		super( ScoreClassifier.class, true, simple, variableLength );
	}
	
	/**
	 * This is the constructor for {@link de.jstacs.Storable}.
	 * 
	 * @param xml the xml representation
	 * 
	 * @throws NonParsableException if the representation could not be parsed.
	 */
	public ScoreClassifierParameterSet( StringBuffer xml ) throws NonParsableException
	{
		super( xml );
	}

	/**
	 * The constructor for a simple, instantiated parameter set.
	 * 
	 * @param alphabet
	 *            the alphabet
	 * @param length
	 *            the length of the sequences, 0 for homogeneous
	 * @param algo
	 *            the choice of algorithm
	 * @param eps
	 *            the threshold for stopping the algorithm
	 * @param lineps
	 *            the threshold for stopping the line search in the algorithm
	 * @param startD
	 *            the start distance for the line search in the algorithm
	 * @param free
	 *            indicates whether only the free parameters or all parameters should be used
	 * @param plugIn
	 * 			  indicates whether to use plug-in parameters or not
	 * 
	 * @throws Exception
	 */
	public ScoreClassifierParameterSet( AlphabetContainer alphabet, int length, byte algo, double eps, double lineps, double startD, boolean free, boolean plugIn ) throws Exception
	{
		super( ScoreClassifier.class, alphabet, length, length == 0 );
		parameters.get( 0 ).setValue( algorithmStrings[getIndex( algorithmStrings, algorithms, algo, false )] );
		parameters.get( 1 ).setValue( eps );
		parameters.get( 2 ).setValue( lineps );
		parameters.get( 3 ).setValue( startD );
		parameters.get( 4 ).setValue( free );
		parameters.get( 5 ).setValue( plugIn );
	}

	@Override
	protected void loadParameters() throws Exception
	{
		initParameterList();
		parameters.add( new CollectionParameter( DataType.BYTE, algorithmStrings, algorithms, "algorithm",
				"the algorithm that should be used for numerical optimization", true ) );
		parameters.add( new SimpleParameter( DataType.DOUBLE, "epsilon",
				"the threshold for stopping the numerical training", true, new NumberValidator<Double>( 0d,
						Double.MAX_VALUE ) ) );
		parameters.add( new SimpleParameter( DataType.DOUBLE, "line epsilon",
				"the threshold for stopping the line search in the numerical training", true, new NumberValidator<Double>( 0d,
						Double.MAX_VALUE ) ) );
		parameters.add( new SimpleParameter( DataType.DOUBLE, "start distance",
				"the start distance for the line search in the numerical training", true, new NumberValidator<Double>( 0d,
						Double.MAX_VALUE ) ) );
		parameters.add( new SimpleParameter( DataType.BOOLEAN, "free parameters",
				"Indicates whether only the free parameters or all parameters should be used.", true,
				new Boolean( true ) ) );
		parameters.add( new SimpleParameter( DataType.BOOLEAN, "plugIn parameter",
				"Indicates whether special plugIn parameters or the zero vector should be used as start parameters. For non-concave problems it is highly recommended to use plugIn parameters.", true,
				new Boolean( true ) ) );
	}


	/**
	 * This method returns <code>true</code> if only the free parameters shall be used.
	 * 
	 * @return <code>true</code> if only the free parameters shall be used
	 */
	public boolean useOnlyFreeParameter()
	{
		return (Boolean) getParameterAt( 4 ).getValue();
	}
	
	public String getInstanceName()
	{
		return getClass().getSimpleName();
	}

	public String getInstanceComment()
	{
		return "holds the parameters for a score classifier";
	}
}
