/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.io;

import java.lang.reflect.Constructor;

import de.jstacs.DataType;
import de.jstacs.InstantiableFromParameterSet;
import de.jstacs.parameters.Parameter;
import de.jstacs.parameters.ParameterException;
import de.jstacs.parameters.ParameterSet;

/**
 * Class that extracts values from <code>Parameter</code>s und creates instances of <code>InstantiableFromParameterSet</code>s
 * from a <code>ParameterSet</code>.
 * 
 * @author Jan Grau
 *
 */
public class ParameterSetParser
{

	/**
	 * Returns the <code>int</code> which is the value of the <code>Parameter par</code>.
	 * @param par the <code>Parameter</code>
	 * @return the <code>int</code> value
	 * @throws WrongParameterTypeException is thrown if <code>par</code> is not an <code>int-Parameter</code>, 
	 * i.e. its datatype is not <code>Parameter.INT</code>
	 */
	public static int getIntFromParameter( Parameter par ) throws WrongParameterTypeException
	{
		if( par.getDatatype() == DataType.INT )
		{
			return (Integer) par.getValue();
		}
		else
		{
			throw new WrongParameterTypeException( "Parameter " + par.getName() + " is not of type int." );
		}
	}

	/**
	 * Returns the <code>float</code> which is the value of the <code>Parameter par</code>.
	 * @param par the <code>Parameter</code>
	 * @return the <code>float</code> value
	 * @throws WrongParameterTypeException is thrown if <code>par</code> is not an <code>float-Parameter</code>, 
	 * i.e. its datatype is not <code>Parameter.FLOAT</code>
	 */
	public static float getFloatFromParameter( Parameter par ) throws WrongParameterTypeException
	{
		if( par.getDatatype() == DataType.FLOAT )
		{
			return (Float) par.getValue();
		}
		else
		{
			throw new WrongParameterTypeException( "Parameter " + par.getName() + " is not of type float." );
		}
	}

	/**
	 * Returns the <code>double</code> which is the value of the <code>Parameter par</code>.
	 * @param par the <code>Parameter</code>
	 * @return the <code>double</code> value
	 * @throws WrongParameterTypeException is thrown if <code>par</code> is not an <code>double-Parameter</code>, 
	 * i.e. its datatype is not <code>Parameter.DOUBLE</code>
	 */
	public static double getDoubleFromParameter( Parameter par ) throws WrongParameterTypeException
	{
		if( par.getDatatype() == DataType.DOUBLE )
		{
			return (Double) par.getValue();
		}
		else
		{
			throw new WrongParameterTypeException( "Parameter " + par.getName() + " is not of type double." );
		}
	}

	/**
	 * Returns the <code>short</code> which is the value of the <code>Parameter par</code>.
	 * @param par the <code>Parameter</code>
	 * @return the <code>short</code> value
	 * @throws WrongParameterTypeException is thrown if <code>par</code> is not an <code>short-Parameter</code>, 
	 * i.e. its datatype is not <code>Parameter.SHORT</code>
	 */
	public static short getShortFromParameter( Parameter par ) throws WrongParameterTypeException
	{
		if( par.getDatatype() == DataType.SHORT )
		{
			return (Short) par.getValue();
		}
		else
		{
			throw new WrongParameterTypeException( "Parameter " + par.getName() + " is not of type short." );
		}
	}

	/**
	 * Returns the <code>long</code> which is the value of the <code>Parameter par</code>.
	 * @param par the <code>Parameter</code>
	 * @return the <code>long</code> value
	 * @throws WrongParameterTypeException is thrown if <code>par</code> is not an <code>long-Parameter</code>, 
	 * i.e. its datatype is not <code>Parameter.LONG</code>
	 */
	public static long getLongFromParameter( Parameter par ) throws WrongParameterTypeException
	{
		if( par.getDatatype() == DataType.LONG )
		{
			return (Long) par.getValue();
		}
		else
		{
			throw new WrongParameterTypeException( "Parameter " + par.getName() + " is not of type long." );
		}
	}

	/**
	 * Returns the <code>byte</code> which is the value of the <code>Parameter par</code>.
	 * @param par the <code>Parameter</code>
	 * @return the <code>byte</code> value
	 * @throws WrongParameterTypeException is thrown if <code>par</code> is not an <code>byte-Parameter</code>, 
	 * i.e. its datatype is not <code>Parameter.BYTE</code>
	 */
	public static byte getByteFromParameter( Parameter par ) throws WrongParameterTypeException
	{
		if( par.getDatatype() == DataType.BYTE )
		{
			return (Byte) par.getValue();
		}
		else
		{
			throw new WrongParameterTypeException( "Parameter " + par.getName() + " is not of type byte." );
		}
	}

	/**
	 * Returns the <code>boolean</code> which is the value of the <code>Parameter par</code>.
	 * @param par the <code>Parameter</code>
	 * @return the <code>boolean</code> value
	 * @throws WrongParameterTypeException is thrown if <code>par</code> is not an <code>boolean-Parameter</code>, 
	 * i.e. its datatype is not <code>Parameter.BOOLEAN</code>
	 */
	public static boolean getBooleanFromParameter( Parameter par ) throws WrongParameterTypeException
	{
		if( par.getDatatype() == DataType.BOOLEAN )
		{
			return (Boolean) par.getValue();
		}
		else
		{
			throw new WrongParameterTypeException( "Parameter " + par.getName() + " is not of type boolean." );
		}
	}

	/**
	 * Returns the <code>String</code> which is the value of the <code>Parameter par</code>.
	 * @param par the <code>Parameter</code>
	 * @return the <code>String</code> value
	 * @throws WrongParameterTypeException is thrown if <code>par</code> is not an <code>String-Parameter</code>, 
	 * i.e. its datatype is not <code>Parameter.STRING</code>
	 */
	public static String getStringFromParameter( Parameter par ) throws WrongParameterTypeException
	{
		if( par.getDatatype() == DataType.STRING )
		{
			return (String) par.getValue();
		}
		else
		{
			throw new WrongParameterTypeException( "Parameter " + par.getName() + " is not of type String." );
		}
	}

	/**
	 * Returns an instance of a subclass of <code>InstantiableFromParameterSet</code> that can be instantiated by
	 * the <code>ParameterSet pars</code>. The instance-class is taken from <code>pars</code> via the <code>getInstanceClass()</code>-method.
	 * @param pars the <code>ParameterSet</code>
	 * @return the instance
	 * @throws NotInstantiableException thrown if <code>getInstanceClass()</code> of <code>pars</code> is <code>null</code>, 
	 * could not be found or cannot be instantiated from <code>pars</code>
	 */
	public static InstantiableFromParameterSet getInstanceFromParameterSet( ParameterSet pars ) throws NotInstantiableException
	{
		Class cl = pars.getInstanceClass();
		if( cl == null )
		{
			throw new NotInstantiableException( "An instance class must be specified in the ParameterSet." );
		}
		else
		{
			return getInstanceFromParameterSet( pars, cl );
		}
	}

	/**
	 * Returns an instance of a subclass of <code>InstantiableFromParameterSet</code> that can be instantiated by
	 * the <code>ParameterSet pars</code>. The instance-class is taken from <code>instanceClass</code>.
	 * @param pars the <code>ParameterSet</code>
	 * @param instanceClass the class that shall be instantiated
	 * @return the instance
	 * @throws NotInstantiableException thrown if <code>instanceClass</code> 
	 * could not be found or cannot be instantiated from <code>pars</code>
	 */
	public static InstantiableFromParameterSet getInstanceFromParameterSet( ParameterSet pars, Class instanceClass )
			throws NotInstantiableException
	{
		Class parClass = pars.getClass();
		Constructor construct = null;
		while( construct == null && parClass != Object.class )
		{
			try
			{
				construct = instanceClass.getConstructor( new Class[] { parClass } );
			} catch( Exception e )
			{
				construct = null;
				parClass = parClass.getSuperclass();
			}
		}
		if( construct == null )
		{
			throw new NotInstantiableException( "No appropriate constructor found." );
		}
		else
		{
			try
			{
				return (InstantiableFromParameterSet) construct.newInstance( new Object[] { pars } );
			} catch( Exception e )
			{
				throw new NotInstantiableException( e.getCause().getMessage() );
			}
		}
	}

	/**
	 * An <code>Exception</code> that is thrown if an instance of some class could not be created.
	 * @author Jan Grau
	 *
	 */
	public static class NotInstantiableException extends ParameterException
	{

		private static final long serialVersionUID = 1L;

		/**
		 * Creates a new instance of a <code>NotInstantiableException</code> from an error-message.
		 * @param msg the message
		 */
		public NotInstantiableException( String msg )
		{
			super( msg );
		}

	}

	/**
	 * An <code<Exception</code> that is thrown if the datatype of a <code>Parameter</code>
	 * is not appropriate for some purpose.
	 * @author Jan Grau
	 *
	 * @see de.jstacs.parameters.Parameter
	 */
	public static class WrongParameterTypeException extends ParameterException
	{

		private static final long serialVersionUID = 1L;

		/**
		 * Creates a new instance of a <code>WrongParameterTypeException</code> from an error-message
		 * @param msg the message
		 */
		public WrongParameterTypeException( String msg )
		{
			super( msg );
		}

	}

}
