/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.models.discrete;

import de.jstacs.InstantiableFromParameterSet;
import de.jstacs.NonParsableException;
import de.jstacs.NotTrainedException;
import de.jstacs.data.Sequence;
import de.jstacs.io.XMLParser;
import de.jstacs.models.AbstractModel;
import de.jstacs.parameters.ParameterSet;

/**
 * This is the main class for all discrete graphical models (DGM)
 * 
 * @author Jens Keilwagen
 * 
 * @see DGMParameterSet
 */
public abstract class DiscreteGraphicalModel extends AbstractModel implements InstantiableFromParameterSet
{
	private static final long serialVersionUID = 1L;

	/**
	 * The current parameter set of the model
	 */
	protected DGMParameterSet params;

	/**
	 * Indicates whether the model is trained or not
	 */
	protected boolean trained;

	/**
	 * The default constructor.
	 * 
	 * @param params the parameter set
	 * 
	 * @throws CloneNotSupportedException if the parameter set could not be cloned
	 * @throws IllegalArgumentException if the parameter set is not instantiated
	 * @throws NonParsableException if the parameter set is not parsable
	 */
	public DiscreteGraphicalModel( DGMParameterSet params ) throws CloneNotSupportedException,
			IllegalArgumentException, NonParsableException
	{
		super( params.getAlphabet(), params.getLength() );
		fromParameterSet( params );
	}

	/**
	 * The constructor for a model in xml format.
	 * 
	 * @param representation
	 *            the model in xml format
	 * 
	 * @throws NonParsableException
	 *             if the StringBuffer could not be parsed
	 */
	public DiscreteGraphicalModel( StringBuffer representation ) throws NonParsableException
	{
		super( representation );
	}

	public DiscreteGraphicalModel clone() throws CloneNotSupportedException
	{
		DiscreteGraphicalModel clone = (DiscreteGraphicalModel) super.clone();
		clone.params = params.clone();
		return clone;
	}

	public final DGMParameterSet getCurrentParameterSet() throws Exception
	{
		return params.clone();
	}

	/**
	 * Returns a short description of the model the was given by the user in the parameter set.
	 * 
	 * @return a short description
	 */
	public final String getDescription()
	{
		return (String) params.getParameterAt( 1 ).getValue();
	}

	/**
	 * This method return the ess that is used in this model.
	 * 
	 * @return the ESS
	 */
	public final double getESS()
	{
		return (Double) params.getParameterAt( 0 ).getValue();
	}

	private void fromParameterSet( ParameterSet parameters ) throws CloneNotSupportedException,
			IllegalArgumentException, NonParsableException
	{
		DGMParameterSet p = (DGMParameterSet) parameters;
		if( !p.hasDefaultOrIsSet() )
		{
			throw new IllegalArgumentException( "Parameters were not correct instantiated" );
		}
		set( p, false );
	}

	protected final void fromXML( StringBuffer representation ) throws NonParsableException
	{
		StringBuffer erg = XMLParser.extractForTag( representation, getXMLTag() );
		try
		{
			set( (DGMParameterSet) XMLParser.extractStorableForTag( erg, "ParameterSet" ), XMLParser
					.extractBooleanForTag( erg, "trained" ) );
		}
		catch( CloneNotSupportedException e )
		{
			NonParsableException n = new NonParsableException( e.getMessage() );
			n.setStackTrace( e.getStackTrace() );
			throw n;
		}
		setFurtherModelInfos( erg );
	}

	public final boolean isTrained()
	{
		return trained;
	}

	public String toString()
	{
		return getDescription();
	}

	public final StringBuffer toXML()
	{
		StringBuffer erg = new StringBuffer();
		XMLParser.appendStorableWithTags( erg, params, "ParameterSet" );
		XMLParser.appendBooleanWithTags( erg, trained, "trained" );
		StringBuffer help = getFurtherModelInfos();
		if( help != null )
		{
			erg.append( help );
		}
		XMLParser.addTags( erg, getXMLTag() );
		return erg;
	}

	/**
	 * Checks some conditions on a sequence.
	 * 
	 * @param sequence
	 *            the sequence
	 * @param startpos
	 *            the startposition
	 * @param endpos
	 *            the endposition
	 * 
	 * @throws NotTrainedException
	 *             if the model is not trained
	 * @throws IllegalArgumentException
	 *             if some constraints are not fulfilled
	 */
	protected void check( Sequence sequence, int startpos, int endpos ) throws NotTrainedException,
			IllegalArgumentException
	{
		if( !trained )
		{
			throw new NotTrainedException();
		}
		else if( !alphabets.checkConsistency( sequence.getAlphabetContainer().getSubContainer( startpos, endpos-startpos+1 ) ) )
		{
			throw new IllegalArgumentException( "This sequence is not possible with the given alphabet." );
		}
		else if( startpos < 0 )
		{
			throw new IllegalArgumentException( "This startposition is impossible. Try: 0 <= startposition" );
		}
	}

	/**
	 * @return the model infos like parameters of the distibution ... in xml format
	 * 
	 * @see DiscreteGraphicalModel#toXML()
	 */
	protected abstract StringBuffer getFurtherModelInfos();

	/**
	 * @return the XML tag that is used in fromXML() and toXML()
	 * 
	 * @see DiscreteGraphicalModel#fromXML(StringBuffer)
	 * @see DiscreteGraphicalModel#toXML()
	 */
	protected abstract String getXMLTag();

	/**
	 * This method replaces the internal model infos with those from the StringBuffer.
	 * 
	 * @param xml
	 *            contains the model infos like parameters of the distibution ... in xml format
	 * 
	 * @throws NonParsableException
	 *             if the StringBuffer could not be parsed
	 * 
	 * @see DiscreteGraphicalModel#fromXML(StringBuffer)
	 */
	protected abstract void setFurtherModelInfos( StringBuffer xml ) throws NonParsableException;

	/**
	 * Sets the parameters as internal parameters and does some essential computations. Used in
	 * fromParameterSet-methods.
	 * 
	 * @param params
	 *            the new ParameterSet
	 * @param trained
	 *            the indicator for the model
	 * 
	 * @throws CloneNotSupportedException
	 *             if the parmeterSet could not be cloned
	 * @throws NonParsableException
	 *             if the parameters of the model could not be parsed
	 */
	protected void set( DGMParameterSet params, boolean trained ) throws CloneNotSupportedException,
			NonParsableException
	{
		if( !params.getInstanceClass().equals(this.getClass()) )
		{
			throw new RuntimeException( "The parameters set can not be used for this class." );
		}
		this.params = params.clone();
		alphabets = params.getAlphabet();
		length = params.getLength();
		this.trained = trained;
	}
}
