package de.jstacs.parameters;

import de.jstacs.DataType;
import de.jstacs.NonParsableException;
import de.jstacs.io.XMLParser;
import de.jstacs.parameters.SimpleParameter.IllegalValueException;

/**
 * This class implements a {@link CollectionParameter} based on an Enum. Internally it is based on string, i.e. the names of the enum constants.
 * The methods {@link Parameter#setDefault(Object)} and {@link Parameter#setValue(Object)} can be used with Strings or with the Enum constants.
 * 
 * @author Jens Keilwagen
 */
public class EnumParameter extends CollectionParameter
{
	private Class<? extends Enum> enumInstance;
	private Enum[] enumConstants; 
	                       
	private static String[] getKeys( Class<? extends Enum> e ) {
		Enum[] all = e.getEnumConstants();
		String[] keys = new String[all.length];
		for( int i = 0; i < all.length; i++ ) {
			keys[i] = all[i].name();
		}
		return keys;
	}
	
	/**
	 * The main constructor.
	 * 
	 * @param enumInstance the enum class, e.q. {@link de.jstacs.data.Sample.PartitionMethod}.class
	 * @param comment the comment for this parameter
	 * @param required <code>true</code> if this <code>EnumParameter</code> is required
	 * @throws ParameterException is never thrown but exists due to the class hierarchy
	 */
	public EnumParameter( Class<? extends Enum> enumInstance, String comment, boolean required ) throws ParameterException{
		super( DataType.STRING, getKeys( enumInstance ), getKeys( enumInstance ),
				enumInstance.getSimpleName(), comment, required );
		this.enumInstance = enumInstance;
		enumConstants = enumInstance.getEnumConstants();
	}
	
	/**
	 * Restores an instance of <code>EnumParameter</code> from a string representation
	 * 
	 * @param representation the string representation
	 * @throws NonParsableException if the representation could not be parsed to a <code>CollectionParameter</code>, a <code>NonParsableException</code> is thrown
	 */
	public EnumParameter(StringBuffer representation) throws NonParsableException{
		super(representation);
	}
	
	protected void appendCollection( StringBuffer buf )
	{
		XMLParser.appendStringWithTags( buf, super.getValue().toString(), "selectedEnum" );
		if( hasDefault() )
		{
			XMLParser.appendStringWithTags( buf, parameters.getParameterAt(getDefault()).getValue().toString(), "defaultSelectedEnum" );
		}
		XMLParser.appendStringWithTags( buf, enumInstance.getName(), "enumName" );
	}
	
	protected void extractCollection( StringBuffer buf ) throws NonParsableException
	{
		try {
			enumInstance = (Class<? extends Enum>)Class.forName( XMLParser.extractStringForTag( buf, "enumName" ) );
			enumConstants = enumInstance.getEnumConstants();
			createParameterSet( getKeys( enumInstance ), getKeys( enumInstance ), null );
			if( hasDefault() )
			{
				setDefault( XMLParser.extractStringForTag( buf, "defaultSelectedEnum" ) );
			}
			setValue( XMLParser.extractStringForTag( buf, "selectedEnum" ) );
		} catch ( Exception e ) {
			throw new NonParsableException( e.getMessage() );
		}
	}
	
	public Enum getValue() {
		//TODO return ...valueOf( (String) super.getValue() );
		int idx = 0;
		Object v = super.getValue();
		while( enumConstants[idx].equals( v ) ) {
			idx++;
		}
		return enumConstants[idx];
	}
	
	public void setValue(Object value) throws IllegalValueException {
		if( value != null && value instanceof Enum ) {
			if( !enumInstance.isInstance( value ) ){
				throw new IllegalValueException( "Wrong Enum type." );
			} else {
				value = ((Enum) value).name();
			}
		}
		super.setValue( value );
	}
	
	public void setDefault(Object defaultValue) throws IllegalValueException{
		if( defaultValue != null && defaultValue instanceof Enum ) {
			if( !enumInstance.isInstance( defaultValue ) ){
				throw new IllegalValueException( "Wrong Enum type." );
			} else {
				defaultValue = ((Enum) defaultValue).name();
			}
		}
		super.setValue( defaultValue );
	}	
}
