/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.parameters;

import de.jstacs.DataType;
import de.jstacs.NonParsableException;
import de.jstacs.data.AlphabetContainer;
import de.jstacs.data.AlphabetContainerParameterSet;
import de.jstacs.io.XMLParser;
import de.jstacs.parameters.validation.NumberValidator;

/**
 * Abstract class for a <code>ParameterSet</code> containing all parameters necessary to construct an
 * <code>Object</code> that implements <code>InstantiableFromParameterSet</code>. This parameter set handles the
 * alphabet container and if necessary the length, so it is well suited as parameter set for AbstractModel and
 * AbstractClassifier.
 * 
 * @author Jan Grau, Jens Keilwagen
 * 
 * @see de.jstacs.InstantiableFromParameterSet
 * @see de.jstacs.parameters.ParameterSet
 * 
 * @see de.jstacs.models.AbstractModel
 * @see de.jstacs.classifier.AbstractClassifier
 */
public abstract class InstanceParameterSet extends ParameterSet
{
	/**
	 * This method tries to find the correct name (String) for your choice. This method is useful if you handle
	 * CollectionParameterSets.
	 * 
	 * @param names
	 *            the names
	 * @param values
	 *            the values, that can be set
	 * @param current
	 *            the value to be set
	 * @param hasAlternative
	 *            indicates whether the last entry of names is an alternative parameter
	 * 
	 * @return the String that has to be used
	 * 
	 * @throws IllegalArgumentException
	 *             if no match could be found
	 * 
	 * @see de.jstacs.parameters.CollectionParameter
	 */
	public static int getIndex( String[] names, Object[] values, Comparable current, boolean hasAlternative )
			throws IllegalArgumentException
	{
		int i = 0, l = values.length;
		if( hasAlternative )
		{
			l--;
		}
		while( i < l && !current.equals( values[i] ) )
		{
			i++;
		}
		if( i == values.length )
		{
			throw new IllegalArgumentException( "Could not find a matching constant." );
		}
		else
		{
			return i;
		}
	}

	/**
	 * The alphabet the model works on
	 */
	protected Parameter alphabet;

	/**
	 * The length of sequences the model can work on or <code>0</code> for arbitrary length
	 */
	protected Parameter length;

	/**
	 * <code>true</code> if the model can handle sequences of variable length
	 */
	private boolean variableLength;

	/**
	 * Constructs a <code>InstanceParameterSet</code> having empty parameter values. This constructor
	 * should only be used if the object can handle sequences of fixed length.
	 * 
	 * @param instanceClass
	 *            the class of the instance
	 * @param discrete
	 *            whether the alphabet should be discrete
	 * @param simple
	 *            whether the alphabet should be simple
	 */
	public InstanceParameterSet( Class instanceClass, boolean discrete, boolean simple )
	{
		this( instanceClass, discrete, simple, false );
	}

	/**
	 * Constructs a <code>InstanceParameterSet</code> having empty parameter values. The user can specify a-priorily
	 * if the object can handle sequences of variable lengths. If it can handle sequences of variable length the object is not queried from
	 * the user as it is <code>0</code> anyway.
	 * 
	 * @param instanceClass
	 *            the (sub-)class
	 * @param discrete
	 *            whether the alphabet should be discrete
	 * @param simple
	 *            whether the alphabet should be simple
	 * @param variableLength
	 *            <code>true</code> if the object can handle sequences of arbitrary length
	 */
	public InstanceParameterSet( Class instanceClass, boolean discrete, boolean simple, boolean variableLength )
	{
		super( instanceClass );
		this.variableLength = variableLength;
		try
		{
			this.alphabet = new ParameterSetContainer( "Alphabet", "The alphabet the model works on",
					new AlphabetContainerParameterSet( discrete, simple ) );
			if( variableLength ) {
				this.length = new SimpleParameter( DataType.INT, "Length",
					"The length of sequences the model can work on", true, new NumberValidator<Integer>( 0,0 ), 0 );
			} else {
				this.length = new SimpleParameter( DataType.INT, "Length",
						"The length of sequences the model can work on", true, new NumberValidator<Integer>( 1,	Integer.MAX_VALUE ) );
			}
		}
		catch( ParameterException doesNotHappen )
		{
		}
		try
		{
			loadParameters();
		}
		catch( Exception e )
		{
			UnsupportedOperationException u = new UnsupportedOperationException( e.getMessage() );
			u.setStackTrace( e.getStackTrace() );
			throw u;
		}
	}

	/**
	 * Constructs a <code>InstanceParameterSet</code> from its XML-representation. Automatically calls the current
	 * implementation of <code>fromXML(StringBuffer)</code> of <code>Storable</code>.
	 * 
	 * @param representation
	 *            the XML-representation
	 * @throws NonParsableException
	 *             a <code>NonParsableException</code> is thrown if <code>representation</code> could not be parsed
	 */
	public InstanceParameterSet( StringBuffer representation ) throws NonParsableException
	{
		super( representation );
	}

	/**
	 * Constructs a <code>InstanceParameterSet</code> from the alphabet and the length. This constructor can be used
	 * to implement a <code>InstanceParameterSet</code> that is already instantiated with known parameter values.
	 * 
	 * @param instanceClass
	 *            the class of the instance
	 * @param alphabet
	 *            the alphabet
	 * @param length
	 *            the length
	 * @param variableLength
	 * 			  whether the object can handle sequences of variable length
	 * @throws Exception
	 *             an <code>Exception</code> is thrown if the alphabet or the length are not in the expected range of
	 *             values
	 */
	public InstanceParameterSet( Class instanceClass, AlphabetContainer alphabet, int length, boolean variableLength ) throws Exception
	{
		this( instanceClass, alphabet.isDiscrete(), alphabet.isSimple(), variableLength );
		this.length.setValue( length );
		this.alphabet.setValue( alphabet.getCurrentParameterSet() );
	}

	/**
	 * Constructs a <code>InstanceParameterSet</code> for an object that can handle sequences of variable length and with the alphabet.
	 * This constructor can be used to implement a <code>InstanceParameterSet</code> that is already instantiated with known parameter
	 * values.
	 * 
	 * <br>
	 * 
	 * The length-parameter is set to <code>0</code>.
	 * 
	 * @param instanceClass
	 *            the (sub-)class
	 * @param alphabet
	 *            the alphabet
	 * @throws Exception
	 *             an <code>Exception</code> is thrown if the alphabet or the length are not in the expected range of
	 *             values
	 */
	public InstanceParameterSet( Class instanceClass, AlphabetContainer alphabet ) throws Exception
	{
		this( instanceClass, alphabet, 0, true );
	}

	@Override
	public boolean hasDefaultOrIsSet()
	{
		boolean erg;
		erg = super.hasDefaultOrIsSet() && alphabet.hasDefaultOrIsSet() && length.hasDefaultOrIsSet();

		if( erg )
		{
			// AlphabetContainer abc = getAlphabet();
			int l = 0;
			if( alphabet != null )
			{
				l = ((AlphabetContainerParameterSet) alphabet.getValue()).getPossibleLength();
			}
			erg &= l == 0 || ((Integer) length.getValue()).intValue() == l;
			if( !erg )
			{
				errorMessage = "The length of the alphabet and the length of the model must match!";
			}
		}
		return erg;
	}

	@Override
	public void reset()
	{
		super.reset();
		length.reset();
		alphabet.reset();
	}

	/**
	 * Returns the alphabet
	 * 
	 * @return the alphabet
	 */
	public AlphabetContainer getAlphabet()
	{
		try
		{
			AlphabetContainer cont = new AlphabetContainer( (AlphabetContainerParameterSet) alphabet.getValue() );
			return cont;
		}
		catch( Exception e )
		{
			e.printStackTrace();
			return null;
		}
	}

	/**
	 * Returns the length of sequences the model can work on
	 * 
	 * @return the length
	 * 
	 * @throws IllegalArgumentException if the length is not correct, i.e. the length is not suitable for the chosen requirements
	 */
	public int getLength() throws IllegalArgumentException
	{
		int l = (Integer) length.getValue();
		if( variableLength && l != 0 )
		{
			throw new IllegalArgumentException( "The model can handle sequences of variable length, but length is defined as " + l );
		}
		return l;
	}

	protected void fromXML( StringBuffer representation ) throws NonParsableException
	{
		representation = XMLParser.extractForTag( representation, "modelInstanceParameterSet" );
		super.fromXML( XMLParser.extractForTag( representation, "superParameters" ) );
		String temp = null;
		try
		{
			temp = representation.toString();
			variableLength = XMLParser.extractBooleanForTag( representation, "variableLength" );
			StringBuffer alphStringB = XMLParser.extractForTag( representation, "alphabet" );
			if( alphStringB == null )
			{
				alphabet = null;
			}
			else
			{
				alphabet = new ParameterSetContainer( alphStringB );
			}
			length = new SimpleParameter( XMLParser.extractForTag( representation, "length" ) );

		}
		catch( NonParsableException e )
		{
			e.printStackTrace();
			System.out.println( "++++++++++++++++++++++++++++++++++++++++++" );
			// System.out.println(temp2);
			System.out.println( "++++++++++++++++++++++++++++++++++++++++++" );
			System.out.println( temp );
			System.out.println( "++++++++++++++++++++++++++++++++++++++++++" );
			throw e;
		}
	}

	@Override
	public int getNumberOfParameters()
	{
		if( variableLength )
		{
			// System.out.println("homogeneous:
			// "+(super.getNumberOfParameters() + 1));
			return super.getNumberOfParameters() + 1;
		}
		else
		{
			// System.out.println("inhomogeneous:
			// "+(super.getNumberOfParameters() + 2));
			return super.getNumberOfParameters() + 2;
		}
	}

	@Override
	protected void replaceParametersWithRangedInstance() throws Exception
	{
		if( alphabet instanceof Rangeable && ((Rangeable) alphabet).isRangeable() )
		{
			alphabet = ((Rangeable) alphabet).getRangedInstance();
		}

		if( length instanceof Rangeable && ((Rangeable) length).isRangeable() )
		{
			length = ((Rangeable) length).getRangedInstance();
		}
		super.replaceParametersWithRangedInstance();
	}

	@Override
	public Parameter getParameterAt( int i )
	{
		if( i < super.getNumberOfParameters() )
		{
			return super.getParameterAt( i );
		}
		else if( i == super.getNumberOfParameters() )
		{
			return alphabet;
		}
		else if( !variableLength )
		{
			return length;
		}
		else
		{
			throw new IndexOutOfBoundsException();
		}
	}

	@Override
	public StringBuffer toXML()
	{
		String superPars = super.toXML().toString();
		StringBuffer buf = new StringBuffer();
		XMLParser.appendStringWithTags( buf, superPars, "superParameters" );
		XMLParser.appendBooleanWithTags( buf, variableLength, "variableLength" );
		if( alphabet != null )
		{
			XMLParser.appendStringWithTags( buf, alphabet.toXML().toString(), "alphabet" );
		}
		XMLParser.appendStringWithTags( buf, length.toXML().toString(), "length" );
		XMLParser.addTags( buf, "modelInstanceParameterSet" );

		return buf;
	}

	public boolean equals( Object o )
	{
		if( o instanceof InstanceParameterSet )
		{
			InstanceParameterSet comp = (InstanceParameterSet) o;
			boolean erg = alternativeInstanceClass.equals( comp.alternativeInstanceClass )
					&& (getLength() == comp.getLength()) && (parameters.size() == comp.parameters.size())
					&& getAlphabet().checkConsistency( comp.getAlphabet() );
			int i = 0;
			while( i < parameters.size() && erg )
			{
				erg &= parameters.get( i ).equals( comp.parameters.get( i++ ) );
			}
			return erg;
		}
		else
		{
			return false;
		}
	}

	public InstanceParameterSet clone() throws CloneNotSupportedException
	{
		InstanceParameterSet res = (InstanceParameterSet) super.clone();
		res.alphabet = alphabet.clone();
		res.length = length.clone();
		return res;
	}
}
