/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.scoringFunctions.directedGraphicalModels.structureLearning.measures.pmmMeasures;

import de.jstacs.NonParsableException;
import de.jstacs.algorithms.graphs.DAG;
import de.jstacs.algorithms.graphs.tensor.SymmetricTensor;
import de.jstacs.algorithms.graphs.tensor.Tensor;
import de.jstacs.data.Sample;
import de.jstacs.io.XMLParser;
import de.jstacs.scoringFunctions.directedGraphicalModels.BayesianNetworkScoringFunction;
import de.jstacs.scoringFunctions.directedGraphicalModels.structureLearning.measures.Measure;
import de.jstacs.scoringFunctions.directedGraphicalModels.structureLearning.measures.btMeasures.BTMutualInformation;

/**
 * Class for the network structure of a {@link BayesianNetworkScoringFunction} that is a permuted Markov model based on mutual information.
 * 
 * @author Jan Grau
 *
 */
public class PMMMutualInformation extends Measure {

	private int clazz;
	private byte order;
	private double[] ess;
	
	/**
	 * Compute mutual information only from foreground data
	 */
	public static final int FG=0;
	/**
	 * Compute mutual information only from background data
	 */
	public static final int BG=1;
	/**
	 * Use both data sets to compute the mutual information
	 */
	public static final int BOTH=2;
	
	/**
	 * Creates a new {@link PMMMutualInformation} of order <code>order</code>.
	 * @param order the order, may be <code>1</code> or <code>2</code>.
	 * @param clazz the classes used for computation of mutual information, one of {@link BTMutualInformation#FG}, {@link BTMutualInformation#BG}, {@link BTMutualInformation#BOTH}
	 * @param ess the equivalent sample size for both classes
	 * @throws Exception
	 */
	public PMMMutualInformation(byte order, int clazz, double[] ess) throws Exception {
		if(order < 1 || order > 2){
			throw new Exception("Only order 1 and 2 allowed (yet).");
		}
		this.order = order;
		this.ess = ess;
		if(clazz == FG || clazz == BG || clazz == BOTH){
			this.clazz = clazz;
		}else{
			throw new Exception("Value of clazz not allowed.");
		}
	}
	
	/**
	 * Re-creates a {@link PMMMutualInformation} from its XML-representation as returned by {@link PMMMutualInformation#toXML()}.
	 * @param buf the XML-representation
	 * @throws NonParsableException is thrown if the XML-code could not be parsed
	 */
	public PMMMutualInformation(StringBuffer buf) throws NonParsableException{
		buf = XMLParser.extractForTag(buf, "pmmMutualInformation");
		order = XMLParser.extractByteForTag(buf, "order");
		clazz = XMLParser.extractIntForTag(buf, "clazz");
		ess = XMLParser.extractDoubleArrayForTag(buf, "ess");
	}

	/* (non-Javadoc)
	 * @see de.jstacs.scoringFunctions.directedGraphicalModels.structureLearning.measures.Measure#clone()
	 */
	public PMMMutualInformation clone() throws CloneNotSupportedException{
		PMMMutualInformation clone = (PMMMutualInformation) super.clone();
		clone.ess = ess.clone();
		return clone;
	}
	
	/* (non-Javadoc)
	 * @see de.jstacs.scoringFunctions.directedGraphicalModels.structureLearning.measures.Measure#getInstanceName()
	 */
	@Override
	public String getInstanceName() {
		String str = "Permuted Markov model of order "+order+" with mutual information of";
		if(clazz == FG){
			return str+" foreground";
		}else if(clazz == BG){
			return str+" background";
		}else{
			return str+" foreground and background";
		}
	}

	/* (non-Javadoc)
	 * @see de.jstacs.scoringFunctions.directedGraphicalModels.structureLearning.measures.Measure#getParents(de.jstacs.data.Sample, de.jstacs.data.Sample, double[], double[], int)
	 */
	@Override
	public int[][] getParents(Sample fg, Sample bg, double[] weightsFg, double[] weightsBg, int length) throws Exception {
		Sample data = null;
		double[] weights = null;
		double ess2 = 0;
		if(clazz == FG){
			data = fg;
			weights = weightsFg;
			ess2 = ess[0];
		}else if(clazz == BG){
			data = bg;
			weights = weightsBg;
			ess2 = ess[1];
		}else{
			data = Sample.union(fg,bg);
			weights = union(new double[][]{weightsFg, weightsBg});
			ess2 = ess[0] + ess[1];
		}
		
		Tensor t = new SymmetricTensor(length,order);
		fillTensor(t, getMI(getStatistics(data, weights, length, ess2), sum(weights) + ess2));
		if(order == 2){
			fillTensor(t, getMI(getStatisticsOrderTwo(data, weights, length, ess2), sum(weights) + ess2));
		}
		int[] o = DAG.computeMaximalHP(t);
		
		return toParents(o, order);
	}

	/* (non-Javadoc)
	 * @see de.jstacs.Storable#toXML()
	 */
	public StringBuffer toXML() {
		StringBuffer buf = new StringBuffer();
		XMLParser.appendByteWithTags(buf, order, "order");
		XMLParser.appendIntWithTags(buf, clazz, "clazz");
		XMLParser.appendDoubleArrayWithTags(buf, ess, "ess");
		XMLParser.addTags(buf, "pmmMutualInformation");
		return buf;
	}

}
