/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.utils.random;

import java.util.Arrays;

/**
 * The container for parameter of a Dirichlet random generator.
 *  
 * @author Jens Keilwagen
 * @see DiMRGParams
 * @see FastDirichletMRGParams
 */
public class DirichletMRGParams extends DiMRGParams
{
	private double[] alpha;
	private double sum;
	
	/**
	 * @see FastDirichletMRGParams#FastDirichletMRGParams(double)
	 */
	public DirichletMRGParams( double alpha, int n ) throws IllegalArgumentException
	{
		if( n < 2 )
		{
			throw new IllegalArgumentException( "The parameter n has to be at least 2." );
		}
		if( alpha <= 0d )
		{
			throw new IllegalArgumentException( "The parameter alpha has to be positive." );
		}
		this.alpha = new double[n];
		Arrays.fill( this.alpha, alpha );
		sum = n*alpha;
	}
	
	public DirichletMRGParams( double... alpha ) throws IllegalArgumentException
	{
		this.alpha = new double[alpha.length];
		sum = 0;
		for( int i = 0; i < alpha.length; i++ )
		{
			if( alpha[i] <= 0d )
			{
				throw new IllegalArgumentException( "Each parameter alpha[i] has to be positive." );
			}
			this.alpha[i] = alpha[i];
			sum += alpha[i];
		}
	}
	
	public int getDimension()
	{
		return alpha.length;
	}
	
	public double getHyperparameter( int i )
	{
		return alpha[i];
	}
	
	public double getSumOfHyperparameter()
	{
		return sum;
	}	
}
