/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.utils.random;

import java.io.IOException;
import java.util.Random;

/**
 * This class is a random generator for pseudo-random numbers.
 * 
 * @author Jens Keilwagen
 */
public class UnivariateRandomGenerator
{
	/**
	 * @return an equal distributed double in [0,1] (p(x) = 1)
	 */
	public static double equalDistributed()
	{
		return equalDistributed( new Random() );
	}
	
	private static double equalDistributed( Random r )
	{
		return r.nextDouble();
	} 
	
	/**
	 * @return an exponential distributed double [0,\infty] (p(x) = exp(-x))
	 */
	public static double exponentialDistributed()
	{
		return exponentialDistributed( new Random() );
	}
	
	private static double exponentialDistributed( Random r )
	{
		return - Math.log(1-equalDistributed( r ));
	}
	
	/**
	 * gamma distributed for alpha \in N.
	 * 
	 * @param alpha hyperparameter alpha
	 * @return a gamma distributed double [0,\infty]
	 */
	public static double erlangDistributed( int alpha ) throws IOException
	{
		if( alpha < 1 )
		{
			throw new IOException( "Only possible for alpha > 0" );
		}
		return erlangDistributed( alpha, new Random() );
	}
	
	public static double erlangDistributed( int alpha, Random r )
	{
		double erg = exponentialDistributed( r );
		for( int i = 1; i < alpha; i++ )
		{
			erg += exponentialDistributed( r );
		}
		return erg;
	}
	
}
