/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.classifier;

import java.util.Arrays;
import java.util.HashSet;

import de.jstacs.NonParsableException;
import de.jstacs.NotTrainedException;
import de.jstacs.data.Sample;
import de.jstacs.data.Sequence;
import de.jstacs.io.XMLParser;
import de.jstacs.results.CategoricalResult;
import de.jstacs.results.NumericalResultSet;
import de.jstacs.utils.IntList;
import de.jstacs.utils.Normalisation;

/**
 * This class allows the user to train the classifier on a given number of
 * classes and to evaluate the classifier on a smaller number of classes by
 * mapping classes together. For instance the user has a classifier for 3
 * classes, but likes to evaluate whether the classifier is able to discriminate
 * between class 1 and class 2 and 3. This is a good example where to use this
 * class. The user has to create its 3-class-classifier, create a instance of
 * this class using its classifier, map the test samples together ({@link MappingClassifier#mapSample(Sample[])})
 * and invoke
 * {@link AbstractClassifier#evaluate(MeasureParameters, boolean, Sample[])}
 * with these mapped sample.
 * 
 * @author Jens Keilwagen
 */
public class MappingClassifier extends AbstractScoreBasedClassifier {

	private AbstractScoreBasedClassifier classifier;

	private int[][] classMapping;

	private static int getNum( int[] mapping ) {
		HashSet<Integer> hash = new HashSet<Integer>();
		for( int i = 0; i < mapping.length; i++ ) {
			if( !hash.contains( mapping[i] ) ) {
				hash.add( mapping[i] );
			}
		}
		return hash.size();
	}

	/**
	 * The main constructor.
	 * 
	 * @param classifier
	 *            the internal used classifier
	 * @param mapping
	 *            the mapping from the classes of the internal classifier to the
	 *            classes of this classifier
	 * 
	 * @throws CloneNotSupportedException
	 */
	public MappingClassifier( AbstractScoreBasedClassifier classifier, int[] mapping ) throws CloneNotSupportedException {
		super( classifier.getAlphabetContainer(), classifier.getLength(), getNum( mapping ) );

		if( mapping.length != classifier.getNumberOfClasses() ) {
			throw new IllegalArgumentException( "The length of the mapping is not correct." );
		}
		IntList[] il = new IntList[getNumberOfClasses()];
		for( int i = 0; i < il.length; i++ ) {
			il[i] = new IntList();
		}
		for( int i = 0; i < mapping.length; i++ ) {
			il[mapping[i]].add( i );
		}
		classMapping = new int[il.length][];
		for( int i = 0; i < il.length; i++ ) {
			if( il[i].length() == 0 ) {
				throw new IllegalArgumentException( "Mapping to class " + i + " is empty" );
			} else {
				classMapping[i] = il[i].toArray();
			}
		}
		this.classifier = classifier.clone();
	}

	/**
	 * This is the constructor for {@link de.jstacs.Storable}.
	 * 
	 * @param representation the xml representation
	 * 
	 * @throws NonParsableException if the representation could not be parsed.
	 */
	public MappingClassifier( StringBuffer representation ) throws NonParsableException {
		super( representation );
	}

	protected void extractFurtherClassifierInfosFromXML( StringBuffer xml ) throws NonParsableException {
		super.extractFurtherClassifierInfosFromXML( xml );
		classifier = (AbstractScoreBasedClassifier)XMLParser.extractStorableForTag( xml, "classifier" );
		classMapping = XMLParser.extractInt2ArrayForTag( xml, "mapping" );
	}
	
	protected StringBuffer getFurtherClassifierInfos()
	{
		StringBuffer xml = super.getFurtherClassifierInfos();
		XMLParser.appendStorableWithTags( xml, classifier, "classifier" );
		XMLParser.appendInt2ArrayWithTags( xml, classMapping, "mapping" );
		return xml;
	}

	protected double getScore( Sequence seq, int i, boolean check ) throws IllegalArgumentException, NotTrainedException, Exception {
		double res = classifier.getScore( seq, classMapping[i][0], true );
		for( int idx = 1; idx < classMapping[i].length; idx++ ) {
			res = Normalisation.getLogSum( res, classifier.getScore( seq, classMapping[i][1], false ) );
		}
		return res;
	}

	public CategoricalResult[] getClassifierAnnotation() {
		return classifier.getClassifierAnnotation();
	}

	public String getInstanceName() {
		return "MappingClassifier of " + classifier.getInstanceName();
	}

	public NumericalResultSet getNumericalCharacteristics() throws Exception {
		return classifier.getNumericalCharacteristics();
	}

	protected String getXMLTag() {
		return getClass().getSimpleName();
	}

	public boolean isTrained() {
		return classifier.isTrained();
	}

	public void train( Sample[] s, double[][] weights ) throws Exception {
		classifier.train( s, weights );
	}

	/**
	 * This method maps the Samples to the internal classes.
	 * 
	 * @param s the array of samples corresponding to the classes of the internal classifier
	 * 
	 * @return the array of samples corresponding to the classes
	 */
	public Sample[] mapSample( Sample[] s ) {
		boolean[] in = new boolean[classifier.getNumberOfClasses()];
		Sample[] mapped = new Sample[classMapping.length];
		try {
			for( int j, i = 0; i < mapped.length; i++ ) {
				Arrays.fill( in, false );
				for( j = 0; j < classMapping[i].length; j++ ) {
					in[classMapping[i][j]] = true;
				}
				mapped[i] = Sample.union( s, in );
			}
		} catch (Exception e) {
			// does not happen
			throw new RuntimeException();
		}
		return mapped;
	}
}
