/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.classifier.assessment;

import java.util.ArrayList;
import java.util.Collection;

import de.jstacs.DataType;
import de.jstacs.NonParsableException;
import de.jstacs.parameters.Parameter;
import de.jstacs.parameters.ParameterSet;
import de.jstacs.parameters.SimpleParameter;
import de.jstacs.parameters.SimpleParameter.IllegalValueException;
import de.jstacs.results.Result;

/**
 * This class is the super-class used by all <code>ClassifierAssessmentAssessParameterSet</code>s.
 * It is a container of user-specified parameters necessary to define a run of an
 * <code>ClassifierAssessment</code>. In contrast to <code>AssessmentAssessParameterSet</code>s
 * that are necessary to start an <code>ClassifierAssessment</code>, 
 * <code>ClassifierAssessmentParameterSet</code>s are necessary to instantiate a
 * <code>ClassifierAssessment</code>-object.
 * 
 * @author andr|e gohr (a0coder (nospam:@) gmail (nospam:.) com)
 *
 */
public class ClassifierAssessmentAssessParameterSet extends ParameterSet {

//	**********************
//	class variables
//	**********************

//	**********************
//	class methods
//	**********************
	
	
	/**
	 * @return all different yet implemented <code>ClassifierAssessmentAssessParameterSets</code>.
	 * This method is used by <code>ClassifierAssessmentTaskParameterSet</code> to allow a user to chose
	 * a certain assessment to assess user specified classifiers.
	 */
	final static public ParameterSet[] getAllClassifierAssessmentAssessParameterSets
	(){
		return new ParameterSet[] {
				  new RepeatedHoldOutAssessParameterSet(),
				  new RepeatedSubSamplingAssessParameterSet(),
				  new KFoldCVAssessParameterSet()
				  };
	}

//	**********************
//	member variables
//	**********************

//	**********************
//	constructors
//	**********************

	/**
	 * inherited from <code>ParameterSet</code>
	 */
	protected ClassifierAssessmentAssessParameterSet
	(Class alternativeInstanceClass) {
	super(alternativeInstanceClass);
	}

	/**
	 * inherited from <code>ParameterSet</code>
	 */
	public ClassifierAssessmentAssessParameterSet
	() throws UnsupportedOperationException {
	super();
	}

	/**
	 * inherited from <code>ParameterSet</code>
	 */
	public ClassifierAssessmentAssessParameterSet
	(StringBuffer representation)
	throws NonParsableException {
	super(representation);
	}

	/**
	 * @param elementLength 	defines the length of elements (sequences) the classifiers
	 * 							to be assessed are able to classify
	 * @param exceptionIfMPNotComputable a <code>ClassifierAssessmentAssessParameterSet</code> is used
	 * 							in combination with an <code>MeasureParameters</code>-object
	 * 							to call <code>assess</code>-methods of <code>CLassifierAssessment</code>s.
	 * 						    If <code>exceptionIfMPNotComputable=true</code> then an expection is thrown
	 * 							in case of a selected measure-parameters that could not be computed.
	 * @throws IllegalValueException is thrown in case of out-of-range or invalid given parameters 
	 */
	public ClassifierAssessmentAssessParameterSet
	(int elementLength, boolean exceptionIfMPNotComputable)
	throws IllegalValueException{
		
		try{
		
			this.loadParameters();
		
		}catch(Exception neverHappens){};
		
		(this.parameters.get(0)).setValue(new Integer(elementLength));
		(this.parameters.get(1)).setValue(new Boolean(exceptionIfMPNotComputable));
	}
	
	
//	**********************
//	member methods
//	**********************

	/**
	 * Initializes the list of {@link Parameter}s in this {@link ParameterSet}.
	 */
	protected void initializeMyParametersArrayList
	(){
		initParameterList(2);
	}
	
	@Override
	protected void loadParameters
	() throws Exception {
		
		this.initializeMyParametersArrayList();
		
		//0-subSequenceLength
		this.parameters.add( new SimpleParameter(DataType.INT,
												 "elementLength", 
												 "Defines the lengths of overlapping windows " +
												 "of data, that should be used to train and test " +
												 "classifiers/models.",
												 true)
							);
		
		//1-excpetionIfMPNotComputable
		this.parameters.add( new SimpleParameter(DataType.BOOLEAN,
												 "exceptionIfMeasureParamaterNotComputable",
												 "True causes ClassiefierAssessment to throw " +
												 "an error if measure-parameters should be computed " +
												 "that are not computable for the given classifiers.",
												 true)
						    );
		

	}

	@Override
	/**
	 * @return 	the class-name of that <code>ClassifierAssessment</code> 
	 * 			which method <code>assess()</code>
	 * 			uses this <code>ClassifierAssessmentAssessParameterSet</code>.
	 */
	public String getInstanceName() {
	return "ClassifierAssessment";
	}

	@Override
	public String getInstanceComment() {
	return "This AssessParameterSet may be used to pass parameters into " +
			"ClassifierAssessment's method assess().";
	}
	
	/**
	 * @return the elementLength defined by this <code>ClassifierAssessmentAssessParameterSet<code>
	 */
	public int getElementLength
	(){
	return ((Integer)(this.getParameterAt(0).getValue())).intValue();
	}
	
	/**
	 * @return 	the flag defined by this <code>ClassifierAssessmentAssessParameterSet<code>
	 * 			(true: an exception is thrown if user selected measure-parameters that could not
	 * 			be computed. false: not)
	 */
	public boolean getExceptionIfMPNotComputable
	(){
	return ((Boolean)(getParameterAt(1).getValue())).booleanValue();
	}

	/**
	 * 
	 * @return collection of parameters containing informations about
	 * this <code>ClassifierAssessmentAssessParameterSet</code>
	 */
	public Collection<Result> 
	getAnnotation(){
	return new ArrayList<Result>(0);
	}
}
