/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.classifier.assessment;

import java.util.ArrayList;

import de.jstacs.DataType;
import de.jstacs.NonParsableException;
import de.jstacs.data.Sample.PartitionMethod;
import de.jstacs.parameters.EnumParameter;
import de.jstacs.parameters.SimpleParameter;
import de.jstacs.parameters.SimpleParameter.IllegalValueException;
import de.jstacs.parameters.validation.NumberValidator;
import de.jstacs.results.CategoricalResult;
import de.jstacs.results.NumericalResult;
import de.jstacs.results.Result;

/**
 * This class implements a <code>ClassifierAssessmentAssessParameterSet</code> that must be used
 * to call method <code>assess()</code> of a <code>KFoldCrossValidation</code>.<br> 
 * It contains user-specific parameters necessary for a run of a <code>KFoldCrossValidation</code>.
 * 
 * @author andr|e gohr (a0coder (nospam:@) gmail (nospam:.) com)
 *
 */
public class KFoldCVAssessParameterSet extends ClassifierAssessmentAssessParameterSet {

//	**********************
//	class variables
//	**********************

//	**********************
//	class methods
//	**********************

//	**********************
//	member variables
//	**********************

//	**********************
//	constructors
//	**********************
	
	/**
	 * inherited from <code>ClassifierAssessmentAssessParameterSet</code>
	 */
	protected KFoldCVAssessParameterSet
	(Class alternativeInstanceClass) {
	super(alternativeInstanceClass);
	}

	/**
	 * inherited from <code>ClassifierAssessmentAssessParameterSet</code>
	 */
	public KFoldCVAssessParameterSet() 
	throws UnsupportedOperationException {
	super();
	}

	/**
	 * inherited from <code>ClassifierAssessmentAssessParameterSet</code>
	 */
	public KFoldCVAssessParameterSet
	(StringBuffer representation)
	throws NonParsableException {
	super(representation);
	}

	/**
	 * @param dataSplitMethod	defines the method used to split user-supplied data into
	 * 							<code>k</code> mutually exclusive, random-splits
	 * 							(available options are: {@link PartitionMethod#PARTITION_BY_NUMBER_OF_ELEMENTS} and {@link PartitionMethod#PARTITION_BY_NUMBER_OF_SYMBOLS}) See docu of <code>Sample</code>
	 * 							for further details.
	 * 
  	 * @param elementLength 	defines the length of elements (sequences) the classifiers
	 * 							to be assessed are able to classify
	 * 
	 * @param exceptionIfMPNotComputable a <code>KFoldCVAssessParameterSet</code> is used
	 * 							in combination with an <code>MeasureParameters</code>-object
	 * 							to call <code>assess</code>-methods of <code>KFoldCrossValidation</code>s.
	 * 						    If <code>exceptionIfMPNotComputable=true</code> then an expection is thrown
	 * 							in case of a selected measure-parameters that could not be computed.
	 * 
	 * @param k					defines the number of mutually exclusive, random-splits of
	 * 							user supplied data. Each part is used once as a test-data-set and 
	 * 							the union of the remaining k-1 patrs is once used as train-data-set.
	 * 							Thus <code>k</code> also defines how many (<code>k</code>) repeated
	 * 							classifier-trainings and classifier-evaluations (test) are performend. 
	 * 
	 * @throws IllegalValueException is thrown in case of out-of-range or invalid given parameters
	 */
	public KFoldCVAssessParameterSet
	(PartitionMethod dataSplitMethod, int elementLength, 
	boolean exceptionIfMPNotComputable, int k) 
	throws IllegalValueException {
	super(elementLength, exceptionIfMPNotComputable);
	
		(this.parameters.get(2)).setValue(new Integer(k));
		
		(this.parameters.get(3)).setValue(dataSplitMethod);
	}
	
	
//	**********************
//	member methods
//	**********************

	protected void initializeMyParametersArrayList
	(){
		initParameterList(4);
	}
	
	
	@Override
	protected void loadParameters
	() throws Exception {
	super.loadParameters();	
				
		//2-k
		this.parameters.add( new SimpleParameter(DataType.INT,
												 "k",
												 "Defines the folds of a KFoldCrossValidation.",
												 true,
												 new NumberValidator<Integer>(2,Integer.MAX_VALUE))
						    );
		

		//3-dataSplitMethod
		this.parameters.add( new EnumParameter(PartitionMethod.class,"The method used to compute the percentages of the partitions",true ) 
							);
	}

	@Override
	public String getInstanceName() {
	return "KFoldCrossValidation";
	}

	@Override
	public String getInstanceComment() {
	return "This KFoldCVAssessParameterSet may be used to pass parameters into " +
			"KFoldCrossValidation's method assess().";
	}

	/**
	 * @return 	the number of mutually exclusive, random-splits of user-supplied data
	 * 			defined by this <code>KFoldCVAssessParameterSet</code>	
	 */
	public int getK
	(){
	return ((Integer)(this.getParameterAt(2).getValue())).intValue();
	}
	
	/**
	 * @return 	The {@link de.jstacs.data.Sample.PartitionMethod} defining how the mutually exclusive, random-splits
	 * 			of user-supplied data are generated. See class <code>Sample</code> for further details.
	 */
	public PartitionMethod getDataSplitMethod
	(){
	return (PartitionMethod) ((EnumParameter) getParameterAt(3)).getValue();
	}

	@Override
	public ArrayList<Result> getAnnotation()
	{
		ArrayList<Result> l = new ArrayList<Result>(2);
		l.add( new NumericalResult( "k", "The folds of a KFoldCrossValidation.", getK() ) );
		l.add( new CategoricalResult( "dataSplitMethod", "Describes how data should be splitted in ClassifierAssessment.evaluateClassifier())",
				getDataSplitMethod().name() ) );
		return l;
	}
	
}
