/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.data;

import java.util.Iterator;
import java.util.LinkedList;

import de.jstacs.DataType;
import de.jstacs.NonParsableException;
import de.jstacs.data.alphabets.ContinuousAlphabet;
import de.jstacs.data.alphabets.DiscreteAlphabet;
import de.jstacs.data.alphabets.ContinuousAlphabet.ContinuousAlphabetParameterSet;
import de.jstacs.data.alphabets.DNAAlphabet.DNAAlphabetParameterSet;
import de.jstacs.data.alphabets.DiscreteAlphabet.DiscreteAlphabetParameterSet;
import de.jstacs.io.XMLParser;
import de.jstacs.parameters.ArrayParameterSet;
import de.jstacs.parameters.CollectionParameter;
import de.jstacs.parameters.ExpandableParameterSet;
import de.jstacs.parameters.Parameter;
import de.jstacs.parameters.ParameterSet;
import de.jstacs.parameters.ParameterSetContainer;
import de.jstacs.parameters.SimpleParameter;
import de.jstacs.parameters.SimpleParameterSet;

/**
 * Class for the <code>ParameterSet</code> of an <code>AlphabetContainer</code>.
 * 
 * @author Jan Grau
 * 
 */
public class AlphabetContainerParameterSet extends ParameterSet
{
	private boolean discrete, simple;

	/**
	 * Creates a new <code>AlphabetContainerParameterSet</code>. If <code>discrete</code> is <code>true</code>,
	 * only the parameters for discrete <code>Alphabet</code>s are defined. Is <code>simple</code> is
	 * <code>true</code>, only a single alphabet is expected.
	 * 
	 * @param discrete
	 *            if the alphabet(s) shall be discrete
	 * @param simple
	 *            if there shall be only a single <code>Alphabet</code>
	 */
	public AlphabetContainerParameterSet( boolean discrete, boolean simple )
	{
		super( AlphabetContainer.class );
		this.discrete = discrete;
		this.simple = simple;
	}

	/**
	 * Creates a new <code>AlphabetContainerParameterSet</code> from its XML-representation.
	 * 
	 * @param representation
	 *            the XML-representation
	 * @throws NonParsableException
	 *             is thrown if <code>representation</code> could not be parsed
	 */
	public AlphabetContainerParameterSet( StringBuffer representation ) throws NonParsableException
	{
		super( representation );
	}

	/**
	 * Creates a new <code>AlphabetContainerParameterSet</code> from a single <code>Alphabet</code>.
	 * 
	 * @param alph
	 *            the <code>Alphabet</code>
	 * @throws Exception
	 *             is thrown if an error occurred during the creation of the appropriate <code>Parameter</code>s
	 */
	public AlphabetContainerParameterSet( Alphabet alph ) throws Exception
	{
		super( AlphabetContainer.class );
		this.discrete = alph instanceof DiscreteAlphabet;
		this.simple = true;
		loadParameters();
		if( discrete )
		{
			this.parameters.get( 0 ).setValue( alph.getCurrentParameterSet() );
		}
		else
		{
			ParameterSet ap = alph.getCurrentParameterSet();
			this.parameters.get( 0 ).setValue( ap.getInstanceName() );
			((CollectionParameter) this.parameters.get( 0 )).getParametersInCollection().getParameterAt(
					((CollectionParameter) this.parameters.get( 0 )).getSelected() ).setValue( ap );
		}
	}

	/**
	 * Creates a new <code>AlphabetContainerParameterSet</code> from an array of <code>Alphabet</code>s.
	 * 
	 * @param alphabets
	 *            the <code>Alphabet</code>s
	 * @throws Exception
	 *             is thrown if an error occurred during the creation of the appropriate <code>Parameter</code>s
	 */
	public AlphabetContainerParameterSet( Alphabet[] alphabets ) throws Exception
	{
		super( AlphabetContainer.class );
		this.simple = false;
		this.discrete = true;
		for( int i = 0; i < alphabets.length; i++ )
		{
			discrete &= (alphabets[i] instanceof DiscreteAlphabet);
		}
		loadParameters();
		AlphabetArrayParameterSet pars = new AlphabetArrayParameterSet( alphabets, discrete );
		parameters.get( 0 ).setValue( pars.getInstanceName() );
		((CollectionParameter) parameters.get( 0 )).getParametersInCollection().getParameterAt(
				((CollectionParameter) parameters.get( 0 )).getSelected() ).setValue( pars );
	}

	/**
	 * Creates a new <code>AlphabetContainerParameterSet</code> from an array of <code>Alphabet</code>s and an
	 * array of <code>int</code>s defining the number of the alphabet in <code>alphabets</code> that is used for
	 * that position in <code>indexes</code>.
	 * 
	 * @param alphabets
	 *            the <code>Alphabet</code>s
	 * @param indexes
	 *            the indexes
	 * @throws Exception
	 *             is thrown if an error occurred during the creation of the appropriate <code>Parameter</code>s
	 */
	public AlphabetContainerParameterSet( Alphabet[] alphabets, int[] indexes ) throws Exception
	{
		super( AlphabetContainer.class );
		this.simple = false;
		this.discrete = true;
		for( int i = 0; i < alphabets.length; i++ )
		{
			discrete &= (alphabets[i] instanceof DiscreteAlphabet);
		}
		loadParameters();
		SectionDefinedAlphabetParameterSet pars = new SectionDefinedAlphabetParameterSet( alphabets, indexes, discrete );
		parameters.get( 0 ).setValue( pars.getInstanceName() );
		((CollectionParameter) parameters.get( 0 )).getParametersInCollection().getParameterAt(
				((CollectionParameter) parameters.get( 0 )).getSelected() ).setValue( pars );
	}

	public AlphabetContainerParameterSet clone() throws CloneNotSupportedException
	{
		AlphabetContainerParameterSet clone = (AlphabetContainerParameterSet) super.clone();

		return clone;
	}

	/**
	 * If this method returns <code>true</code> all postions use <code>DiscreteAlphabetParameterSet</code>s.
	 * 
	 * @return <code>true</code> if all postions are discrete
	 */
	public boolean isDiscrete()
	{
		return discrete;
	}

	/**
	 * If this method returns <code>true</code> all postions use the same alphabet.
	 * 
	 * @return <code>true</code> if all postions use the same alphabet
	 */
	public boolean isSimple()
	{
		return simple;
	}

	public StringBuffer toXML()
	{
		StringBuffer buf = super.toXML();
		XMLParser.addTags( buf, "superParameters" );
		XMLParser.appendBooleanWithTags( buf, discrete, "discrete" );
		XMLParser.appendBooleanWithTags( buf, simple, "simple" );
		XMLParser.addTags( buf, "alphabetContainerParameterSet" );
		return buf;
	}

	protected void fromXML( StringBuffer representation ) throws NonParsableException
	{
		representation = XMLParser.extractForTag( representation, "alphabetContainerParameterSet" );
		super.fromXML( XMLParser.extractForTag( representation, "superParameters" ) );
		discrete = XMLParser.extractBooleanForTag( representation, "discrete" );
		simple = XMLParser.extractBooleanForTag( representation, "simple" );
	}

	/**
	 * Returns the length of the alphabet that can be instantiated using this set.
	 * 
	 * @return the length
	 */
	public int getPossibleLength()
	{
		Object o = parameters.get( 0 ).getValue();
		if( o instanceof DNAAlphabetParameterSet
				|| o instanceof DiscreteAlphabetParameterSet
				|| o instanceof ContinuousAlphabetParameterSet )
		{
			return 0;
		}
		else
		{
			ParameterSet set = (ParameterSet) o;
			return (Integer) set.getParameterAt( 0 ).getValue();
		}
	}

	@Override
	protected void loadParameters() throws Exception
	{
		//TODO JAN: ParameterSet simpleDNA = new DNAAlphabet.DNAAlphabetParameterSet(); ...
		ParameterSet simpleDisc = new DiscreteAlphabet.DiscreteAlphabetParameterSet();
		ParameterSet simpleCont = new ContinuousAlphabet.ContinuousAlphabetParameterSet();

		Object[] vals = null;
		String[] names = null;
		String[] comments = null;

		if( simple && discrete )
		{
			initParameterList();
			parameters.add( new ParameterSetContainer( "Alphabet", "Set a discrete alphabet.", simpleDisc ) );
		}
		else
		{
			if( !discrete && simple )
			{

				vals = new Object[]{ simpleDisc, simpleCont };
				names = new String[]{ simpleDisc.getInstanceName(), simpleCont.getInstanceName() };
				comments = new String[]{ simpleDisc.getInstanceComment(), simpleCont.getInstanceComment() };

			}
			else
			{

				ParameterSet arrayParameters = new AlphabetArrayParameterSet( discrete );

				ParameterSet sectionParameters = new SectionDefinedAlphabetParameterSet( discrete );

				if( !discrete )
				{
					vals = new Object[]{ simpleDisc, simpleCont, arrayParameters, sectionParameters };
					names = new String[]{ simpleDisc.getInstanceName(), simpleCont.getInstanceName(),
							arrayParameters.getInstanceName(), sectionParameters.getInstanceName() };
					comments = new String[]{ simpleDisc.getInstanceComment(), simpleCont.getInstanceComment(),
							arrayParameters.getInstanceComment(), sectionParameters.getInstanceComment() };
				}
				else
				{
					vals = new Object[]{ simpleDisc, arrayParameters, sectionParameters };
					names = new String[]{ simpleDisc.getInstanceName(), arrayParameters.getInstanceName(),
							sectionParameters.getInstanceName() };
					comments = new String[]{ simpleDisc.getInstanceComment(), arrayParameters.getInstanceComment(),
							sectionParameters.getInstanceComment() };
				}

			}

			initParameterList();
			parameters.add( new CollectionParameter( DataType.PARAMETERSET, names, vals, comments, "Type",
					"Select the type of the alphabet", true ) );
		}
	}

	@Override
	public String getInstanceName()
	{
		return "Alphabet";
	}

	@Override
	public String getInstanceComment()
	{
		return "Set an alphabet for your model or data.";
	}

	/**
	 * Class for the parameters of an array of <code>Alphabet</code>s where each alphabet may be used for one or more
	 * sections of positions.
	 * 
	 * @author Jan Grau
	 * 
	 */
	public static class SectionDefinedAlphabetParameterSet extends ExpandableParameterSet
	{

		private boolean discrete;

		/**
		 * Creates a new <code>SectionDefinedAlphabetParameterSet</code> for a set of discrete or continuous
		 * alphabets.
		 * 
		 * @param discrete
		 *            <code>true</code> if the alphabets may be discrete only
		 * @throws Exception
		 *             is thrown if the <code>Parameter</code>s could not be created
		 */
		public SectionDefinedAlphabetParameterSet( boolean discrete ) throws Exception
		{
			super(
					null,
					new SimpleParameterSet(
							new Parameter[]{
									(discrete ? new ParameterSetContainer( "Alphabet",
											"Set the parameters of the alphabet.",
											new DiscreteAlphabet.DiscreteAlphabetParameterSet() )
											: new CollectionParameter( new ParameterSet[]{
													new DiscreteAlphabet.DiscreteAlphabetParameterSet(),
													new ContinuousAlphabet.ContinuousAlphabetParameterSet() },
													"Type of alphabet", "Select the type of the alphabet", true )),
									new SimpleParameter(
											DataType.STRING,
											"Section",
											"The section for that this alphabet is applied. Use &quot;,&quot; to separate different positions and &quot;-&quot; to indicate ranges, e.g. 1-3,5.",
											true ) } ), "Alphabet", "Set the alphabet");
			this.template.getParameterAt( 1 ).setNeededReference( this );
			this.discrete = discrete;
		}

		/**
		 * Creates a new <code>SectionDefinedAlphabetParameterSet</code> from an array of <code>Alphabet</code>s,
		 * an array of indexes that define the index of the <code>Alphabet<>/code> in <code>alphabets/code>
		 * belonging to that position in <code>indexes</code> and a parameter if the alphabets may be discrete only.
		 * @param alphabets the alphabets
		 * @param indexes the indexes
		 * @param discrete <code>true</code> if the alphabets are discrete
		 * @throws Exception is thrown if the <code>Parameter</code>s could not be created
		 */
		public SectionDefinedAlphabetParameterSet( Alphabet[] alphabets, int[] indexes, boolean discrete )
				throws Exception
		{
			this( discrete );
			loadParameters();
			parameters.get( 0 ).setValue( indexes.length );
			for( int i = 0; i < alphabets.length; i++ )
			{
				addParameterToSet();
			}

			for( int i = 0; i < alphabets.length; i++ )
			{

				ParameterSet temp = (ParameterSet) parameters.get( i + 1 ).getValue();
				if( discrete )
				{
					temp.getParameterAt( 0 ).setValue( alphabets[i].getCurrentParameterSet() );
				}
				else
				{
					ParameterSet ap = alphabets[i].getCurrentParameterSet();
					temp.getParameterAt( 0 ).setValue( ap.getInstanceName() );
					((CollectionParameter) temp.getParameterAt( 0 )).getParametersInCollection().getParameterAt(
							((CollectionParameter) temp.getParameterAt( 0 )).getSelected() ).setValue( ap );
				}
				StringBuffer sections = new StringBuffer();
				for( int j = 0; j < indexes.length; j++ )
				{
					if( indexes[j] == i )
					{
						sections.append( (j + 1) + ", " );
					}
				}
				sections.delete( sections.length() - 2, sections.length() );
				temp.getParameterAt( 1 ).setValue( sections.toString() );
			}
		}

		/**
		 * Creates a new <code>SectionDefinedAlphabetParameterSet</code> from its XML-representation.
		 * 
		 * @param representation
		 *            the XML-representation
		 * @throws NonParsableException
		 *             is thrown if <code>representation</code> could not be parsed
		 */
		public SectionDefinedAlphabetParameterSet( StringBuffer representation ) throws NonParsableException
		{
			super( representation );
		}

		protected void loadParameters() throws Exception
		{
			initParameterList();
			SimpleParameter length = new SimpleParameter( DataType.INT, "Length", "The length of the array.", true);
			length.setRangeable( false );
			length.setNeededReference( this );
			this.parameters.add( length );
			super.loadParameters();
		}

		public StringBuffer toXML()
		{
			StringBuffer buf = super.toXML();
			XMLParser.addTags( buf, "superParameters" );
			XMLParser.appendBooleanWithTags( buf, discrete, "discrete" );
			XMLParser.addTags( buf, "sectionDefinedAlphabetParameterSet" );
			return buf;
		}

		public void fromXML( StringBuffer representation ) throws NonParsableException
		{
			representation = XMLParser.extractForTag( representation, "sectionDefinedAlphabetParameterSet" );
			super.fromXML( XMLParser.extractForTag( representation, "superParameters" ) );
			if(this.parameters != null){
				this.parameters.get( 0 ).setNeededReference( this );
				for(int i=1;i<this.parameters.size();i++){
					((ParameterSetContainer)this.parameters.get( i )).getValue().getParameterAt( 1 ).setNeededReference( this );
				}
			}
			discrete = XMLParser.extractBooleanForTag( representation, "discrete" );
		}

		@Override
		public boolean hasDefaultOrIsSet()
		{
			if( !super.hasDefaultOrIsSet() )
			{
				return false;
			}
			else
			{
				Iterator<Parameter> it = parameters.iterator();
				boolean[] set = new boolean[(Integer) it.next().getValue()];
				LinkedList<Integer> tempList;
				int no = 0;
				while( it.hasNext() )
				{
					no++;
					ParameterSet temp = (ParameterSet) it.next().getValue();
					String section = (String) temp.getParameterAt( 1 ).getValue();
					try
					{
						tempList = parseSections( section );
						Iterator<Integer> posIt = tempList.iterator();
						while( posIt.hasNext() )
						{
							int curr = posIt.next();
							if( curr >= set.length )
							{
								errorMessage = "Position " + (curr + 1) + " out of range defined by length.";
								return false;
							}
							else if( set[curr] )
							{
								errorMessage = "Alphabet for position " + (curr + 1) + " defined at least twice.";
								return false;
							}
							else
							{
								set[curr] = true;
							}

						}

					}
					catch( Exception e )
					{
						errorMessage = "Malformed section definition no. " + no;
						e.printStackTrace();
						return false;
					}
				}
				for( int i = 0; i < set.length; i++ )
				{
					if( !set[i] )
					{
						errorMessage = "No alphabet defined for position " + (i+1) + ".";
						return false;
					}
				}
				errorMessage = null;
				return true;
			}
		}

		/**
		 * Parsed the sections as defined in <code>sections</code> to a list of <code>Integer</code>s. Sections may
		 * be defined as ranges (e.g. &quot;3-6&quot;), as lists (e.g. &quot;3,4,5,6&quot;), or as combinations of both
		 * (e.g. (&quot;3-5,6&quot;)
		 * 
		 * @param sections
		 *            the sections
		 * @return the list of positions
		 * @throws Exception
		 *             is thrown if <code>sections</code> could not be parsed, contains a position more than once or
		 *             contains positions that are out of range
		 */
		public static LinkedList<Integer> parseSections( String sections ) throws Exception
		{
			String[] secs = sections.split( "(\\s*,\\s*)" );
			LinkedList<Integer> list = new LinkedList<Integer>();
			String[] temp;
			for( int i = 0; i < secs.length; i++ )
			{
				// if(secs[i].indexOf('-') > -1){
				temp = secs[i].split( "(\\s*-\\s*)" );
				// }else{
				// temp = new String[]{secs[i]};
				// }
				if( temp.length == 2 )
				{
					int start = Integer.parseInt( temp[0] ) - 1;
					int end = Integer.parseInt( temp[1] ) - 1;
					start = start <= end ? start : end;
					end = end >= start ? end : start;
					if( start < 0 )
					{
						throw new Exception( "Malformed sections, no negative values allowed." );
					}
					for( int j = start; j <= end; j++ )
					{
						list.add( j );
					}
				}
				else if( temp.length == 1 )
				{
					int pos = Integer.parseInt( temp[0] ) - 1;
					if( pos < 0 )
					{
						throw new Exception( "Malformed sections, no negative values allowed." );
					}
					list.add( pos );
				}
				else
				{
					throw new Exception( "Malformed sections." );
				}
			}
			return list;
		}

		@Override
		public SectionDefinedAlphabetParameterSet clone() throws CloneNotSupportedException
		{
			try
			{
				SectionDefinedAlphabetParameterSet clone = (SectionDefinedAlphabetParameterSet) super.clone();
				if(clone.parameters != null){
					clone.parameters.get( 0 ).setNeededReference( clone );
					for(int i=1;i<clone.parameters.size();i++){
						((ParameterSetContainer)clone.parameters.get( i )).getValue().getParameterAt( 1 ).setNeededReference( clone );
					}
				}
				return clone;
			}
			catch( Exception e )
			{
				e.printStackTrace();
				throw new CloneNotSupportedException( e.getCause().getMessage() );
			}

		}

		@Override
		public String getInstanceName()
		{
			return "Alphabets defined by section";
		}

		@Override
		public String getInstanceComment()
		{
			return "Set the alphabets for all positions.";
		}
		

	}

	/**
	 * Class for the parameters of an array of <code>Alphabet</code>s of defined length.
	 * 
	 * @author Jan Grau
	 * 
	 */
	public static class AlphabetArrayParameterSet extends ArrayParameterSet
	{
		private boolean discrete;

		/**
		 * Creates a new <code>AlphabetArrayParameterSet</code> from the information if the array shall contain only
		 * the parameters for discrete <code>Alphabet</code>s.
		 * 
		 * @param discrete
		 *            <code>true</code> if only the parameters for disrete <code>Alphabet</code>s shall be created
		 * @throws Exception
		 *             is thrown if the parameters could not be created
		 */
		public AlphabetArrayParameterSet( boolean discrete ) throws Exception
		{
			super( null, (discrete ? new DiscreteAlphabet.DiscreteAlphabetParameterSet() : new SimpleParameterSet(
					new Parameter[]{ new CollectionParameter( new ParameterSet[]{
							new DiscreteAlphabet.DiscreteAlphabetParameterSet(),
							new ContinuousAlphabet.ContinuousAlphabetParameterSet() }, "Type of alphabet",
							"Select the type of the alphabet", true ) } )

			), "Alphabet", "Set the alphabet" );
		}

		/**
		 * Creates a new <code>AlphabetArrayParameterSet</code> from its XML-representation.
		 * 
		 * @param representation
		 *            the XML-representation
		 * @throws NonParsableException
		 *             is thrown if <code>representation</code> could not be parsed
		 */
		public AlphabetArrayParameterSet( StringBuffer representation ) throws NonParsableException
		{
			super( representation );
		}

		/**
		 * Creates a new <code>AlphabetArrayParameterSet</code> from an array of <code>Alphabet</code>s and the
		 * information if all alphabets are discrete
		 * 
		 * @param alphabets
		 *            the alphabets
		 * @param discrete
		 *            is these are discrete
		 * @throws Exception
		 *             is thrown if the parameters could not be created
		 */
		public AlphabetArrayParameterSet( Alphabet[] alphabets, boolean discrete ) throws Exception
		{
			this( discrete );
			loadParameters();
			this.parameters.get( 0 ).setValue( alphabets.length );
			loadParameters();
			for( int i = 0; i < alphabets.length; i++ )
			{
				if( discrete )
				{
					parameters.get( i + 1 ).setValue( alphabets[i].getCurrentParameterSet() );
				}
				else
				{
					ParameterSet ap = alphabets[i].getCurrentParameterSet();
					((ParameterSet) parameters.get( i + 1 ).getValue()).getParameterAt( 0 ).setValue(
							ap.getInstanceName() );
					((CollectionParameter) ((ParameterSet) parameters.get( i + 1 ).getValue()).getParameterAt( 0 ))
							.getParametersInCollection().getParameterAt(
									((CollectionParameter) ((ParameterSet) parameters.get( i + 1 ).getValue())
											.getParameterAt( 0 )).getSelected() ).setValue( ap );
				}
			}
		}

		public AlphabetArrayParameterSet clone() throws CloneNotSupportedException
		{
			try
			{
				AlphabetArrayParameterSet clone = (AlphabetArrayParameterSet) super.clone();
				return clone;
			}
			catch( Exception e )
			{
				throw new CloneNotSupportedException( e.getCause().getMessage() );
			}
		}

		public StringBuffer toXML()
		{
			StringBuffer buf = super.toXML();
			XMLParser.addTags( buf, "superParameters" );
			XMLParser.appendBooleanWithTags( buf, discrete, "discrete" );
			XMLParser.addTags( buf, "alphabetArrayParameterSet" );
			return buf;
		}

		public void fromXML( StringBuffer representation ) throws NonParsableException
		{
			representation = XMLParser.extractForTag( representation, "alphabetArrayParameterSet" );
			super.fromXML( XMLParser.extractForTag( representation, "superParameters" ) );
			discrete = XMLParser.extractBooleanForTag( representation, "discrete" );
		}

		@Override
		public String getInstanceName()
		{
			return "Alphabet-array";
		}

		@Override
		public String getInstanceComment()
		{
			return "An array of alphabets where each position can have its own alphabet.";
		}
	}
}
