/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.data;

import javax.naming.OperationNotSupportedException;

import de.jstacs.WrongAlphabetException;
import de.jstacs.data.alphabets.ComplementableDiscreteAlphabet;
import de.jstacs.data.sequences.ArbitrarySequence;
import de.jstacs.data.sequences.ByteSequence;
import de.jstacs.data.sequences.DiscreteSequence;
import de.jstacs.data.sequences.IntSequence;
import de.jstacs.data.sequences.RecursiveSequence;
import de.jstacs.data.sequences.ShortSequence;
import de.jstacs.data.sequences.WrongSequenceTypeException;
import de.jstacs.data.sequences.annotation.SequenceAnnotation;

/**
 * This is the main class for all sequences. All sequences are immutable.
 * 
 * @author Jens Keilwagen
 */
public abstract class Sequence implements Comparable<Sequence>
{

	/**
	 * the underlying alphabets
	 */
	protected AlphabetContainer alphabetCon;
	
	/**
	 * The pointer to the reverse complement
	 */
	protected Sequence rc;
	
	/**
	 * The annotation of the sequence.
	 */
	protected SequenceAnnotation[] annotation;
	
	/**
	 * This constructor creates an instance with the AlphabetContainer and the annotation, but without the content.
	 * The content has to be set by the constructor of the extending class.  
	 * 
	 * @param container the AlpahbetContainer of the sequence
	 * @param annotation the annotation of the sequence
	 */
	protected Sequence( AlphabetContainer container, SequenceAnnotation[] annotation )
	{
		if( container == null )
		{
			throw new NullPointerException();
		}
		this.alphabetCon = container;
		if(annotation != null){
			this.annotation = annotation.clone();
		}
	}

	/**
	 * Returns the continuous value of position <code>pos</code>.
	 * 
	 * @param pos
	 *            the position
	 * 
	 * @return the continuous value of position <code>pos</code>
	 */
	public abstract double continuousVal( int pos );

	/**
	 * Returns the discrete value of position <code>pos</code>.
	 * 
	 * @param pos
	 *            the position
	 * 
	 * @return the discrete value of position <code>pos</code>
	 */
	public abstract int discreteVal( int pos );

	public boolean equals( Object o )
	{
		if( o == null )
		{
			return false;
		}
		if( !(o instanceof Sequence) )
		{
			return false;
		}
		Sequence s = (Sequence) o;
		return compareTo( s ) == 0 && alphabetCon.checkConsistency( s.alphabetCon );
	}

	/**
	 * Return the alphabets used in this sequence.
	 * 
	 * @return the alphabets used in this sequence
	 */
	public final AlphabetContainer getAlphabetContainer()
	{
		return alphabetCon;
	}
	
	/**
	 * Returns the annotation of the sequence.
	 * 
	 * @return the annotation of the sequence (can be null).
	 */
	public final SequenceAnnotation[] getAnnotation(){
		if(annotation != null){
			return annotation.clone();
		}else{
			return null;
		}
	}

	/**
	 * This constructor should be used if one wants to create a sample of composite sequences. With this constructor you
	 * are enabled to create a Sample where every sequence has the same AlphabetContainer-instance.
	 * 
	 * <br>
	 * <br>
	 * 
	 * Internally it is checked that the AlphabetContainer matches with the subsequence.
	 * 
	 * @param abc
	 *            the new AlphabetContainer
	 * @param starts the start positions of the junks
	 * @param lengths the length for each junk
	 * 
	 * 
	 * @return the composite sequence
	 */
	public Sequence getCompositeSequence( AlphabetContainer abc, int[] starts, int[] lengths )
	{
		return new CompositeSequence( abc, this, starts, lengths );
	}

	/**
	 * This is an very efficient way to create a composite sequence for sequences with a simple AlphabetContainer.
	 * 
	 * @param starts
	 *            the index of the start position
	 * @param lengths
	 *            the length of the new sequence
	 * 
	 * @return the composite sequence
	 */
	public Sequence getCompositeSequence( int[] starts, int[] lengths )
	{
		return new CompositeSequence( this, starts, lengths );
	}

	/**
	 * This method should be used if one wants to create a sample of subsequences of defined length. With this
	 * constructor you are enabled to create a Sample where every sequence has the same AlphabetContainer-instance.
	 * 
	 * <br>
	 * <br>
	 * 
	 * Internally it is checked that the AlphabetContainer matches with the subsequence.
	 * 
	 * @param abc
	 *            the new AlphabetContainer
	 * @param start
	 *            the index of the start position
	 *            
	 * @return the subsequence
	 */
	public final Sequence getSubSequence( AlphabetContainer abc, int start )
	{
		return getSubSequence( abc, start, getLength() - start );
	}

	/**
	 * This method should be used if one wants to create a sample of subsequences of defined length. With this
	 * constructor you are enabled to create a Sample where every sequence has the same AlphabetContainer-instance.
	 * 
	 * <br>
	 * <br>
	 * 
	 * Internally it is checked that the AlphabetContainer matches with the subsequence.
	 * 
	 * @param abc
	 *            the new AlphabetContainer
	 * @param start
	 *            the index of the start position
	 * @param length
	 *            the length of the new sequence
	 *            
	 * @return the subsequence
	 */
	public Sequence getSubSequence( AlphabetContainer abc, int start, int length )
	{
		if( start == 0 && length == getLength() )
		{
			return this;
		}
		else
		{
			return new SubSequence( abc, this, start, length );
		}
	}

	/**
	 * This is an very efficient way to create a subsequence/suffix for sequences with a simple AlphabetContainer.
	 * 
	 * @param start
	 *            the index of the start position
	 *            
	 * @return the subsequence
	 */
	public final Sequence getSubSequence( int start )
	{
		return getSubSequence( start, getLength() - start );
	}

	/**
	 * This is an very efficient way to create a subsequence of defined length for sequences with a simple
	 * AlphabetContainer.
	 * 
	 * @param start
	 *            the index of the start position
	 * @param length
	 *            the length of the new sequence
	 *            
	 * @return the subsequence
	 */
	public Sequence getSubSequence( int start, int length )
	{
		if( start == 0 && length == getLength() )
		{
			return this;
		}
		else
		{
			return new SubSequence( this, start, length );
		}
	}

	/**
	 * This method allows to append annotation to a sequence.
	 * 
	 * @param add whether to add the new annotation to the existing or not
	 * @param annotation the new annotation
	 * 
	 * @return the new annotated sequence
	 * 
	 * @see Sequence#flatCloneWithoutAnnotation()
	 */
	public Sequence annotate(boolean add, SequenceAnnotation... annotation) {
		Sequence seq = this.flatCloneWithoutAnnotation();
		if(add){
			int num = 0;
			if(annotation != null){
				num += annotation.length;
			}
			if(this.annotation != null){
				num += this.annotation.length;
			}
			SequenceAnnotation[] temp = new SequenceAnnotation[num];
			num = 0;
			for(;annotation != null && num < annotation.length;num++){
				temp[num] = annotation[num];
			}
			for(int i=0;this.annotation != null && i < this.annotation.length;i++){
				temp[num + i] = this.annotation[i];
			}
			seq.annotation = temp;
		}else{
			if(annotation != null){
				seq.annotation = annotation.clone();
			}else{
				seq.annotation = null;
			}
		}
		return seq;
	}
	
	/**
	 * Works in analogy to {@link Sequence#clone()}, but does not clone the annotation. This method is used in {@link Sequence#annotate(boolean, SequenceAnnotation...)}.
	 * @return the cloned {@link Sequence} without annotation
	 */
	protected abstract Sequence flatCloneWithoutAnnotation() ;

	/**
	 * Returns the length of the sequence
	 * 
	 * @return the length
	 */
	public abstract int getLength();

	/**
	 * Returns a String representation of the sequence (normally the sequence in its original alphabet)
	 * 
	 * @return the sequence as <code>String</code>
	 */
	public String toString()
	{
		return toString( alphabetCon.getDelim(), 0, getLength() );
	}

	/**
	 * Returns a string representation of the sequence (normally the sequence in its original alphabet) with
	 * default delimiter as separator.
	 * 
	 * @param start
	 *            the start index (inclusive)
	 * 
	 * @return the sequence as <code>String</code>
	 * 
	 * @see Sequence#toString(String, int, int)
	 */
	public String toString( int start )
	{
		return toString( alphabetCon.getDelim(), start, getLength() );
	}

	/**
	 * Returns a string representation of the sequence (normally the sequence in its original alphabet) with
	 * default delimiter as separator.
	 * 
	 * @param start
	 *            the start index (inclusive)
	 * @param end
	 *            the end index (exclusive)
	 * 
	 * @return the sequence as <code>String</code>
	 *
	 * @see Sequence#toString(String, int, int)
	 */
	public String toString( int start, int end )
	{
		return toString( alphabetCon.getDelim(), start, end );
	}

	public int compareTo( Sequence s )
	{
		//old return this.toString().compareTo( s.toString() );
		int c = alphabetCon.compareTo( s.alphabetCon );
		if( c == 0 )
		{
			int l = getLength(), seqL = s.getLength();
			if( l == seqL )
			{
				int i = 0;
				if( alphabetCon.isDiscrete() )
				{
					int thisVal = 0, seqVal = 0;
					while( i < l && (thisVal = discreteVal(i)) == (seqVal = s.discreteVal(i)) )
					{
						i++;
					}
					return thisVal - seqVal;
				}
				else
				{
					double thisVal = 0, seqVal = 0;
					while( i < l && (thisVal = continuousVal(i)) == (seqVal = s.continuousVal(i)) )
					{
						i++;
					}
					return (int) Math.signum( thisVal - seqVal );
				}
			}
			else
			{
				return l - seqL;
			}
		}
		else
		{
			return c;
		}		
	}

	/**
	 * This method converts a continuous value in a discrete one.
	 * 
	 * @param pos the position
	 * @param content the value at this position
	 * 
	 * @return the discrete value for this position
	 */
	protected int toDiscrete( int pos, double content )
	{
		return alphabetCon.toDiscrete( pos, content );
	}

	/**
	 * Returns a string representation of the sequence (normally the sequence in its original alphabet) with <code>delim</code> as
	 * separator.
	 * 
	 * @param delim
	 *            the delimiter/separator
	 * @param start
	 *            the start index (inclusive)
	 * @param end
	 *            the end index (exclusive)
	 * 
	 * @return the sequence as <code>String</code>
	 */
	public String toString( String delim, int start, int end )
	{
		int i = start, l = end - start;
		StringBuffer buf = new StringBuffer( l * (delim.length() == 0 ? 1 : 6) );

		if( l > 0 )
		{
			buf.append( alphabetCon.getSymbol( i, continuousVal( i++ ) ) );
			while( i < end )
			{
				buf.append( delim + alphabetCon.getSymbol( i, continuousVal( i++ ) ) );
			}
		}
		return buf.toString();
	}

	/**
	 * Creates a sequence from a string based on the given AlphabetContainer using the standard delimiter for this
	 * AlphabetContainer.
	 * 
	 * @param con
	 *            the AlphabetContainer
	 * @param sequence
	 *            the string containing the sequence
	 * 
	 * @return a new sequence instance
	 * 
	 * @throws WrongAlphabetException
	 *             if the <code>sequence</code> is not defined over <code>alphabetContainer</code>
	 * @throws IllegalArgumentException
	 *             if the delimiter is empty and the AlphabetContainer is not discrete
	 */
	public static Sequence create( AlphabetContainer con, String sequence ) throws WrongAlphabetException, IllegalArgumentException
	{
		return create( con, sequence, con.getDelim() );
	}

	/**
	 * Creates a sequence from a string based on the given AlphabetContainer using the given delimiter.
	 * 
	 * @param con
	 *            the AlphabetContainer
	 * @param sequence
	 *            the string containing the sequence
	 * @param delim
	 *            the delimiter
	 * 
	 * @return a new sequence instance
	 * 
	 * @throws WrongAlphabetException
	 *             if the <code>sequence</code> is not defined over <code>alphabetContainer</code>
	 * @throws IllegalArgumentException
	 *             if the delimiter is empty and the alphabetContainer is not discrete
	 */
	public static Sequence create( AlphabetContainer con, String sequence, String delim ) throws WrongAlphabetException, IllegalArgumentException
	{
		return create( con, null, sequence, delim );
	}
	/**
	 * Creates a sequence from a string based on the given AlphabetContainer using the given delimiter.
	 * 
	 * @param con
	 *            the AlphabetContainer
	 * @param annotation the annotation for the sequence
	 * @param sequence
	 *            the string containing the sequence
	 * @param delim
	 *            the delimiter
	 * 
	 * @return a new sequence instance
	 * 
	 * @throws WrongAlphabetException
	 *             if the <code>sequence</code> is not defined over <code>alphabetContainer</code>
	 * @throws IllegalArgumentException
	 *             if the delimiter is empty and the alphabetContainer is not discrete
	 */
	public static Sequence create( AlphabetContainer con, SequenceAnnotation[] annotation, String sequence, String delim ) throws WrongAlphabetException, IllegalArgumentException
	{
		try
		{
			if( con.isDiscrete() )
			{
				// create pure discrete sample
				int l = (int) con.getMaximalAlphabetLength();
				if( l <= Byte.MAX_VALUE )
				{
					return new ByteSequence( con, annotation, sequence, delim );
				}
				else if( l <= Short.MAX_VALUE )
				{
					return new ShortSequence( con, annotation, sequence, delim );
				}
				else if( l <= Integer.MAX_VALUE )
				{
					return new IntSequence( con, annotation, sequence, delim );
				}
				else
				{
					throw new WrongAlphabetException( "Could not encode. Too many symbols." );
				}
			}
			else
			{
				// create hybrid or pure continuous sample
				return new ArbitrarySequence( con, annotation, sequence, delim );
			}
		}
		catch( WrongSequenceTypeException e )
		{
			RuntimeException doesNotHappen = new RuntimeException( e.getMessage() );
			doesNotHappen.setStackTrace( e.getStackTrace() );				
			throw doesNotHappen;
		}
	}

	/**
	 * This method returns a new instance of sequence containing the reverse current sequence.
	 * 
	 * <br>
	 * 
	 * So for instance invoking this method on the sequence &quot;TAATA&quot; returns &quot;ATAAT&quot;.
	 * 
	 * @return the reverse sequence
	 * 
	 * @throws OperationNotSupportedException
	 *             if the current sequence is based on an AlphabetContainer that is not simple.
	 */
	public final Sequence reverse() throws OperationNotSupportedException
	{
		return reverse( 0, getLength() );
	}
	
	/**
	 * This method returns a new instance of sequence containing a part of the reverse current sequence.
	 * 
	 * @param start the start position (inclusive) in the original sequence
	 * @param end the end position (exclusive) in the original sequence
	 * 
	 * @return the reverse sequence
	 * 
	 * @throws OperationNotSupportedException
	 *             if the current sequence is based on an AlphabetContainer that is not simple.
	 */
	public Sequence reverse( int start, int end ) throws OperationNotSupportedException
	{
		if( alphabetCon.isSimple() )
		{
			int i = 0, j = end;
			try
			{
				if( this instanceof DiscreteSequence )
				{
					int[] erg = new int[j-start];
					for( j--; j >= start; j--, i++ )
					{
						erg[i] = discreteVal( j );
					}
					return new IntSequence( alphabetCon, erg );
				}
				else
				{
					double[] erg = new double[j-start];
					for( j--; j >= start; j--, i++ )
					{
						erg[i] = continuousVal( j );
					}
					return new ArbitrarySequence( alphabetCon, erg );
				}
			}
			catch( Exception e )
			{
				// the current sequence is defined on the correct alphabet => so no WrongAlphabetException can occur
				RuntimeException doesNotHappen = new RuntimeException( e.getMessage() );
				doesNotHappen.setStackTrace( e.getStackTrace() );				
				throw doesNotHappen;
			}
		}
		else
		{
			throw new OperationNotSupportedException( "The sequence has to be simple." );
		}
	}

	/**
	 * This method returns a new instance of sequence containing the complementary current sequence.
	 * 
	 * <br>
	 * 
	 * So for instance invoking this method on the sequence &quot;TAATA&quot; with an {@link AlphabetContainer} on
	 * {@link de.jstacs.data.alphabets.DNAAlphabet} returns &quot;ATTAT&quot;.
	 * 
	 * @return the complementary sequence
	 * 
	 * @throws OperationNotSupportedException
	 *             if the current sequence is not based on a {@link ComplementableDiscreteAlphabet}
	 *             
	 * @see ComplementableDiscreteAlphabet
	 */
	public Sequence complement() throws OperationNotSupportedException
	{
		return complement( 0, getLength() );
	}
	
	/**
	 * This method returns a new instance of sequence containing a part of the complementary current sequence.
	 * 
	 * <br>
	 * 
	 * So for instance invoking this method on the sequence &quot;TAATA&quot; with an {@link AlphabetContainer} on
	 * {@link de.jstacs.data.alphabets.DNAAlphabet} returns &quot;ATTAT&quot;.
	 * 
	 * @param start the start position (inclusive) in the original sequence 
	 * @param end the end position (exclusive) in the original sequence
	 * 
	 * @return the complementary sequence
	 * 
	 * @throws OperationNotSupportedException
	 *             if the current sequence is not based on a {@link ComplementableDiscreteAlphabet}
	 *             
	 * @see ComplementableDiscreteAlphabet
	 */
	public Sequence complement( int start, int end ) throws OperationNotSupportedException
	{
		if( alphabetCon.isReverseComplementable() )
		{
			ComplementableDiscreteAlphabet cda = (ComplementableDiscreteAlphabet) alphabetCon.getAlphabetAt( 0 );
			try
			{
				if( cda.length() > 127)
				{
					int[] erg = new int[end-start];
					for( int i = 0, j = start; j < end; i++, j++ )
					{
						erg[i] = cda.getComplementaryCode( discreteVal( j ) );
					}

					return new IntSequence( alphabetCon, erg );
				}else{
					byte[] erg = new byte[end-start];
					for( int i = 0, j = start; j < end; i++, j++ )
					{
						erg[i] = (byte) cda.getComplementaryCode( discreteVal( j ) );
					}

					return new ByteSequence( alphabetCon, erg );
				}
			}catch( Exception e )
			{
				// the current sequence is defined on the correct alphabet => so no WrongAlphabetException can occur
				RuntimeException doesNotHappen = new RuntimeException( e.getMessage() );
				doesNotHappen.setStackTrace( e.getStackTrace() );				
				throw doesNotHappen;
			}
		}
		else
		{
			throw new OperationNotSupportedException( "The alphabet of sequence has to be complementable." );
		}
	}

	/**
	 * This method returns a new sequence instance containing the complementary current sequence. For more details
	 * see the methods <code>reverse()</code> and <code>complement()</code>.
	 * 
	 * @return the reverse complementary sequence
	 * 
	 * @throws OperationNotSupportedException
	 *             if the current sequence is not discrete and simple
	 * 
	 * @see Sequence#reverse()
	 * @see Sequence#complement()
	 * @see ComplementableDiscreteAlphabet
	 */
	public Sequence reverseComplement() throws OperationNotSupportedException
	{
		return reverseComplement( 0, getLength() );
	}
	
	/**
	 * This method returns a new sequence instance containing a part of the complementary current sequence. For more details
	 * see the methods <code>reverse()</code> and <code>complement()</code>.
	 * 
	 * @param start the start position (inclusive) in the original sequence 
	 * @param end the end position (exclusive) in the original sequence
	 * 
	 * @return the reverse complementary sequence
	 * 
	 * @throws OperationNotSupportedException
	 *             if the current sequence is not discrete and simple
	 * 
	 * @see Sequence#reverse()
	 * @see Sequence#complement()
	 * @see ComplementableDiscreteAlphabet
	 */
	public Sequence reverseComplement( int start, int end ) throws OperationNotSupportedException
	{
		if( rc != null && start == 0 && end == getLength() )
		{
			return rc;
		}
		else if( alphabetCon.isReverseComplementable() )
		{
			ComplementableDiscreteAlphabet cda = (ComplementableDiscreteAlphabet) alphabetCon.getAlphabetAt( 0 );
			int i = 0, j = end;
			try
			{
				Sequence revComp;
				if(cda.length() > 127){
					int[] erg = new int[end-start];
					for( j--; j >= start; j--, i++ )
					{
						erg[i] = cda.getComplementaryCode( discreteVal( j ) );
					}

					revComp = new IntSequence( alphabetCon, erg );
					
				}else{
					byte[] erg = new byte[end-start];
					for( j--; j >= start; j--, i++ )
					{
						erg[i] = (byte) cda.getComplementaryCode( discreteVal( j ) );
					}

					revComp = new ByteSequence( alphabetCon, erg );
				}
				if( start == 0 && end == getLength() )
				{
					rc = revComp;
					rc.rc = this;
				}
				return revComp;
			}
			catch( Exception e )
			{
				// the current sequence is defined on the correct alphabet => so no WrongAlphabetException can occur
				RuntimeException doesNotHappen = new RuntimeException( e.getMessage() );
				doesNotHappen.setStackTrace( e.getStackTrace() );				
				throw doesNotHappen;
			}
		}
		else
		{
			throw new OperationNotSupportedException( "The alphabet of sequence has to be reverse-complementable." );
		}
	}
	
	public int hashCode()
	{
	    int len = getLength();
	    if( alphabetCon.isDiscrete() )
	    {
	    	//@see String#hashCode()
			int h = 0;
	        for (int i = 0; i < len; i++) {
	            h = 31*h + discreteVal( i );
	        }
	        return h;
	    }
	    else
	    {
	    	//@see String#hashCode()
	    	//@see Double#hashCode()
	    	double h = 0;
	        for (int i = 0; i < len; i++) {
	            h = 31*h + continuousVal( i );
	        }
	        long bits = Double.doubleToLongBits(h);
	    	return (int)(bits ^ (bits >>> 32));
	    }
    }

	/**
	 * The class handles composite sequences. A composite sequence consists of several (partial) sequences. A biological
	 * example are promoters like in eucaryots (-10 and -35-box).
	 * 
	 * @author Jens Keilwagen
	 * 
	 */
	protected static class CompositeSequence extends RecursiveSequence
	{
		private static final long serialVersionUID = 1L;

		private int[] starts, lengths;

		private CompositeSequence( AlphabetContainer abc, SequenceAnnotation[] annotation, Sequence seq, int[] starts, int[] lengths, boolean check )
		{
			super( abc, annotation, seq );
			// TODO make this faster, no new object
			if( check && !abc.checkConsistency( seq.getAlphabetContainer().getCompositeContainer( starts, lengths ) ) )
			{
				throw new IllegalArgumentException( "Wrong AlphabetContainer." );
			}
			if( starts.length != lengths.length )
			{
				throw new IllegalArgumentException( "starts and lengths have to be from the same dimension" );
			}
			this.starts = starts.clone();
			this.lengths = lengths.clone();
			for( int i = 0; i < lengths.length; i++ )
			{
				if( starts[i] < 0 || seq.getLength() <= starts[i] )
				{
					throw new IllegalArgumentException( i + "-th start has to be from range [0," + (seq.getLength() - 1)
							+ "]" );
				}
				if( lengths[i] < 0 || seq.getLength() < starts[i] + lengths[i] )
				{
					throw new IllegalArgumentException( i + "-th length has to be from range [0,"
							+ (seq.getLength() - starts[i]) + "]" );
				}
				this.starts[i] = starts[i];
				this.lengths[i] = lengths[i];
			}
		}

		/**
		 * This is an very effient way to create a composite sequence for sequences with a simple AlphabetContainer.
		 * 
		 * @param seq
		 *            the original sequence
		 * @param starts
		 *            the index of the start position
		 * @param lengths
		 *            the length of the new sequence
		 */
		public CompositeSequence( Sequence seq, int[] starts, int[] lengths )
		{
			this( seq.getAlphabetContainer().getCompositeContainer( starts, lengths ),null, seq, starts, lengths, false );
		}

		/**
		 * This constructor should be used if one wants to create a sample of composite sequences. With this constructor
		 * you are enabled to create a Sample where every sequence has the same AlphabetContainer-instance.
		 * 
		 * <br>
		 * <br>
		 * 
		 * Internally it is checked that the AlphabetContainer matches with the subsequence.
		 * 
		 * @param abc
		 *            the new AlphabetContainer
		 * @param seq
		 *            the original sequence
		 * @param starts
		 *            the array of indices of the start position
		 * @param lengths
		 *            the array of (sub)length of the new sequence
		 */
		public CompositeSequence( AlphabetContainer abc, Sequence seq, int[] starts, int[] lengths )
		{
			this( abc, null, seq, starts, lengths, true );
		}

		protected int getIndex( int pos )
		{
			int sum = 0, i = 0;
			while( i < lengths.length && sum + lengths[i] <= pos )
			{
				sum += lengths[i++];
			}
			return starts[i] + (pos - sum);
		}

		public int getLength()
		{
			int sum = 0;
			for( int i = 0; i < lengths.length; i++ )
			{
				sum += lengths[i];
			}
			return sum;
		}

		@Override
		protected Sequence flatCloneWithoutAnnotation() {
			return new CompositeSequence(alphabetCon,null,content,starts,lengths,false);
		}
	}

	/**
	 * This class handles subsequences. Subsequences are often used to extract the sequence of a sliding window on a
	 * long sequence. The class is implemented in such a way, that it avoids chains of SubSequences.
	 * 
	 * @author Jens Keilwagen
	 */
	protected static class SubSequence extends RecursiveSequence
	{
		private static final long serialVersionUID = 1L;

		private int start, length;

		private SubSequence( AlphabetContainer abc, SequenceAnnotation[] annotation, Sequence seq, int start, int length, boolean check )
		{
			super( abc,annotation, (seq instanceof SubSequence) ? ((SubSequence) seq).content : seq );
			// TODO make this faster, no new object
			if( check && !abc.checkConsistency( seq.getAlphabetContainer().getSubContainer( start, length ) ) )
			{
				throw new IllegalArgumentException( "Wrong AlphabetContainer." );
			}
			if( start < 0 || start > seq.getLength() )
			{
				throw new IllegalArgumentException( "Illegal start position: start="+start+", length="+seq.getLength() );
			}
			if( length < 0 || start + length > seq.getLength() )
			{
				throw new IllegalArgumentException( "Illegal length." );
			}
			if( seq instanceof SubSequence )
			{
				SubSequence s = (SubSequence) seq;
				this.start = s.start + start;
			}
			else
			{
				this.start = start;
			}
			this.length = length;
		}

		/**
		 * This constructor should be used if one wants to create a sample of subsequences of defined length. With this
		 * constructor you are enabled to create a Sample where every sequence has the same AlphabetContainer-instance.
		 * 
		 * <br>
		 * <br>
		 * 
		 * Internally it is checked that the AlphabetContainer matches with the subsequence.
		 * 
		 * @param abc
		 *            the new AlphabetContainer
		 * @param seq
		 *            the original sequence
		 * @param start
		 *            the index of the start position
		 * @param length
		 *            the length of the new sequence
		 */
		public SubSequence( AlphabetContainer abc, Sequence seq, int start, int length )
		{
			this( abc, null, (seq instanceof SubSequence) ? ((SubSequence) seq).content : seq, start, length, true );
		}

		/**
		 * This is an very efficient way to create a subsequence of defined length for sequences with a simple
		 * AlphabetContainer.
		 * 
		 * @param seq
		 *            the original sequence
		 * @param start
		 *            the index of the start position
		 * @param length
		 *            the length of the new sequence
		 */
		public SubSequence( Sequence seq, int start, int length )
		{
			this( seq.getAlphabetContainer().getSubContainer( start, length ), null, seq, start, length, false );
		}

		public Sequence reverseComplement( int start, int end ) throws OperationNotSupportedException
		{
			if( rc != null && start == 0 && end == getLength() )
			{
				return rc;
			}
			else if( content.rc != null )
			{
				Sequence revComp = new SubSequence( content.rc, content.rc.getLength() - this.start - end, end-start );
				if( start == 0 && end == getLength() )
				{
					rc = revComp;
					rc.rc = this;
				}
				return revComp;
			}
			return super.reverseComplement(start, end);
		}
		
		protected int getIndex( int pos )
		{
			if( pos < 0 || pos >= length )
			{
				throw new ArrayIndexOutOfBoundsException( pos );
			}
			return start + pos;
		}

		public int getLength()
		{
			return length;
		}

		@Override
		protected Sequence flatCloneWithoutAnnotation() {
			return new SubSequence(alphabetCon,null,content,start,length,false);
		}
	}
}