/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.data.alphabets;

import de.jstacs.DataType;
import de.jstacs.NonParsableException;
import de.jstacs.data.Alphabet;
import de.jstacs.io.XMLParser;
import de.jstacs.parameters.SimpleParameter;

/**
 * Class for a continuous alphabet.
 * 
 * @author Jens Keilwagen
 */
public class ContinuousAlphabet extends Alphabet
{
	private static final String XML_TAG = "ContinuousAlphabet";

	/**
	 * The minmal and the maximal value.
	 */
	private double min, max;

	private ContinuousAlphabetParameterSet parameters;

	/**
	 * The constructor for the <code>InstantiableFromParameterSet</code> interface.
	 * 
	 * @param parameters
	 *            the ParameterSet
	 */
	public ContinuousAlphabet( ContinuousAlphabetParameterSet parameters )
	{
		this( (Double) parameters.getParameterAt( 0 ).getValue(), (Double) parameters.getParameterAt( 1 ).getValue() );
		try
		{
			this.parameters = (ContinuousAlphabetParameterSet) parameters.clone();
		}
		catch( Exception e )
		{
			this.parameters = null;
		}
	}

	/**
	 * Creates a new <code>ContinuousAlphabet</code> from a minimum and a maximum value
	 * 
	 * @param min
	 *            the minimum
	 * @param max
	 *            the maximum
	 * @throws IllegalArgumentException
	 *             is thrown if minimum or maximum could not be set
	 */
	public ContinuousAlphabet( double min, double max ) throws IllegalArgumentException
	{
		if( Double.isInfinite( min ) || Double.isNaN( min ) || Double.isNaN( max ) || Double.isInfinite( max ) )
		{
			throw new IllegalArgumentException( "min and max have to be numbers (not infinity, NaN, ...)" );
		}
		if( min >= max )
		{
			throw new IllegalArgumentException( "constraint violated: min < max" );
		}
		this.min = min;
		this.max = max;
	}

	/**
	 * Extracts the alphabet from the StringBuffer.
	 * 
	 * @param xml
	 *            the XML stream
	 * 
	 * @throws NonParsableException
	 *             if the stream is not parsable
	 */
	public ContinuousAlphabet( StringBuffer xml ) throws NonParsableException
	{
		StringBuffer help = XMLParser.extractForTag( xml, XML_TAG );
		min = XMLParser.extractDoubleForTag( help, "MIN" );
		max = XMLParser.extractDoubleForTag( help, "MAX" );
	}

	public ContinuousAlphabetParameterSet getCurrentParameterSet() throws Exception
	{
		if( parameters != null )
		{
			return (ContinuousAlphabetParameterSet) parameters.clone();
		}
		else
		{
			return new ContinuousAlphabetParameterSet( min, max );
		}
	}

	public StringBuffer toXML()
	{
		StringBuffer xml = new StringBuffer( 100 );
		xml.append( "\t" );
		XMLParser.appendDoubleWithTags( xml, min, "MIN" );
		xml.append( "\t" );
		XMLParser.appendDoubleWithTags( xml, max, "MAX" );
		XMLParser.addTags( xml, XML_TAG );
		return xml;
	}

	public int compareTo( Alphabet a )
	{
		if( !getClass().equals( a.getClass() ) )
		{
			return getClass().getName().compareTo( a.getClass().getName() );
		}
		else
		{
			ContinuousAlphabet b = (ContinuousAlphabet) a;
			int s = (int) Math.signum( min - b.min );
			if( s == 0 )
			{
				return (int) Math.signum( max - b.max );
			}
			else
			{
				return s;
			}
		}
	}

	public double getMin()
	{
		return min;
	}

	/**
	 * Returns the maximal value of this alphabet.
	 * 
	 * @return the maximal value of this alphabet
	 */
	public double getMax()
	{
		return max;
	}

	/**
	 * Returns <code>true</code> if <code>candidat</code> is an element of the internal interval.
	 * 
	 * @param candidat
	 *            the value to be tested
	 * 
	 * @return <code>true</code> if <code>candidat</code> is an element of the internal interval
	 */
	public final boolean isEncodedSymbol( double candidat )
	{
		return (min <= candidat) && (candidat <= max);
	}

	public double length()
	{
		return max - min;
	}

	public String toString()
	{
		return "[" + min + ", " + max + "]";
	}

	/**
	 * Class for the <code>ParameterSet</code> of a <code>ContinuousAlphabet</code>.
	 * 
	 * @author Jan Grau
	 * 
	 */
	public static class ContinuousAlphabetParameterSet extends AlphabetParameterSet
	{
		/**
		 * Creates a new <code>ContinuousAlphabetParameterSet</code> with empty values.
		 */
		public ContinuousAlphabetParameterSet()
		{
			super( ContinuousAlphabet.class );
		}

		/**
		 * Creates a new <code>ContinuousAlphabetParameterSet</code> from a minimum and a maximum value
		 * 
		 * @param min
		 *            the minimum
		 * @param max
		 *            the maximum
		 * @throws Exception
		 *             is thrown if minimum or maximum could not be set
		 */
		public ContinuousAlphabetParameterSet( double min, double max ) throws Exception
		{
			this();
			loadParameters();
			parameters.get( 0 ).setValue( new Double( min ) );
			parameters.get( 1 ).setValue( new Double( max ) );
		}

		/**
		 * Creates a new <code>ContinousParameterSet</code> from its XML-representation.
		 * 
		 * @param representation
		 *            the XML-representation
		 * @throws NonParsableException
		 *             is thrown if <code>representation</code> could not be parsed
		 */
		public ContinuousAlphabetParameterSet( StringBuffer representation ) throws NonParsableException
		{
			super( representation );
		}

		@Override
		protected void loadParameters() throws Exception
		{
			initParameterList();
			parameters.add( new SimpleParameter( DataType.DOUBLE, "Minimum", "The minimal value of the alphabet.",
					true ) );
			parameters
					.add( new SimpleParameter( DataType.DOUBLE, "Maximum", "The maximum value of the alphabet", true ) );
		}

		@Override
		public String getInstanceComment()
		{
			return "An alphabet that consists of real values between a minimum and a maximum value.";
		}
	}
}