/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.data.sequences;

import de.jstacs.data.AlphabetContainer;
import de.jstacs.data.Sequence;
import de.jstacs.data.sequences.annotation.SequenceAnnotation;

/**
 * This is the main class for subsequences, composite sequences, ... . All these sequences are defined on an existing
 * sequence.
 * 
 * @author Jens Keilwagen
 */
public abstract class RecursiveSequence extends Sequence
{
	/**
	 * The internal sequence
	 */
	protected Sequence content;

	/**
	 * This constructor creates a recursive sequence.
	 * 
	 * @param alphabet the AlphabetContainer
	 * @param annotation the annotation of the recursive sequence
	 * @param seq the sequence
	 */
	public RecursiveSequence( AlphabetContainer alphabet, SequenceAnnotation[] annotation, Sequence seq )
	{
		super( alphabet, annotation );
		content = seq;
	}
	
	/**
	 * This constructor creates a recursive sequence using the annotation of the given sequence.
	 * 
	 * @param alphabet the AlphabetContainer
	 * @param seq the sequence
	 */
	public RecursiveSequence( AlphabetContainer alphabet, Sequence seq )
	{
		this(alphabet,seq.getAnnotation(),seq);
	}

	@Override
	public final double continuousVal( int pos )
	{
		return content.continuousVal( getIndex( pos ) );
	}

	@Override
	public final int discreteVal( int pos )
	{
		return content.discreteVal( getIndex( pos ) );
	}

	/**
	 * Return the index in the internal sequence
	 * 
	 * @param pos
	 *            the index in the external sequence
	 * 
	 * @return the index in the internal sequence
	 */
	protected abstract int getIndex( int pos );
	
	
}
