/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.models.discrete.inhomogeneous.parameters;

import de.jstacs.DataType;
import de.jstacs.NonParsableException;
import de.jstacs.data.AlphabetContainer;
import de.jstacs.models.discrete.inhomogeneous.BayesianNetworkModel;
import de.jstacs.models.discrete.inhomogeneous.StructureLearner.LearningType;
import de.jstacs.models.discrete.inhomogeneous.StructureLearner.ModelType;
import de.jstacs.parameters.EnumParameter;
import de.jstacs.parameters.SimpleParameter;
import de.jstacs.parameters.SimpleParameter.IllegalValueException;
import de.jstacs.parameters.validation.NumberValidator;

/**
 * The ParameterSet for the class BayesianNetworkModel.
 * 
 * @author Jens Keilwagen
 */
public class BayesianNetworkModelParameterSet extends IDGMParameterSet
{
	/**
	 * The constructor for the {@link de.jstacs.Storable} interface.
	 * 
	 * @param s
	 *            the StringBuffer
	 * 
	 * @throws NonParsableException
	 *             if the StringBuffer is not parsable
	 */
	public BayesianNetworkModelParameterSet( StringBuffer s ) throws NonParsableException
	{
		super( s );
	}

	/**
	 * The simple constructor for an empty parameter set of a BayesianNetworkModel.
	 */
	public BayesianNetworkModelParameterSet()
	{
		super( BayesianNetworkModel.class );
	}

	/**
	 * This is the constructor of a filled parameter set for a BayesianNetworkModel.
	 * 
	 * @param alphabet
	 *            the alphabet container that is used in the model
	 * @param length
	 *            the length of the model (has to be positive)
	 * @param ess
	 *            the equivalent sample size (has to be positive)
	 * @param description
	 *            a short description of the model (used for a better handling of the object by the user)
	 * @param model
	 *            the type of model: IMM, PMM or BN
	 * @param order
	 *            the order of the model
	 * @param method
	 *            the method how to learn the structure (only relevant for PMM, BN): ML_OR_MAP or BMA
	 * 
	 * @see de.jstacs.models.discrete.inhomogeneous.StructureLearner.ModelType#IMM
	 * @see de.jstacs.models.discrete.inhomogeneous.StructureLearner.ModelType#PMM
	 * @see de.jstacs.models.discrete.inhomogeneous.StructureLearner.ModelType#BN
	 * 
	 * @see de.jstacs.models.discrete.inhomogeneous.StructureLearner.LearningType#ML_OR_MAP
	 * @see de.jstacs.models.discrete.inhomogeneous.StructureLearner.LearningType#BMA
	 * 
	 * @throws Exception if something went wrong
	 */
	public BayesianNetworkModelParameterSet( AlphabetContainer alphabet, int length, double ess, String description,
			ModelType model, byte order, LearningType method ) throws Exception
	{
		super( BayesianNetworkModel.class, alphabet, length, ess, description );
		parameters.get( 2 ).setValue( model );
		parameters.get( 3 ).setValue( order );
		parameters.get( 4 ).setValue( method );
	}

	protected void loadParameters() throws Exception
	{
		super.loadParameters();
		parameters.add( new EnumParameter( ModelType.class, "the standard model that should be learned", true ) );
		parameters.add( new SimpleParameter( DataType.BYTE, "Markov order",
				"the used markov order is the number of used dependencies for (each) random variable", true,
				new NumberValidator<Byte>( (byte) 0, Byte.MAX_VALUE ) ) );
		parameters.add( new EnumParameter( LearningType.class, "the learning method for the parameters of the model", true ) );
	}

	/**
	 * This method allows a simple change of the model type.
	 * 
	 * @param modelType &quot;iMM&quot;, &quot;pMM&quot; or &quot;BN&quot;
	 * 
	 * @throws IllegalValueException if the <code>modelType</code> is illegal
	 */
	public void setModelType( String modelType ) throws IllegalValueException
	{
		parameters.get( 2 ).setValue( modelType );
	}

	public String getInstanceComment()
	{
		return "holds the parameters for a standard model (i.e. inhomogeneous Markov model (iMM), permuted Markov model (pMM) or Bayesian network (BN))";
	}

	/**
	 * This method returns a short description of the model.
	 * 
	 * @return a short description of the model
	 */
	public String getModelInstanceName()
	{
		return getModelInstanceName( (ModelType) parameters.get( 2 ).getValue(), (Byte) parameters.get( 3 ).getValue(),
					(LearningType) parameters.get( 4 ).getValue(), (Double) parameters.get( 0 ).getValue() );
	}
}
