/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.parameters;

import de.jstacs.DataType;
import de.jstacs.NonParsableException;
import de.jstacs.io.XMLParser;
import de.jstacs.parameters.SimpleParameter.DatatypeNotValidException;
import de.jstacs.parameters.SimpleParameter.IllegalValueException;

/**
 * Class for a <code>Parameter</code> that provides a collection of possible values. The user can then select one
 * or more values out of the collection as values of this <code>Parameter</code>.
 * 
 * @author Jan Grau
 * @see CollectionParameter
 *
 */
public class MultiSelectionCollectionParameter extends CollectionParameter implements RangeIterator {

    private boolean[] selected;
    private boolean[] defaultSelected;
    private int current;
    private boolean ranged;
    
    /**
	 * Constructor for a <code>MultiSelectionCollectionParameter</code>. The first option in the selection is selected by default.
	 * @param datatype the data type of the parameters in the collection
	 * @param keys the keys/names of the values in the collection, this is the name the user will see in the user interface
	 * @param values the values the names stand for. This array must be of the same length as the keys. A key at a certain position belongs to the value at the same position in the array.
	 * @param name the name of the parameter
	 * @param comment a comment on the parameter
	 * @param required true if the parameter is required, false otherwise
	 * @throws InconsistentCollectionException if the length of the keys and the values is different or the collection is inconsistent for some other reason, an <code>InconsistentCollectionException</code is thrown
	 * @throws IllegalValueException if one of the values in <code>values</code> is not of type <code>datatype</code>, an <code>IllegalValueException</code> is thrown
	 * @throws DatatypeNotValidException is thrown if the <code>datatype</code> is not one of the allowed values
	 */
    public MultiSelectionCollectionParameter(DataType datatype, String[] keys, Object[] values, String name, String comment, boolean required) throws InconsistentCollectionException, IllegalValueException, DatatypeNotValidException{
        super(datatype,keys,values,name,comment,required);
        selected = new boolean[keys.length];
        defaultSelected = selected.clone();
        setValue(new String[]{keys[0]});
    }
    
    /**
	 * Constructor for a <code>MultiSelectionCollectionParameter</code>. The first option in the selection is selected by default.
	 * @param datatype the data type of the parameters in the collection
	 * @param keys the keys/names of the values in the collection, this is the name the user will see in the user interface
	 * @param values the values the names stand for. This array must be of the same length as the keys. A key at a certain position belongs to the value at the same position in the array.
	 * @param comments the comments on the values in the collection
	 * @param name the name of the parameter
	 * @param comment a comment on the parameter
	 * @param required true if the parameter is required, false otherwise
	 * @throws InconsistentCollectionException if the length of the keys and the values is different or the collection is inconsistent for some other reason, an <code>InconsistentCollectionException</code is thrown
	 * @throws IllegalValueException if one of the values in <code>values</code> is not of type <code>datatype</code>, an <code>IllegalValueException</code> is thrown
	 * @throws DatatypeNotValidException is thrown if the <code>datatype</code> is not one of the allowed values
	 */
    public MultiSelectionCollectionParameter(DataType datatype, String[] keys, Object[] values, String[] comments, String name, String comment, boolean required) throws InconsistentCollectionException, IllegalValueException, DatatypeNotValidException{
        super(datatype,keys,values,comments,name,comment,required);
        selected = new boolean[keys.length];
        defaultSelected = selected.clone();
        setValue(new String[]{keys[0]});
    }
    
    /**
     * Creates a new <code>MultiSelectionCollectionParameter</code> from an array of <code>ParameterSet</code>s.  The first option in the selection is selected by default.
     * @param values the options/values in the collection
     * @param name the name of this <code>MultiSelectionCollectionParameter</code>
     * @param comment the comment on this <code>MultiSelectionCollectionParameter</code>
     * @param required <code>true</code> if this <code>MultiSelectionCollectionParameter</code> is required
     */
    public MultiSelectionCollectionParameter(ParameterSet[] values, String name, String comment, boolean required){
        super(values,name,comment,required);
        selected = new boolean[values.length];
        defaultSelected = selected.clone();
        try {
			setValue(new String[]{values[0].getInstanceName()});
		} catch (IllegalValueException doesnothappen) {
			doesnothappen.printStackTrace();
		}
    }
    
    /**
     * Creates a new <code>MultiSelectionCollectionParameter</code> from its XML-representation
     * @param representation the XML-representation
     * @throws NonParsableException is thrown if <code>representation</code> could not be parsed
     */
    public MultiSelectionCollectionParameter(StringBuffer representation) throws NonParsableException{
        super(representation);
    }
    
    /**
	 * Creates a new <code>MultiSelectionCollectionParameter</code> from the necessary field. This constructor should be used to clone
	 * a current instance.
	 * @param options the options of the <code>CollectionParameter</code>
	 * @param selected the currently selected values
	 * @param defaultSelected the values selected by default
	 * @param userSelected <code>true</code> is the current value was selected by the user
	 * @param name the name
	 * @param comment the comment
	 * @param required <code>true</code> if this <code>CollectionParameter</code> is required
	 * @param datatype the datatype of this <code>CollectionParameter</code>
	 * @param errorMessage the error-message of the last error or <code>null</code>
	 * @param current the currently used value in the set of selected value
     * @param makeRanged replace the <code>Parameter</code> in <code>options</code> with their ranged equivalent (if allowed)
     * @throws Exception 
	 */
    protected MultiSelectionCollectionParameter(ParameterSet options, boolean[] selected, boolean[] defaultSelected, boolean userSelected, String name, String comment, boolean required, DataType datatype, String errorMessage, int current, boolean makeRanged) throws Exception{
        super(options,0,0,userSelected,name,comment,required,datatype,errorMessage, false);
        this.selected = selected.clone();
        this.defaultSelected = defaultSelected.clone();
        this.current = current;
        this.ranged = makeRanged;
        if(ranged){
            this.parameters.makeRanged();
        }
    }
    
    public MultiSelectionCollectionParameter clone() throws CloneNotSupportedException{
        MultiSelectionCollectionParameter clone = (MultiSelectionCollectionParameter) super.clone();
        clone.defaultSelected = defaultSelected.clone();
        clone.selected = selected.clone();
        return clone;
        
    }
    
    /**
     * Sets the selection of the option <code>key</code> to the value of <code>selected</code>
     * @param key the key of the option
     * @param selected the selection value
     * @return true if the key could be found and set, false otherwise
     */
    public boolean setSelected(String key, boolean selected){
        for(int i=0;i<parameters.getNumberOfParameters();i++){
            if(parameters.getParameterAt(i).getName().equals(key)){
                this.selected[i] = selected;
                return true;
            }
        }
        return false;
    }
    
    /**
     * Returns the selection value of the option <code>key</code>
     * @param key the key of the option
     * @return the selection value or false if no such option exists
     */
    public boolean isSelected(String key){
        for(int i=0;i<parameters.getNumberOfParameters();i++){
            if(parameters.getParameterAt(i).getName().equals(key)){
                return this.selected[i];
            }
        }
        return false;
    }
    
    /**
     * Returns the value for the option with key <code>key</code>.
     * @param key the key
     * @return the value or <code>null</code> if the corresponding option either does not exists or is not selected
     */
    public Object getValueFor(String key){
        for(int i=0;i<parameters.getNumberOfParameters();i++){
            if(parameters.getParameterAt(i).getName().equals(key)){
                if(this.selected[i]){
                    return parameters.getParameterAt(i).getValue();
                }else{
                    return null;
                }
            }
        }
        return null;
    }
    
    public boolean checkValue(Object value) {
        //System.out.println("VALUE:::::::::: "+value);
        if(value instanceof String){
            value = new String[]{(String)value};
        }
        if(value instanceof String[]){
            String[] tmp = (String[]) value;
            if(tmp.length == 0 && isRequired()){
                errorMessage = "At least one value must be selected!";
                return false;
            }
            for(int i=0;i<tmp.length;i++){
                if(! super.checkValue(tmp[i])){
                    return false;
                }
            }
            errorMessage = null;
            return true;
        }else{
            errorMessage = "Value not of the correct type";
            return false;
        }
    }

    public void setValue(Object value) throws IllegalValueException {
        if(value instanceof String){
            value = new String[]{(String)value};
        }
        if(checkValue(value)){
            selected = new boolean[selected.length];
            current = -1;
            String[] tmp = (String[]) value;
            for(int i=0; i<tmp.length;i++){
                for(int j=0;j<parameters.getNumberOfParameters();j++){
                    if(parameters.getParameterAt(j).getName().equals(tmp[i])){
                        selected[j] = true;
                        if(current == -1 || j < current){
                            current = j;
                        }
                    }
                }
            }
            userSelected = true;
        }
    }

    
    
    public int getSelected(){
        return current;
    }
    
    /**
     * Sets the selection of option no. <code>idx</code> to <code>selected</code>
     * @param idx the index of the option
     * @param selected the selection value
     * @return true if the option exists and could be set, false otherwise
     */
    public boolean setSelected(int idx, boolean selected){
        if(idx < this.selected.length){
            this.selected[idx] = selected;
            return true;
        }else{
            return false;
        }
    }
    
    /**
     * Returns <code>true</code> if the option at position <code>idx</code> is selected.
     * @param idx the position
     * @return if the option at position <code>idx</code> is selected
     */
    public boolean isSelected(int idx){
        return idx < selected.length && selected[idx];
    }
    
    /**
     * Returns the value of the option no. <code>idx</code>
     * @param idx the index of the option
     * @return the value or <code>null</code> if the corresponding option either does not exists or is not selected
     */
    public Object getValueFor(int idx){
        if(idx < selected.length && selected[idx]){
            return parameters.getParameterAt(idx).getValue();
        }else{
            return null;
        }
    }
    
    /**
     * Returns the values of all selected options as an array.
     * @return the values
     */
    public Object[] getValues() {
        int count = 0;
        for(int i=0;i<selected.length;i++){
            if(selected[i]){
                count++;
            }
        }
        Object[] values = new Object[count];
        count = 0;
        for(int i=0;i<parameters.getNumberOfParameters();i++){
            if(selected[i]){
                values[count++] = parameters.getParameterAt(i).getValue();
            }
        }
        return values;
    }
    
    public Object getValue(){
        if(current > -1){
            return parameters.getParameterAt(current).getValue();
        }else{
            return null;
        }
    }

    public boolean hasDefaultOrIsSet() {
        int numSelected = 0;
        for(int i=0;i<selected.length;i++){
            if(selected[i]){
                numSelected ++;
                if(! parameters.getParameterAt(i).hasDefaultOrIsSet()){
                    System.out.println("parameter "+parameters.getParameterAt(i).getName()+" not set!!");
                    return false;
                }
            }
        }
        if(numSelected < 1 && isRequired()){
            return false;
        }else{
            return true;
        }
    }

    public boolean isSet() {
        return userSelected;
    }

    public boolean isAtomic() {
        for(int i=0;i<parameters.getNumberOfParameters();i++){
            if(parameters.getParameterAt(i).getValue() instanceof ParameterSet){
                return false;
            }
        }
        return true;
    }

    public void simplify() {
        for(int i=0;i<parameters.getNumberOfParameters();i++){
            if(!selected[i] && parameters.getParameterAt(i).getValue() instanceof ParameterSet){
                parameters.getParameterAt(i).reset();
            }else{
                parameters.getParameterAt(i).simplify();
            }
        }
    }

    public void reset() {
        selected = defaultSelected.clone();
        simplify();
        
    }

    public void setDefault(Object defaultValue) throws IllegalValueException {
        setValue(defaultValue);
        defaultSelected = selected.clone();
        userSelected = false;
    }

    public StringBuffer toXML() {
        StringBuffer sup = super.toXML();
        XMLParser.addTags(sup,"superParameter");
        XMLParser.appendBooleanArrayWithTags(sup,selected,"selected");
        XMLParser.appendBooleanArrayWithTags(sup,defaultSelected,"defaultSelected");
        XMLParser.appendIntWithTags(sup,current,"current");
        XMLParser.addTags(sup,"multiSelectionCollectionParameter");
        return sup;
    }

    protected void fromXML(StringBuffer representation) throws NonParsableException {
        representation = XMLParser.extractForTag(representation,"multiSelectionCollectionParameter");
        super.fromXML(XMLParser.extractForTag(representation,"superParameter"));
        selected = XMLParser.extractBooleanArrayForTag(representation,"selected");
        defaultSelected = XMLParser.extractBooleanArrayForTag(representation,"defaultSelected");
        current = XMLParser.extractIntForTag(representation,"current");
    }

	public boolean next() throws ParameterException {
		int next = -1;
		for(int i=current + 1;i<selected.length;i++){
			if(selected[i]){
				next = i;
                break;
			}
		}
		if(next != -1){
			current = next;
			return true;
		}else{
			return false;
		}
	}

	public void resetToFirst() {
		for(int i=0;i<selected.length;i++){
			if(selected[i]){
				current = i;
                break;
			}
		}
	}
    

	public int getNumberOfValues() {
		int count = 0;
		for(int i=0;i<selected.length;i++){
			if(selected[i]){
				count ++;
			}
		}
		return count;
	}
    
	/**
	 * Returns the number of calls to {@link MultiSelectionCollectionParameter#next()} that can
	 * be called before <code>false</code> is returned.
	 * @param afterIdx the index after which shall be counted
	 * @return the number of calls to {@link MultiSelectionCollectionParameter#next()}
	 */
    public int getNumberOfNexts(int afterIdx) {
        int count = 0;
        for(int i=afterIdx + 1;i<selected.length;i++){
            if(selected[i]){
                count ++;
            }
        }
        return count;
    }

	public String valuesToString() {
		StringBuffer tmp = new StringBuffer();
		for(int i=0;i<selected.length;i++){
			if(selected[i]){
				tmp.append(parameters.getParameterAt(i).getName()+", ");
			}
		}
		tmp.delete(tmp.length() - 2,tmp.length());
		return "[" + tmp.toString() + "]";
	}

    public boolean isRanged() {
        return getNumberOfValues() > 1;
    }

}
