/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.results;

import de.jstacs.DataType;
import de.jstacs.NonParsableException;
import de.jstacs.io.XMLParser;
import de.jstacs.parameters.SimpleParameter.IllegalValueException;

/**
 * A class for categorical results (i.e. non-numerical results) for primitives and <code>Strings</code>
 * 
 * @author Jan Grau
 * 
 */
public class CategoricalResult extends SimpleResult
{

	/**
	 * Creates a new <code>CategoricalResult</code> from its XML-representation.
	 * 
	 * @param representation
	 *            the XML-representation
	 * @throws NonParsableException
	 *             a <code>NonParsableException</code> is thrown if <code>representation</code> could not be parsed
	 */
	public CategoricalResult( StringBuffer representation ) throws NonParsableException
	{
		super( representation );
	}

	/**
	 * Creates a result of a primitive categorical datatype or a <code>String</code>
	 * 
	 * @param datatype
	 *            the datatype
	 * @param name
	 *            the name of the result
	 * @param comment
	 *            a comment on the result
	 * @param result
	 *            the result itself
	 * @throws IllegalValueException
	 *             if the result value is not of the expected datatype an <code>IllegalValueException</code> is thrown
	 */
	public CategoricalResult( DataType datatype, String name, String comment, Comparable result )
			throws IllegalValueException
	{
		super( name, comment, datatype );
		if( !(datatype == DataType.STRING || datatype == DataType.BOOLEAN) )
		{
			throw new IllegalValueException( "Datatype must be categorical!" );
		}
		setResult( result );
	}

	/**
	 * Creates a result of a <code>String</code>.
	 * 
	 * @param name
	 *            the name of the result
	 * @param comment
	 *            a comment on the result
	 * @param result
	 *            the result itself
	 */
	public CategoricalResult( String name, String comment, String result )
	{
		super( name, comment, DataType.STRING );
		this.result = result;
	}
	
	/**
	 * Creates a result of a <code>boolean</code>.
	 * 
	 * @param name
	 *            the name of the result
	 * @param comment
	 *            a comment on the result
	 * @param result
	 *            the result itself
	 */
	public CategoricalResult( String name, String comment, boolean result )
	{
		super( name, comment, DataType.BOOLEAN );
		this.result = result;
	}

	public StringBuffer toXML()
	{
		StringBuffer buf = super.toXML();
		XMLParser.addTags( buf, "categoricalResult" );
		return buf;
	}

	protected void fromXML( StringBuffer representation ) throws NonParsableException
	{
		super.fromXML( XMLParser.extractForTag( representation, "categoricalResult" ) );
	}
}
