/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.scoringFunctions;

import de.jstacs.NonParsableException;
import de.jstacs.Storable;
import de.jstacs.data.AlphabetContainer;
import de.jstacs.data.Sequence;
import de.jstacs.utils.DoubleList;
import de.jstacs.utils.IntList;

/**
 * This is the main class for all ScoringFunctions that allow to score subsequences of arbitrary length. This
 * ScoringFunction should be the super class for non-motif ScoringFunctions like homogeneous Markov models, cyclic
 * Markov models, ...
 * 
 * @author Jens Keilwagen
 */
public abstract class VariableLengthScoringFunction extends AbstractNormalizableScoringFunction
{

	/**
	 * This is the main constructor that creates an instance that models sequence s of arbitrary length.
	 * 
	 * @param alphabets the AlphabetContainer
	 */
	protected VariableLengthScoringFunction( AlphabetContainer alphabets )
	{
		this( alphabets, 0 );
	}

	/**
	 * This is the main constructor that creates an instance that models sequence s of a given length.
	 * 
	 * @param alphabets the AlphabetContainer
	 * @param length the length of the modeled sequences
	 */
	protected VariableLengthScoringFunction( AlphabetContainer alphabets, int length )
	{
		super( alphabets, length );
		if( !alphabets.isSimple() || !alphabets.isDiscrete() )
		{
			throw new IllegalArgumentException( "The AlphabetContainer has to be simple and discrete." );
		}
	}

	/**
	 * This is the constructor for {@link Storable}.
	 * 
	 * @param source the xml representation
	 * 
	 * @throws NonParsableException if the representation could not be parsed.
	 */
	protected VariableLengthScoringFunction( StringBuffer source ) throws NonParsableException
	{
		super( source );
	}

	public double getNormalizationConstant()
	{
		return getNormalizationConstant( length );
	}

	/**
	 * This method returns the normalization constant for a given sequence length.
	 * 
	 * @param length
	 *            the sequence length
	 * 
	 * @return the normalization constant
	 * 
	 * @see NormalizableScoringFunction#getNormalizationConstant()
	 */
	public abstract double getNormalizationConstant( int length );

	public double getPartialNormalizationConstant( int parameterIndex ) throws Exception
	{
		return getPartialNormalizationConstant( parameterIndex, length );
	}

	/**
	 * This method returns the partial normalization constant for a given parameter index and sequence length.
	 * 
	 * @param parameterIndex
	 *            the index of the parameter
	 * @param length
	 *            the sequence length
	 * 
	 * @return the partial normalization constant
	 * 
	 * @throws Exception
	 *             if something went wrong
	 * 
	 * @see NormalizableScoringFunction#getPartialNormalizationConstant(int)
	 */
	public abstract double getPartialNormalizationConstant( int parameterIndex, int length ) throws Exception;

	public double getLogScore( Sequence seq, int start )
	{
		if( length != 0 )
		{
			return getLogScore( seq, start, length );
		}
		else
		{
			return getLogScore( seq, start, seq.getLength() - start );
		}
	}

	/**
	 * This method computes the logarithm of the score for a given subsequence.
	 * 
	 * @param seq
	 *            the sequence
	 * @param start
	 *            the start index
	 * @param length
	 *            the end index
	 * 
	 * @return the logarithm of the score
	 * 
	 * @see de.jstacs.scoringFunctions.ScoringFunction#getLogScore(Sequence, int)
	 */
	public abstract double getLogScore( Sequence seq, int start, int length );

	public double getLogScoreAndPartialDerivation( Sequence seq, int start, IntList indices, DoubleList dList )
	{
		if( length != 0 )
		{
			return getLogScoreAndPartialDerivation( seq, start, length, indices, dList );
		}
		else
		{
			return getLogScoreAndPartialDerivation( seq, start, seq.getLength() - start, indices, dList );
		}
	}

	/**
	 * This method computes the logarithm of the score and the partial derivations for a given subsequence.
	 * 
	 * @param seq
	 *            the sequence
	 * @param start
	 *            the start index
	 * @param length
	 *            the end index
	 * @param indices
	 *            the list for the indices of the parameters
	 * @param dList
	 *            the list for the partial derivations
	 * 
	 * @return the logarithm of the score
	 * 
	 * @see de.jstacs.scoringFunctions.ScoringFunction#getLogScoreAndPartialDerivation(Sequence, int, IntList,
	 *      DoubleList)
	 */
	public abstract double getLogScoreAndPartialDerivation( Sequence seq, int start, int length, IntList indices,
			DoubleList dList );

	/**
	 * This method returns the stationary symbol distribution. For DNA this is the stationary mono nucleotide
	 * distribution. This method is used to determine the contrast for
	 * {@link de.jstacs.motifDiscovery.Mutable#expand(int, double[], double[], double)},
	 * {@link de.jstacs.motifDiscovery.Mutable#shift(double[], double[], double)} and
	 * {@link de.jstacs.motifDiscovery.Mutable#shrink(double[], double[], double)}.
	 * 
	 * @return the stationary symbol distribution
	 * 
	 * @see Mutable
	 */
	public abstract double[] getStationarySymbolDistribution();

	/**
	 * This method sets the hyperparameters for the model parameters by evaluating the given statistic. The statistic can
	 * be interpreted as follows: The model has seen a number of sequences. From these sequences it is only known how
	 * long (<code>length</code>) and how often (<code>weight</code>) they have been seen.  
	 * 
	 * @param length the non-negative lengths of the sequences
	 * @param weight the non-negative weight for the corresponding sequence
	 * 
	 * @throws Exception if something went wrong
	 * 
	 * @see Mutable
	 */
	public abstract void setStatisticForHyperparameters( int[] length, double[] weight ) throws Exception;
}