/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.scoringFunctions.directedGraphicalModels.structureLearning.measures;

import de.jstacs.NonParsableException;
import de.jstacs.data.Sample;
import de.jstacs.io.XMLParser;
import de.jstacs.scoringFunctions.directedGraphicalModels.BayesianNetworkScoringFunction;

/**
 * Class for a network structure of a {@link BayesianNetworkScoringFunction} that is an inhomogeneous Markov model. The order of the Markov model can be defined by the user. A Markov model
 * of order <code>0</code> is also know as position weight matrix (PWM), a Markov model of order <code>1</code> is also known as weight array matrix (WAM) model.
 * 
 * @author Jan Grau
 *
 */
public class InhomogeneousMarkov extends Measure {

	private int order;
	
	/**
	 * Creates the structure of an inhomogeneous Markov model of order <code>order</code>.
	 * @param order the order
	 */
	public InhomogeneousMarkov(int order) {
		this.order = order;
	}
	
	/**
	 * Re-creates an {@link InhomogeneousMarkov} structure from its XML-representation as returned by {@link InhomogeneousMarkov#toXML()}.
	 * @param buf the XML-representation
	 * @throws NonParsableException is thrown if the XML-code could not be parsed
	 */
	public InhomogeneousMarkov(StringBuffer buf) throws NonParsableException{
		buf = XMLParser.extractForTag(buf, "inhomogeneousMarkov");
		this.order = XMLParser.extractIntForTag(buf, "order");
	}
	
	/**
	 * Returns the order of the Markov model as defined in the constructor
	 * @return the order
	 */
	public int getOrder(){
		return order;
	}
	
	
	/* (non-Javadoc)
	 * @see de.jstacs.scoringFunctions.directedGraphicalModels.structureLearning.measures.Measure#clone()
	 */
	public InhomogeneousMarkov clone() throws CloneNotSupportedException{
		return (InhomogeneousMarkov) super.clone();
	}

	/* (non-Javadoc)
	 * @see de.jstacs.scoringFunctions.directedGraphicalModels.structureLearning.measures.Measure#getInstanceName()
	 */
	@Override
	public String getInstanceName() {
		return "Inhomogeneous Markov model of order "+order;
	}

	/* (non-Javadoc)
	 * @see de.jstacs.scoringFunctions.directedGraphicalModels.structureLearning.measures.Measure#getParents(de.jstacs.data.Sample, de.jstacs.data.Sample, double[], double[], int)
	 */
	@Override
	public int[][] getParents(Sample fg, Sample bg, double[] weightsFg, double[] weightsBg, int length) throws Exception {
		int[][] parents = new int[length][];
		for(int i=0;i<parents.length;i++){
			parents[i] = new int[(order < i ? order : i) + 1];
			for(int j=i;j>=i-order && j>=0;j--){
				parents[i][parents[i].length - (i-j) -1]=j;
			}
		}
		return parents;
	}

	
	
	/* (non-Javadoc)
	 * @see de.jstacs.scoringFunctions.directedGraphicalModels.structureLearning.measures.Measure#isShiftable()
	 */
	@Override
	public boolean isShiftable() {
		return true;
	}

	/* (non-Javadoc)
	 * @see de.jstacs.Storable#toXML()
	 */
	public StringBuffer toXML() {
		StringBuffer buf = new StringBuffer();
		XMLParser.appendIntWithTags(buf, order, "order");
		XMLParser.addTags(buf, "inhomogeneousMarkov");
		return buf;
	}

}
