/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.utils;

/**
 * A simple list of primitive type double.
 * 
 * @author Jens Keilwagen
 */
public final class DoubleList
{
	private int size;

	private double[] array;

	/**
	 * This is the default constructor that creates a DoubleList with initial length 10.
	 */
	public DoubleList()
	{
		this( 10 );
	}

	/**
	 * This is the default constructor that creates a DoubleList with initial length <code>size</code>.
	 * 
	 * @param size the initial size; has to be positive
	 * 
	 * @throws IllegalArgumentException if the size is less than 1 
	 */
	public DoubleList( int size ) throws IllegalArgumentException
	{
		super();
		if( size <= 0 )
		{
			throw new IllegalArgumentException( "The size has to be positive." );
		}
		this.size = 0;
		array = new double[size];
	}

	/**
	 * Adds the element <code>val</code> at the end of the list.
	 * 
	 * @param val
	 *            the element that should be inserted.
	 */
	public final void add( double val )
	{
		if( array.length == size )
		{
			double[] help = new double[2 * array.length];
			System.arraycopy( array, 0, help, 0, size );
			array = help;
		}
		array[size++] = val;
	}

	/**
	 * Adds the element <code>val</code> from fromIndex</code> to <code>toIndex</code> (exclusive).
	 * 
	 * @param val
	 *            the element that should be inserted
	 * @param fromIndex
	 *            the start index (inclusive)
	 * @param toIndex
	 *            the end index (exclusive)
	 */
	public final void add( double val, int fromIndex, int toIndex )
	{
		if( toIndex > array.length )
		{
			double[] help = new double[toIndex];
			System.arraycopy( array, 0, help, 0, size );
			array = help;
		}
		while( fromIndex < toIndex )
		{
			array[fromIndex++] = val;
		}
		if( size < toIndex )
		{
			size = toIndex;
		}
	}

	/**
	 * Removes all elements from the list.
	 */
	public final void clear()
	{
		size = 0;
	}

	/**
	 * Returns the element with the specified <code>index</code>.
	 * 
	 * @param index
	 *            the specified index
	 * 
	 * @return the corresponding element
	 */
	public final double get( int index )
	{
		if( index < size )
		{
			return array[index];
		}
		else
		{
			throw new ArrayIndexOutOfBoundsException();
		}
	}
	
	/**
	 * Returns the number of inserted elements.
	 *  
	 * @return the number of inserted elements
	 */
	public final int length()
	{
		return size;
	}

	/**
	 * This method returns a double array containing all elements of the list.
	 *  
	 * @return a double array containing all elements of the list.
	 */
	public double[] toArray()
	{
		double[] erg = new double[size];
		System.arraycopy( array, 0, erg, 0, size );
		return erg;
	}

	/**
	 * Multiplies all value in the list from index <code>start</code> to <code>end</code> with the value <code>factor</code>
	 * 
	 * @param start the start index (inclusive)
	 * @param end the end index (exclusive)
	 * @param factor the factor
	 */
	public void multiply( int start, int end, double factor )
	{
		while( start < end )
		{
			array[start++] *= factor;		
		}
	}
}
