/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.utils;

import java.io.IOException;
import java.io.OutputStream;

/**
 * This class is for any output. For example:
 * <ul>
 * <li> It enables you to hide the output (use <code>new SafeOutputStream( null )</code>).
 * <li> It enables you to write the output in the standard output stream (use
 * <code>new SafeOutputStream( System.out )</code>).
 * <li> It enables you to write the output to a file (use
 * <code>new SafeOutputStream( new FileOutputStream( ... ) )</code>).
 * <li> ...
 * </ul>
 * 
 * The user has to close the stream by hand. It is not closed by the <code>finalize</code> method.
 * 
 * @author Jens Keilwagen
 */
public class SafeOutputStream extends OutputStream
{
	/**
	 * This stream can be used to as default stream.
	 */
	public static final OutputStream DEFAULT_STREAM = System.out;
	
	private OutputStream ostream;

	/**
	 * Creates a new SafeOutputstream
	 * 
	 * @param ostream
	 *            the internal used OutputStream
	 */
	public SafeOutputStream( OutputStream ostream )
	{
		this.ostream = ostream;
	}

	/**
	 * Closes this <code>SafeOutputStream</code> by closing the <code>OutputStream</code> 
	 * this stream was constructed of.
	 * If this SaveOutputStream was constructed using <code>System.out</code> this 
	 * <code>OutputStream</code> will be closed. No outputs via <code>System.out</code>
	 * will be possible any more.
	 */
	public void close() throws IOException
	{
		if( ostream != null )
		{
			ostream.flush();
			ostream.close();
		}
	}

	/**
	 * Indicates whether the instance is doing something or not.
	 * 
	 * @return <code>true<code> if the internal OutputStream is <code>null</code>
	 */
	public boolean doesNothing()
	{
		return ostream == null;
	}

	public void flush() throws IOException
	{
		if( ostream != null )
		{
			ostream.flush();
		}
	}

	/**
	 * Returns the internal used OutputStream.
	 * 
	 * @return the internal used OutputStream
	 */
	public OutputStream getOutputStream()
	{
		return ostream;
	}

	/**
	 * Writes a linebreak.
	 * 
	 * @throws IOException
	 *             if something is wrong.
	 * @see SafeOutputStream#write(int)
	 */
	public void writeln() throws IOException
	{
		writeln( "" );
	}

	/**
	 * Writes the String and a linebreak.
	 * 
	 * @param s
	 *            the String
	 * 
	 * @throws IOException
	 *             if something is wrong.
	 * @see SafeOutputStream#write(int)
	 */
	public void writeln( String s ) throws IOException
	{
		write( s + "\n" );
	}

	/**
	 * Writes the String.
	 * 
	 * @param s
	 *            the String
	 * 
	 * @throws IOException
	 *             if something is wrong.
	 * @see SafeOutputStream#write(int)
	 */
	public void write( String s ) throws IOException
	{
		if( ostream != null )
		{
			ostream.write( s.getBytes() );
		}
	}

	public void write( int b ) throws IOException
	{
		if( ostream != null )
		{
			ostream.write( b );
		}
	}

	/**
	 * This method does not close the internal stream.
	 */
	protected void finalize() throws IOException
	{
		flush();
	}
}
