/*
 * This file is part of Jstacs.
 * 
 * Jstacs is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * Jstacs is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * Jstacs. If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.data.sequences;

import java.util.Random;

import de.jstacs.WrongAlphabetException;
import de.jstacs.data.Sequence;
import de.jstacs.data.sequences.annotation.SequenceAnnotation;

/**
 * Class for a permuted sequence.
 * 
 * @author Jan Grau
 */
public class PermutedSequence extends RecursiveSequence {

	private static Random r = new Random();

	private int[] permutation;

	/**
	 * This constructor creates an instance by shuffling the symbols.
	 * 
	 * @param seq
	 *            the initial sequence
	 * 
	 * @throws WrongAlphabetException
	 *             if the alphabet container is not simple
	 */
	public PermutedSequence( Sequence seq ) throws WrongAlphabetException {
		super( seq.getAlphabetContainer(), seq );
		if( !seq.getAlphabetContainer().isSimple() ) {
			throw new WrongAlphabetException( "Alphabet must be simple" );
		}
		permutation = new int[seq.getLength()];
		for( int i = 0; i < permutation.length; i++ ) {
			permutation[i] = i;
		}
		int temp;
		int temp2;
		for( int i = permutation.length - 1; i >= 0; i-- ) {
			temp = r.nextInt( i + 1 );
			temp2 = permutation[i];
			permutation[i] = permutation[temp];
			permutation[temp] = temp2;
		}
	}

	private PermutedSequence( Sequence seq, SequenceAnnotation[] annotation, int[] perm ) {
		super( seq.getAlphabetContainer(), annotation, seq );
		this.permutation = perm;
	}

	/* (non-Javadoc)
	 * @see de.jstacs.data.sequences.RecursiveSequence#getIndex(int)
	 */
	@Override
	protected int getIndex( int pos ) {
		return permutation[pos];
	}

	/* (non-Javadoc)
	 * @see de.jstacs.data.Sequence#getLength()
	 */
	@Override
	public int getLength() {
		return permutation.length;
	}

	/* (non-Javadoc)
	 * @see de.jstacs.data.Sequence#flatCloneWithoutAnnotation()
	 */
	@Override
	protected Sequence flatCloneWithoutAnnotation() {
		try {
			return new PermutedSequence( this.content, null, permutation );
		} catch ( Exception doesnothappen ) {
			throw new RuntimeException( doesnothappen );
		}
	}

}
