/*
 * This file is part of Jstacs.
 * 
 * Jstacs is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * Jstacs is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * Jstacs. If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.models.mixture.gibbssampling;

import de.jstacs.NonParsableException;
import de.jstacs.io.XMLParser;
import de.jstacs.utils.DoubleList;

/**
 * In this class the Variance-Ratio method of Gelman and Rubin is implemented to
 * test the length of the burn-in phase of a multi-chain Gibbs Sampling (number
 * of chains >2). The number of initial iterations is calculated by comparing
 * the variances between the different chains with the variances within the
 * chains. The method returns the same length of the burn-in for all sampled
 * chains.
 * 
 * @author Berit Haldemann
 */
public class VarianceRatioBurnInTest extends AbstractBurnInTest {

	/**
	 * The threshold for testing the end of the burn-in phase with the
	 * Variance-Ratio burn-in test.
	 */
	private double threshold;

	/**
	 * Constructor for initialization of the {@link VarianceRatioBurnInTest}
	 * given a fixed threshold value of 1.2.
	 * 
	 * @param starts
	 *            the number of chains resulting from the number of starts of
	 *            the Gibbs Sampler, must be greater than 2
	 */
	public VarianceRatioBurnInTest( int starts ) {
		this( starts, 1.2 );
	}

	/**
	 * Constructor for initialization of the {@link VarianceRatioBurnInTest}.
	 * 
	 * @param starts
	 *            the number of chains resulting from the number of starts of
	 *            the Gibbs Sampler, must be greater than 2
	 * @param t
	 *            the threshold value for testing the end of the burn-in phase
	 *            with the Variance-Ratio burn-in test, the value has to be
	 *            greater than 1 since the tested potential scale reduction
	 *            factor R converges to 1
	 * 
	 * @see VarianceRatioBurnInTest#computeLengthOfBurnIn()
	 */
	public VarianceRatioBurnInTest( int starts, double t ) {
		super( starts );
		if( starts < 3 ) {
			throw new IllegalArgumentException( "The number of chains (starts of the Gibbs Sampler) must be greater than 2." );
		}
		if( t <= 1 ) {
			throw new IllegalArgumentException( "The given threshold value for t has to be greater than 1." );
		}
		this.threshold = t;

	}

	/**
	 * The standard constructor for the interface {@link de.jstacs.Storable}.
	 * 
	 * @param rep
	 *            the {@link StringBuffer} containing the model
	 * 
	 * @throws NonParsableException
	 *             if the {@link StringBuffer} can not be parsed
	 */
	public VarianceRatioBurnInTest( StringBuffer rep ) throws NonParsableException {
		super( rep );
	}

	/* (non-Javadoc)
	 * @see de.jstacs.models.mixture.gibbssampling.AbstractBurnInTest#getXMLTag()
	 */
	protected String getXMLTag() {
		return getClass().getSimpleName();
	}

	/* (non-Javadoc)
	 * @see de.jstacs.models.mixture.gibbssampling.AbstractBurnInTest#getFurtherInformation()
	 */
	protected StringBuffer getFurtherInformation() {
		StringBuffer furtherinf = new StringBuffer( 2000 );
		XMLParser.appendDoubleWithTags( furtherinf, threshold, "threshold" );
		return furtherinf;
	}

	/* (non-Javadoc)
	 * @see de.jstacs.models.mixture.gibbssampling.AbstractBurnInTest#setFurtherInformation(java.lang.StringBuffer)
	 */
	protected void setFurtherInformation( StringBuffer xml ) throws NonParsableException {

		threshold = XMLParser.extractDoubleForTag( xml, "threshold" );

	}

	/* (non-Javadoc)
	 * @see de.jstacs.models.mixture.gibbssampling.BurnInTest#getInstanceName()
	 */
	public String getInstanceName() {
		return "Variance-Ratio burn-in test of Gelman and Rubin for " + values.length + " different chains with threshold " + threshold;
	}

	/**
	 * Computes and returns the length of the burn-in phase given by the
	 * Variance-Ratio burn-in test. To get an effective test the number of Gibbs
	 * iterations should be greater than 250. Fewer iterations may lead to
	 * inaccuracy in the calculation of the different required variances.
	 * 
	 * @see de.jstacs.models.mixture.gibbssampling.AbstractBurnInTest#computeLengthOfBurnIn()
	 */
	protected int computeLengthOfBurnIn() {

		if( values[0].length() < 250 ) {

			return values[0].length();

		} else {
			int m = values.length;
			DoubleList meanOfchains = new DoubleList( m );
			for( int it = 250; it < values[0].length(); it = it + 2 ) {

				// defining the window for the calculations
				int n = it / 2;
				int start = it - n;

				// calculation of the means of the different chains
				meanOfchains.clear();
				for( int i = 0; i < m; i++ ) {
					meanOfchains.add( values[i].mean( start, it ) );
				}

				// calculation of the means of the calculated means
				double meanOfChainMeans = meanOfchains.mean( 0, meanOfchains.length() );

				// calculation of the measure B
				double b = 0;
				for( int i = 0; i < m; i++ ) {
					b += ( meanOfchains.get( i ) - meanOfChainMeans ) * ( meanOfchains.get( i ) - meanOfChainMeans );
				}
				b = b / ( (double)m - 1d );

				// calculation of the measure W
				double w = 0;
				for( int i = 0; i < m; i++ ) {
					for( int j = start; j < it; j++ ) {
						w += ( values[i].get( j ) - meanOfchains.get( i ) ) * ( values[i].get( j ) - meanOfchains.get( i ) );
					}
				}
				w = w / ( (double)m * ( (double)n - 1d ) );

				// calculation of the estimator of sigma
				double sigma = ( 1d - ( 1d / (double)n ) ) * w;
				sigma += ( ( 1d + ( 1d / (double)m ) ) * b );

				// calculation of the potential scale reduction factor R
				double r = Math.sqrt( sigma / w );

				// test if the burn-in is ended
				if( r < threshold ) {
					return it;
				}
			}
			return values[0].length();
		}
	}
}
