package de.jstacs.motifDiscovery;

import de.jstacs.data.Sequence;
import de.jstacs.motifDiscovery.MotifDiscoverer.KindOfProfile;
import de.jstacs.results.ImageResult;
import de.jstacs.utils.REnvironment;

/**
 * This class contains static methods for the MotifDiscoverer.
 * 
 * @author Jens Keilwagen
 * 
 * @see de.jstacs.motifDiscovery.MotifDiscoverer
 */
public class MotifDiscovererToolBox
{

	private final static String PLOT = "plot( 0:(length(profile)-1),profile, \"l\", ylab=\"score\", xlab=\"position\", ylim=lim );\n";
	
	private final static String ANNOTATE = "abline( h = threshold, lty=2, col=2 );\n" +
			"for( i in 1:length(profile)){\n" +
			"if( profile[i] >= threshold ){ h = paste( i-1, \": \", substr(seq,i,i+w-1), sep=\"\" ); text( i-1, profile[i], h, pos=4, col=4 ); }\n" +
			"}"; 

	/**
	 * This method creates a simple plot of the profile of scores for a sequence and a start position.
	 * 
	 * @param motifDisc the MotifDiscoverer
	 * @param component the index of the component
	 * @param motif the index of the motif
	 * @param sequence the sequence
	 * @param startpos the start position in the sequence
	 * @param r the R environment that is used for drawing the plot
	 * @param width the width of the image in pixel
	 * @param height the height of the image in pixel
	 * @param kind the kind of the score profile
	 * 
	 * @return an image packed in a result
	 * 
	 * @throws Exception if some thing went wrong
	 */
	public static ImageResult plot( MotifDiscoverer motifDisc, int component, int motif, Sequence sequence,
			int startpos, REnvironment r, int width, int height, KindOfProfile kind) throws Exception
	{
		r.createVector( "profile", motifDisc.getProfileOfScoresFor( component, motif, sequence, startpos, kind ) );
		r.voidEval( "lim=c(min(profile),max(profile))" );
		return new ImageResult( "profile", "the profile of scores for " + sequence.toString( startpos ), r.plot( PLOT,
				width, height ) );
	}

	/**
	 * This method creates a plot of the profile of scores for a sequence and a start position and annotates bindings sites in the plot.
	 * 
	 * @param motifDisc the MotifDiscoverer
	 * @param component the index of the component
	 * @param motif the index of the motif
	 * @param sequence the sequence
	 * @param startpos the start position in the sequence
	 * @param r the R environment that is used for drawing the plot
	 * @param width the width of the image in pixel
	 * @param height the height of the image in pixel
	 * @param yMin the minimal value of y-axis
	 * @param yMax the maximal value of the y-axis
	 * @param threshold the threshold
	 * @param kind the kind of the score profile
	 * 
	 * @return an image packed in a result
	 * 
	 * @throws Exception if some thing went wrong
	 */
	public static ImageResult plotAndAnnotate( MotifDiscoverer motifDisc, int component, int motif, Sequence sequence,
			int startpos, REnvironment r, int width, int height, double yMin, double yMax, double threshold, KindOfProfile kind )
			throws Exception
	{
		r.createVector( "profile", motifDisc.getProfileOfScoresFor( component, motif, sequence, startpos , kind) );
		r.voidEval( "lim=c(" + yMin + "," + yMax + ")" );
		r.voidEval( "threshold = " + threshold );
		String seq = sequence.toString( startpos );
		r.voidEval( "seq = \"" + seq + "\"" );
		r.voidEval( "w = " + motifDisc.getMotifLength( motifDisc.getGlobalIndexOfMotifInComponent( component, motif ) ) );
		return new ImageResult( "annotated profile with threshold " + threshold, "the annotated profile of scores for "	+ seq, 
				r.plot( PLOT + ANNOTATE, width, height ) );
	}
}
