/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.parameters;

import de.jstacs.NonParsableException;
import de.jstacs.io.XMLParser;

/**
 * A class for a <code>ParameterSet</code> that can be expanded by additional <code>Parameter</code>s at runtime.
 * The <code>Parameter</code> are all of type <code>ParameterSetContainer</code>. The contents of these <code>Parameter</code>s
 * are provided by a <code>ParameterSet</code>-template, which is cloned each time the <code>addParameterToSet()</code>-method
 * is called and set as value of a <code>ParameterSetContainer</code> that is added to the <code>Parameter</code>s
 * of the <code>ExpandableParameterSet</code>. Already added <code>Parameter</code>s can also be removed from the set using the <code>removeParameterFromSet</code>-method.
 * @author Jan Grau
 *
 */
public class ExpandableParameterSet extends ParameterSet {

	/**
	 * The template for each <code>ParameterSet</code>
	 */
	protected ParameterSet template;
	/**
	 * A template for the name of the enclosing <code>ParameterSetContainer</code>
	 */
	protected String nameTemplate;
	/**
	 * A template for the comment of the enclosing <code>ParameterSetContainer</code>
	 */
	protected String commentTemplate;
	
	private int count;
	
	private int initCount;

	/**
	 * Creates a new <code>ExpandableParameterSet</code> from a <code>Class</code> that can be instantiated using this
	 * <code>ExpandableParameterSet</code> and templates for the <code>ParameterSet</code> in each element of the array,
     * the name and the comment that are displayed for the <code>ParameterSetContainer</code>s enclosing the <code>ParameterSet</code>s.
     * Initially, the <code>loadParameters</code>-method creates one copy of the template.
     * 
	 * @param template the template of the <code>ParameterSet</code>
     * @param nameTemplate the name-template
     * @param commentTemplate the comment-template
	 */
	public ExpandableParameterSet( ParameterSet template, String nameTemplate, String commentTemplate){
		super();
		this.template = template;
		this.nameTemplate = nameTemplate;
		this.commentTemplate = commentTemplate;
		this.count = 0;
		this.initCount = 1;
	}
	
	/**
	 * Creates a new <code>ExpandableParameterSet</code> from a <code>Class</code> that can be instantiated using this
	 * <code>ExpandableParameterSet</code> and templates for the <code>ParameterSet</code> in each element of the array,
     * the name and the comment that are displayed for the <code>ParameterSetContainer</code>s enclosing the <code>ParameterSet</code>s.
     * Using this constructor, the <code>loadParameters()</code>-method initially creates <code>initCount</code> copies of the template.
     * 
	 * @param template the template of the <code>ParameterSet</code>
     * @param nameTemplate the name-template
     * @param commentTemplate the comment-template
     * @param initCount the number of initial copies of the template
	 */
	public ExpandableParameterSet( ParameterSet template, String nameTemplate, String commentTemplate, int initCount){
		super();
		this.template = template;
		this.nameTemplate = nameTemplate;
		this.commentTemplate = commentTemplate;
		this.count = 0;
		this.initCount = initCount;
	}
	
	/**
	 * Creates a new <code>ExpandableParameterSet</code> from its XML-representation.
	 * @param representation the XML-represenation
	 * @throws NonParsableException is thrown if <code>representation</code> could not be parsed.
	 */
	public ExpandableParameterSet(StringBuffer representation) throws NonParsableException{
		super(representation);
	}
	
	
    /**
     * Creates a new <code>ExpandableParameterSet</code> from a <code>ParameterSet</code>-array.
     * The first element of this array is taken as a template for further <code>addParameterToSet()</code>-calls.
     * @param templateAndContent the content (and template)
     * @param nameTemplate the name-template
     * @param commentTemplate the comment-template
     */
    public ExpandableParameterSet(ParameterSet[] templateAndContent, String nameTemplate, String commentTemplate){
        if(templateAndContent.length == 0){
            throw new IllegalArgumentException("You must provide at least one ParameterSet.");
        }
        this.template = templateAndContent[0];
        this.nameTemplate = nameTemplate;
        this.commentTemplate = commentTemplate;
        
        if(notAllGivenParameterSetsAreOfTemplateType(templateAndContent)){
        	throw new IllegalArgumentException("At least one of the given ParameterSets is not of " +
        			"the specified template-Type");
        }
        
        initParameterList(templateAndContent.length);
        for(int i=0;i<templateAndContent.length;i++){
            parameters.add(new ParameterSetContainer(nameTemplate + " no. "+(i+1),
                    commentTemplate,templateAndContent[i]));
        }
        this.count=templateAndContent.length;
        this.initCount = this.count;
    }
    
	public ExpandableParameterSet clone() throws CloneNotSupportedException{
		ExpandableParameterSet clone = (ExpandableParameterSet) super.clone();
		clone.template = template.clone();
		return clone;
	}
	
	@Override
	protected void loadParameters() throws Exception {
		if(parameters == null){
			initParameterList();
			for(int i=0;i<initCount;i++){
				addParameterToSet();
			}
		}
	}

    @Override
    protected void replaceParametersWithRangedInstance() throws Exception{
        this.template.makeRanged();
        super.replaceParametersWithRangedInstance();
    }
    
	/**
	 * Adds a new <code>ParameterSetContainer</code> containing a clone of the <code>ParameterSet</code>-template
	 * to the set of <code>Parameter</code>s
	 * @throws CloneNotSupportedException is thrown if the template could not be cloned
	 */
	public void addParameterToSet() throws CloneNotSupportedException{
		if( parameters == null )
		{
			try
			{
				loadParameters();
                if(ranged){
                    replaceParametersWithRangedInstance();
                }
			} catch( Exception e )
			{
				e.printStackTrace();
			}
		}
		ParameterSetContainer simplePar = new ParameterSetContainer(
                nameTemplate + " no. "+(++count),
                commentTemplate,
				template.clone());
		parameters.add(simplePar);
		
	}
	
	/**
	 * First removes all previous added <code>ParameterSetContainer</code> and afterwards adds all given 
	 * <code>ParameterSet</code>s (in the given order) enclosed in new <code>ParameterSetContainer</code>s.
	 * 
	 * @param paramSetArray the <code>ParameterSet</code>s to be set
	 * @return true, if all given ParameterSets are of the same class as the defined template-ParameterSet of
	 * this <code>ExpandableParameterSet</code>; false otherwise 
	 * (In this case, neither the previous added <code>ParameterSetContainer</code> are removed nor is any
	 * given <code>ParameterSet</code> added. Hence the <code>ExpandableParameterSet</code> stays unchanged)
	 */
	public boolean replaceContentWith
	(ParameterSet[] paramSetArray){
		
		if(notAllGivenParameterSetsAreOfTemplateType(paramSetArray)){
			return false;
		}
		
		initParameterList(paramSetArray.length);
        for(int i=0;i<paramSetArray.length;i++){
            parameters.add(new ParameterSetContainer(nameTemplate + " no. "+(i+1),
                    commentTemplate,paramSetArray[i]));
        }
        this.count=paramSetArray.length;	
        
    return true;
	}
	
	
	private boolean notAllGivenParameterSetsAreOfTemplateType
	(ParameterSet[] temp){
		
		Class clazz = template.getClass();
		
		for(int i=0; i<temp.length;i++){
			if(temp[i].getClass() != clazz) return true;
		}
	
	return false;
	}
	
	/**
	 * Returns true if there is still a <code>Parameter</code> that can be removed from the set.
	 * @return is a <code>Parameter</code> can be removed
	 */
	public boolean parameterRemovable(){
		return count > 0;
	}
	
	/**
	 * Removes the last <code>Parameter</code> from set.
	 *
	 */
	public void removeParameterFromSet(){
		if( parameters == null )
		{
			try
			{
				loadParameters();
                if(ranged){
                    replaceParametersWithRangedInstance();
                }
			} catch( Exception e )
			{
				e.printStackTrace();
			}
		}
		
		if(count > 0){
			parameters.remove(parameters.size() - 1);
			count--;
		}
	}
	
	public StringBuffer toXML(){
		StringBuffer buf = super.toXML();
		XMLParser.addTags(buf,"superParameters");
		XMLParser.appendStorableWithTags(buf,template,"template");
		XMLParser.appendStringWithTags(buf,nameTemplate,"nameTemplate");
		XMLParser.appendStringWithTags(buf,commentTemplate,"commentTemplate");
		XMLParser.appendIntWithTags(buf, initCount, "initCount");
		XMLParser.addTags(buf,"expandableParameterSet");
		return buf;
	}
	
	protected void fromXML(StringBuffer representation) throws NonParsableException{
		representation = XMLParser.extractForTag(representation,"expandableParameterSet");
		super.fromXML(XMLParser.extractForTag(representation,"superParameters"));
		template = (ParameterSet) XMLParser.extractStorableForTag(representation,"template");
		nameTemplate = XMLParser.extractStringForTag(representation,"nameTemplate");
		commentTemplate = XMLParser.extractStringForTag(representation,"commentTemplate");
		initCount = XMLParser.extractIntForTag(representation, "initCount");
	}

}
