/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.parameters;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

import de.jstacs.NonParsableException;
import de.jstacs.Storable;
import de.jstacs.io.XMLParser;

/**
 * (Container) class for a set of {@link Parameter}s.
 * This is the base class for other {@link ParameterSet}s that provide more specialized methods, e.g. for 
 * expanding the set of {@link Parameter}s at runtime ( {@link ExpandableParameterSet} ) or defining an array
 * of {@link ParameterSet}s from a common template ( {@link ArrayParameterSet} ).
 * 
 * @author Jan Grau
 * 
 */
public abstract class ParameterSet implements Storable, Cloneable, RangeIterator
{
	/**
	 * The set of parameters
	 */
	protected ParameterList parameters;

	/**
	 * The error-message of the last error or <code>null</code>
	 */
	protected String errorMessage;
	
    /**
     * Indicates if the {@link Parameter}s of this {@link ParameterSet}
     * that implement {@link Rangeable} and return <code>true</code>
     * shall be replaced by their ranged instances
     */
    protected boolean ranged = false;
    
    /**
     * Is this {@link ParameterSet} is contained in a {@link ParameterSetContainer},
     * this variable holds a reference to that {@link ParameterSetContainer}.
     */
    protected ParameterSetContainer parent;
    
    private long id;

	/**
	 * Constructs a new {@link ParameterSet} with empty parameter values.
	 * The set of parameters is loaded by the {@link ParameterSet#loadParameters()}-method
	 * as soon as one of the accession methods, e.g. {@link ParameterSet#getNumberOfParameters()},
	 * is called.
	 * 
	 * @throws UnsupportedOperationException
	 *             thrown if the <code>ParameterSet</code> could not be constructed
	 */
	public ParameterSet() throws UnsupportedOperationException
	{
		this.id = System.currentTimeMillis() + this.hashCode();
	}

	/**
	 * Creates a full clone (deep copy) of this {@link ParameterSet}. As
	 * a convenience-method the user can use
	 * <code>fillWithStandardFieldsForClone(ParameterSet)</code> on a newly
	 * created instance of a subclass of <code>ParameterSet</code> to obtain a
	 * clone/copy of all standard member variables (those already defined in
	 * {@link ParameterSet}) in the passed {@link ParameterSet}.
	 * Using this method, the cloning-process becomes merely three-step:<br>
	 * <ul>
	 * <li> Create a new instance of your subclass of {@link ParameterSet}>,
	 * most likely with an empty constructor or the one taking just the
	 * instance-class
	 * <li> call <code>this.fillWithStandardFieldsForClone</code> on this
	 * instance
	 * <li> return the instance
	 * </ul>
	 * This method fulfills the conventions of {@link Object}'s method {@link Object#clone()}
	 * 
	 * @return a deep clone of this {@link ParameterSet}
	 */
	public ParameterSet clone() 
	throws CloneNotSupportedException{
		
		ParameterSet ret =  (ParameterSet)super.clone();
		
		if(this.parameters!=null){
			ret.initParameterList( this.parameters.size() );
			for(int i=0; i<this.parameters.size();i++){
				ret.parameters.add(i,this.parameters.get(i).clone());
			}
			Iterator<Parameter> it = ret.parameters.iterator();
			while(it.hasNext()){
				Parameter par = it.next();
				if(par.getNeededReferenceId() != null){
					ParameterSet set = findParameterSet( par.getNeededReferenceId(), false );
					if(set != null){
						par.setNeededReference( set );
					}
				}
			}
		}
		propagateId( this.id, this, false );
		return ret;
	}

	/**
	 * Initializes the internal set of {@link Parameter}s,
	 * which is a {@link ParameterList}.
	 */
	protected final void initParameterList(){
		this.parameters = new ParameterList();
	}
	
	/**
	 * Initializes the internal set of {@link Parameter}s,
	 * which is a {@link ParameterList}, with an initial number of {@link Parameter}s
	 * of <code>initCapacity</code>.
	 * @param initCapacity the initial number of {@link Parameter}s
	 */
	protected final void initParameterList(int initCapacity){
		this.parameters = new ParameterList(initCapacity);
	}
	
	/**
	 * Loads the parameters for this {@link ParameterSet}. This is in
	 * most cases done by calling {@link ParameterSet#initParameterList()} or {@link ParameterSet#initParameterList(int)}
	 * to initialize {@link ParameterSet#parameters} and afterwards filling {@link ParameterSet#parameters} with instances of subclasses of {@link Parameter}.
	 * @throws Exception an <code>Exception</code> is thrown if the parameters could not be loaded
	 * @see ParameterSet#parameters
	 * @see de.jstacs.parameters.Parameter
	 */
	protected abstract void loadParameters() throws Exception;

	/**
	 * Returns <code>true</code> if the parameters of this
	 * {@link ParameterSet} have already been loaded using the
	 * <code>loadParameters()</code>-method
	 * 
	 * @return the state of the parameters
	 */
	public boolean parametersLoaded()
	{
		return ( parameters != null );
	}

	/**
	 * Returns the index of the first parameter in the set of required
	 * parameters that has not been set. If all required parameters have been
	 * set, it returns -1;
	 * 
	 * @return the index of the required parameter that has not been set
	 */
	/*public int getIndexOfRequiredParameterNotSet()
	{
		if( requiredParameters == null )
		{
			return -1;
		}
		else
		{
			for( int i = 0; i < requiredParameters.length; i++ )
			{
				if( !requiredParameters[i].isSet() )
				{
					return i;
				}
			}
			return -1;
		}
	}*/

	/**
	 * Returns true if this {@link ParameterSet} contains only atomic
	 * parameters, i.e. the parameters do not contain {@link ParameterSet}s
	 * themselves.
	 * 
	 * @return if atomic
	 */
	public boolean isAtomic()
	{
		if( parameters == null )
		{
			try
			{
				loadParameters();
                if(ranged){
                    replaceParametersWithRangedInstance();
                }
			} catch( Exception e )
			{
				e.printStackTrace();
				return false;
			}
		}
		for( int i = 0; i < parameters.size(); i++ )
		{
			if( !parameters.get( i ).isAtomic() )
			{
				return false;
			}
		}
		return true;
	}

	/**
	 * Returns true if all parameters in this {@link ParameterSet} are
	 * either set by the user or have default values. If any additional
	 * constraints are required on your parameters you should either specify
	 * some <code>ParameterValidator</code> on these parameters or implement
	 * these constraints by overriding this method in your implementation of
	 * {@link ParameterSet}. It is recommended to specify a useful
	 * remark which constraint failed in the member-variable
	 * <code>errorMessage</code>, which will be displayed to the user. In the
	 * overriding method super.hasDefaultOrIsSet() should be called prior to
	 * checking specific constraints.
	 * 
	 * @return true if all parameters have some allowed value set
	 */
	public boolean hasDefaultOrIsSet()
	{
		if( parameters == null )
		{
			try
			{
				loadParameters();
                if(ranged){
                    replaceParametersWithRangedInstance();
                }
			} catch( Exception e )
			{
				errorMessage = "Parameters could not be loaded";
				e.printStackTrace();
				return false;
			}
		}
		for( int i = 0; i < parameters.size(); i++ )
		{
			if( parameters.get( i ).isRequired() && ( !parameters.get( i ).hasDefaultOrIsSet() ) )
			{
				//errorMessage = "Parameter no."+i+" ("+parameters.get(i).getName()+") was not set.";
                /*if(parameters.get(i).getErrorMessage() != null){
                    errorMessage = "Parameter no. "+(i+1)+" had the following error: "+parameters.get(i).getErrorMessage();
                }*/
				return false;
			}
		}
        errorMessage = null;
		return true;
	}

	/**
	 * Returns the message of the last error that occurred. If no error occurred this method returns <code>null</code>.
	 * @return the message
	 */
	public String getErrorMessage()
	{
		return errorMessage;
	}

	/**
	 * Constructs a {@link ParameterSet} out of an array of parameters.
	 * The parameters are not cloned, but passed by reference.
	 * 
	 * @param parameters
	 *            the parameters
	 */
	protected ParameterSet( Parameter[] parameters )
	{
		this.id = System.currentTimeMillis() + this.hashCode();
		initParameterList( parameters.length );
		for( int i = 0; i < parameters.length; i++ )
		{
			this.parameters.add( parameters[i] );
		}
	}

	/**
	 * Constructs a {@link ParameterSet} out of an <code>ArrayList</code>
	 * of parameters. The parameters are not cloned, but passed by reference.
	 * 
	 * @param parameters
	 *            the parameters
	 */
	protected ParameterSet( ArrayList<Parameter> parameters )
	{
		this.id = System.currentTimeMillis() + this.hashCode();
		initParameterList();
		Iterator<Parameter> parIt = parameters.iterator();
		while( parIt.hasNext() )
		{
			this.parameters.add( parIt.next() );
		}
	}

	/**
	 * Constructs a {@link ParameterSet} out of an XML representation
	 * 
	 * @param representation
	 *            the XML representation
	 * @throws NonParsableException
	 *             if the {@link ParameterSet} could not be reconstructed
	 *             out of the representation, a
	 *             <code>NonParsableException</code> is thrown
	 */
	public ParameterSet( StringBuffer representation ) throws NonParsableException
	{
		fromXML( representation );
	}

	/**
	 * Returns the number of parameters in set
	 * 
	 * @return the number of parameters
	 */
	public int getNumberOfParameters()
	{
		if( parameters == null )
		{
			try
			{
				loadParameters();
                if(ranged){
                    replaceParametersWithRangedInstance();
                }
			} catch( Exception e )
			{
				e.printStackTrace();
				return 0;
			}
		}
		if(parameters != null){
			return parameters.size();
		}else{
			return 0;
		}
	}

	/**
	 * Returns the parameter at position <code>i</code>
	 * 
	 * @param i
	 *            the position
	 * @return the parameter
	 */
	public Parameter getParameterAt( int i )
	{
		if( parameters == null )
		{
			try
			{
				loadParameters();
                if(ranged){
                    replaceParametersWithRangedInstance();
                }
			} catch( Exception e )
			{
				e.printStackTrace();
				return null;
			}
		}
		return parameters.get( i );
	}

	/**
	 * Replaces all {@link Parameter}s in this {@link ParameterSet}
	 * by their equivalents implementing the {@link Rangeable} interface.
	 * @throws Exception is thrown if these instances could not be created
	 */
    public void makeRanged() throws Exception{
        if(parameters != null){
            replaceParametersWithRangedInstance();
        }
        this.ranged = true;
    }
    
    public boolean next() throws ParameterException
    {
    	if( ranged )
    	{
    		RangeIterator ri;
    		for( int i=0; i<parameters.size(); i++ )
    		{
				if( parameters.get(i) instanceof RangeIterator )
				{
					ri = (RangeIterator)parameters.get(i);
					if( ri.next() )
					{
						return true;
					}
					else
					{
						ri.resetToFirst();
					}
				}
    		}
    	}
    	return false;
    }
    
    public void resetToFirst()
    {
		if( ranged )
		{
			for( int i=0; i<parameters.size(); i++ )
			{
				if( parameters.get(i) instanceof RangeIterator )
				{
					((RangeIterator)parameters.get(i)).resetToFirst();
				}
			}
		}
    }
    
	public int getNumberOfValues()
	{
		if( ranged )
		{
			int erg = 1;
			for( int i=0; i<parameters.size(); i++ )
			{
				if( parameters.get(i) instanceof RangeIterator )
				{
					erg *= ((RangeIterator)parameters.get(i)).getNumberOfValues();
				}
			}
			return erg;
		}
		else
		{
			return 1;
		}
	}
	
	public boolean isRanged()
	{
        if( ranged )
        {
        	return getNumberOfValues()>1;
        }
        else
        {
        	return false;
        }
    }
	
	public String valuesToString()
	{
		String erg = null;
		for( int i=0; i<parameters.size(); i++ )
		{
			if( ranged && parameters.get(i) instanceof RangeIterator )
			{
				if( erg == null )
				{
					erg = ((RangeIterator)parameters.get(i)).valuesToString();
				}
				else
				{
					erg += " x " + ((RangeIterator)parameters.get(i)).valuesToString();
				}
			}
			else
			{
				if( erg == null )
				{
					erg = "[" + parameters.get(i).getValue().toString() + "]"; 
				}
				else
				{
					erg += " x [" + parameters.get(i).getValue().toString() + "]";
				}
			}
		}
		return erg;
	}

	/**
	 * Replaces all {@link Parameter}s in this {@link ParameterSet}
	 * by their equivalents implementing the {@link Rangeable} interface.
	 * @throws Exception is thrown if these instances could not be created
	 */
	protected void replaceParametersWithRangedInstance() throws Exception{
	    for(int i=0;i<parameters.size();i++){
	        if(parameters.get(i) instanceof Rangeable){
	            Rangeable r = (Rangeable) parameters.get(i);
                if(r.isRangeable()){
                	parameters.set(i,r.getRangedInstance());
                }
            }
        }
    }

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.bicgh.seqsignals.platform.Storable#toXML()
	 */
	public StringBuffer toXML()
	{
		StringBuffer buf = new StringBuffer();
		if( parameters != null )
		{
			StringBuffer buf2 = new StringBuffer();
			XMLParser.appendIntWithTags( buf2, parameters.size(), "numberOfParameters" );
			for( int i = 0; i < parameters.size(); i++ )
			{
				if( parameters.get( i ) == null )
				{
					XMLParser.appendStringWithTags( buf2, "null", "parameter" );
				}
				else
				{
					StringBuffer buf3 = new StringBuffer();
					XMLParser.appendStringWithTags( buf3, parameters.get( i ).getClass().getName(), "className" );
					buf3.append( parameters.get( i ).toXML() );
					XMLParser.addTags( buf3, "parameter" );
					buf2.append( buf3 );
				}
			}
			XMLParser.addTags( buf2, "set" );
			buf.append( buf2 );
		}
		else
		{
			XMLParser.appendStringWithTags( buf, "null", "set" );
		}
		XMLParser.appendBooleanWithTags(buf,ranged,"ranged");
		XMLParser.appendLongWithTags( buf, id, "id" );
		XMLParser.addTags( buf, "parameterSet" );

		return buf;
	}

	/**
	 * Simplifies all parameters in this {@link ParameterSet}
	 * 
	 * @see de.jstacs.parameters.Parameter#simplify()
	 * 
	 */
	public void simplify()
	{
		if( parameters == null )
		{
			return;
		}
		for( int i = 0; i < parameters.size(); i++ )
		{
			if( parameters.get( i ) instanceof CollectionParameter )
			{
				parameters.get( i ).simplify();
			}
			else if( parameters.get( i ).getValue() instanceof ParameterSet )
			{
				( (ParameterSet) parameters.get( i ).getValue() ).simplify();
			}
		}
	}

	/**
	 * Resets all parameters in this {@link ParameterSet} to their
	 * default values or <code>null</code> if not default value was provided.
	 * 
	 * @see de.jstacs.parameters.Parameter#reset()
	 */
	public void reset()
	{
		/*if( parameters == null )
		{
			return;
		}
		for( int i = 0; i < parameters.size(); i++ )
		{
			parameters.get( i ).reset();
		}*/
        parameters = null;
	}

	
	/**
	 * Parses the instance fields of a {@link ParameterSet} from the XML-representation
	 * as returned by {@link ParameterSet#toXML()}.
	 * @param representation the XML-representation
	 * @throws NonParsableException is thrown if the XML-code could not be parsed
	 */
	protected void fromXML( StringBuffer representation ) throws NonParsableException
	{

		representation = XMLParser.extractForTag( representation, "parameterSet" );
		StringBuffer buf = XMLParser.extractForTag( representation, "set" );
		if( !buf.toString().equals( "null" ) )
		{
			int numPars = XMLParser.extractIntForTag( buf, "numberOfParameters" );
			parameters = new ParameterList( numPars );
			for( int i = 0; i < numPars; i++ )
			{
				String par = XMLParser.extractStringForTag( buf, "parameter" );
				if( par.equals( "null" ) )
				{
					parameters.add( null );
				}
				else
				{
					StringBuffer buf2 = new StringBuffer( par );
					String className = XMLParser.extractStringForTag( buf2, "className" );
					try
					{
						parameters.add( (Parameter) Class.forName( className ).getConstructor( new Class[] { StringBuffer.class } )
								.newInstance( buf2 ) );
					} catch( Exception e )
					{
						System.out.println( "parameter: " + i );
						System.out.println( "class    : " + className );
						System.out.println( "buffer   : " + par );
						NonParsableException n = new NonParsableException( e.getMessage() );
						n.setStackTrace( e.getStackTrace() );
						System.out.print("ParameterSet: ");
						e.printStackTrace();
						throw n;
					}
				}
			}
		}
		StringBuffer help = XMLParser.extractForTag(representation,"ranged");
		if( help == null )
		{
			ranged = false;
		}
		else
		{
			ranged = Boolean.parseBoolean( help.toString() );
		}
		id = XMLParser.extractLongForTag( representation, "id" );
		propagateId(id,this, false);
		recieveId();
	}
	
	/**
	 * Searches for all {@link Parameter#neededReferenceId}s in the hierarchy below this
	 * {@link ParameterSet} and sets the corresponding {@link Parameter#neededReference}s.
	 */
	protected void recieveId(){
		if(parameters != null){
			Iterator<Parameter> it = parameters.iterator();
			while(it.hasNext()){
				Parameter par = it.next();
				Long neededId = par.getNeededReferenceId();
				if(neededId != null){
					par.setNeededReference( findParameterSet( neededId, false ) );
				}
			}
		}
	}
	
	private ParameterSet findParameterSet(long id, boolean fwd){
		if(id == this.id){
			return this;
		}else{
			if( !fwd && this.getParent() != null && this.getParent().getParent() != null){
				ParameterSet set = this.getParent().getParent().findParameterSet( id, false );
				if(set != null){
					return set;
				}else{
					return null;
				}
			}else if( parameters != null){
				Iterator<Parameter> it = parameters.iterator();
				while(it.hasNext()){
					Parameter par = it.next();
					if(par instanceof ParameterSetContainer){
						ParameterSet set = ((ParameterSetContainer) par).getValue().findParameterSet( id, true );
						if(set != null){
							return set;
						}
					}
				}
			}
			
			return null;
		}
	}
	
	/**
	 * Propagates the id of this {@link ParameterSet} to all {@link Parameter}s
	 * above and below in the hierarchy.
	 */
	protected void propagateId(){
		propagateId( id, this, false );
	}
	
	private boolean propagateId(long id, ParameterSet set, boolean fwd){
		if( !fwd && this.getParent() != null && this.getParent().getParent() != null){
			return this.getParent().getParent().propagateId( id, set, false );
		}else if(parameters != null){
			Iterator<Parameter> it = this.parameters.iterator();
			while(it.hasNext()){
				Parameter par = it.next();
				if(par.getNeededReferenceId() != null && par.getNeededReferenceId() == id){
					par.setNeededReference( set );
					return true;
				}
				if(par instanceof ParameterSetContainer){
					boolean re = ((ParameterSetContainer) par).getValue().propagateId( id, set, true );
					if(re){
						return true;
					}
				}
			}
			return false;
		}
		return false;
	}
	
	/**
	 * Returns the id of this {@link ParameterSet}.
	 * @return the id
	 */
	public long getId(){
		return id;
	}
	

	/**
	 * Returns the enclosing {@link ParameterSetContainer} of this {@link ParameterSet}
	 * or <code>null</code> if none exists.
	 * @return the parent
	 */
	public ParameterSetContainer getParent() {
		return parent;
	}


	/**
	 * Sets the enclosing {@link ParameterSetContainer} of this {@link ParameterSet}
	 * to <code>parent</code>.
	 * @param parent the new parent
	 */
	public void setParent( ParameterSetContainer parent ) {
		this.parent = parent;
	}
	
	/**
	 * Class for a {@link java.util.List} of {@link Parameter}s that basically has the same functionality
	 * as {@link ArrayList}, but additionally takes care of the references {@link Parameter#parent}.
	 * 
	 * @author Jan Grau
	 *
	 */
	public class ParameterList extends ArrayList<Parameter>{

		/**
		 * 
		 */
		private static final long serialVersionUID = 5646161850340681495L;

		/**
		 * Creates a new, empty {@link ParameterList}.
		 */
		public ParameterList() {
			super();
		}

		/**
		 * Creates a new {@link ParameterList} from an existing {@link Collection}
		 * of {@link Parameter}s. This may be another {@link ParameterList}.
		 * @param c the collection
		 */
		public ParameterList( Collection<? extends Parameter> c ) {
			super( c );
			Iterator<? extends Parameter> it = c.iterator();
			while(it.hasNext()){
				it.next().setParent(ParameterSet.this);
			}
		}

		/**
		 * Creates a new, empty {@link ParameterList} with a defined initial capacity.
		 * @param initialCapacity the initial capacity
		 */
		public ParameterList( int initialCapacity ) {
			super( initialCapacity );
		}

		@Override
		public void add( int index, Parameter element ) {
			element.setParent(ParameterSet.this);
			super.add( index, element );
		}

		@Override
		public boolean add( Parameter o ) {
			o.setParent(ParameterSet.this);
			return super.add( o );
		}

		@Override
		public boolean addAll( Collection<? extends Parameter> c ) {
			Iterator<? extends Parameter> it = c.iterator();
			while(it.hasNext()){
				it.next().setParent(ParameterSet.this);
			}
			return super.addAll( c );
		}

		@Override
		public boolean addAll( int index, Collection<? extends Parameter> c ) {
			Iterator<? extends Parameter> it = c.iterator();
			while(it.hasNext()){
				it.next().setParent(ParameterSet.this);
			}
			return super.addAll( index, c );
		}
		
		
		
	}

}
