/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.parameters.validation;

import java.util.Iterator;
import java.util.LinkedList;

import de.jstacs.NonParsableException;
import de.jstacs.io.XMLParser;

/**
 * Class for a <code>ParameterValidator</code> that is based on <code>Constraint</code>s. Each instance
 * of a <code>ConstraintValidator</code> may contain a set of constraints that are all fulfilled, if the 
 * <code>checkValue()</code>-method returns <code>true</code>
 * @author Jan Grau
 *
 */
public class ConstraintValidator implements ParameterValidator {

	private LinkedList<Constraint> constraints;
	private String errorMessage;
	
    /**
     * Creates a new <code>ConstraintValidator</code> having an empty list of <code>Constraints</code>,
     * i.e. the constraints of this <code>ConstraintValidator</code> are always fulfilled before 
     * additional <code>Constraint</code>s are added using the <code>addConstraint()</code>-method.
     *
     */
	public ConstraintValidator(){
		constraints = new LinkedList<Constraint>();
	}
	
    /**
     * Creates a new <code>ConstraintValidator</code> from its XML-representation.
     * @param representation the XML-representation
     * @throws NonParsableException is thrown if <code>representation</code> could not be parsed.
     */
	public ConstraintValidator(StringBuffer representation) throws NonParsableException{
		fromXML(representation);
	}
	
	public ConstraintValidator clone() throws CloneNotSupportedException{
		ConstraintValidator clone = new ConstraintValidator();
		Iterator<Constraint> constIt = constraints.iterator();
		while(constIt.hasNext()){
			clone.constraints.add(constIt.next().clone());
		}
		clone.errorMessage = errorMessage;
		return clone;
	}
	
    /**
     * Adds an additional <code>Constraint</code> to the list of <code>Constraint</code>s.
     * @param c the <code>Constraint</code> to be added
     */
	public void addConstraint(Constraint c){
		constraints.add(c);
	}
	
	public boolean checkValue(Object value) {
		Iterator<Constraint> it = constraints.iterator();
		boolean checkPassed = true;
		errorMessage = null;
		while(it.hasNext()){
			if(! it.next().check(value)){
				if(errorMessage == null){
					errorMessage = it.next().getErrorMessage();
				}else{
					errorMessage += "; "+it.next().getErrorMessage();
				}
				checkPassed = false;
			}
		}
		return checkPassed;
	}

	public String getErrorMessage() {
		return errorMessage;
	}

	public StringBuffer toXML() {
		StringBuffer buf = new StringBuffer();
		XMLParser.appendStringWithTags(buf,errorMessage,"errorMessage");
		XMLParser.appendIntWithTags(buf,constraints.size(),"size");
		Iterator<Constraint> it = constraints.iterator();
		StringBuffer buf2 = new StringBuffer();
		while(it.hasNext()){
			XMLParser.appendStorableWithTags(buf2,it.next(),"constraint");
		}
		XMLParser.addTags(buf2,"constraints");
		buf.append(buf2);
		XMLParser.addTags(buf,"referenceValidator");
		return buf;
		
	}

	/**
	 * Parses a {@link ConstraintValidator} from the XML-representation
	 * as returned by {@link ConstraintValidator#toXML()}.
	 * @param representation the XML-representation
	 * @throws NonParsableException thrown if the XML-code could not be parsed
	 */
	public void fromXML(StringBuffer representation)
			throws NonParsableException {
		representation = XMLParser.extractForTag(representation,"referenceValidator");
		errorMessage = XMLParser.extractStringForTag(representation,"errorMessage");
		int size = XMLParser.extractIntForTag(representation,"size");
		constraints = new LinkedList<Constraint>();
		representation = XMLParser.extractForTag(representation,"constraints");
		for(int i=0;i<size;i++){
			constraints.add( (Constraint) XMLParser.extractStorableForTag(representation,"constraint") );
		}

	}

}
