/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.scoringFunctions;


/**
 * The interface for normalizable ScoringFunctions.
 * 
 * @author Jens Keilwagen, Jan Grau
 */
public interface NormalizableScoringFunction extends ScoringFunction
{
	/**
	 * Returns the size of the event space of the random variables that are affected by parameter no. <code>index</code>,
	 * i.e. the product of the sizes of the alphabets at the position of each random variable affected by parameter
	 * <code>index</code>. For DNA-alphabets this corresponds to 4 for a PWM, 16 for a WAM except position 0, ...
	 * 
	 * @param index
	 *            the index of the parameter
	 * @return the size of the event space
	 */
	public abstract int getSizeOfEventSpaceForRandomVariablesOfParameter( int index );

	/**
	 * Returns the sum of the scores over all sequences of the event space.
	 * 
	 * @return the normalization constant Z
	 */
	public abstract double getNormalizationConstant();

	/**
	 * Returns the partial normalization constant for the parameter with index <code>parameterIndex</code>. This is
	 * the partial derivation of the normalization constant for the parameter with index <code>parameterIndex</code>
	 * <code>\frac{\partial Z(\lambda)}{\partial \lambda_{index}}</code>.
	 * 
	 * @param parameterIndex
	 *            the index of the parameter
	 * @return the partial normalization constant
	 * 
	 * @throws Exception if something went wrong with the Normalization
	 */
	public abstract double getPartialNormalizationConstant( int parameterIndex ) throws Exception;

	/**
	 * Returns the equivalent sample size of this model, i.e. the equivalent sample size for the class or component that
	 * is represented by this model.
	 * 
	 * @return the equivalent sample size.
	 */
	public abstract double getEss();

	/**
	 * This method computes a value that is proportional to
	 * 
	 * <p>
	 * <code>getESS()*Math.log( getNormalizationConstant() ) + Math.log( prior ).</code>
	 * </p>
	 * 
	 * where <code>prior<code> is the prior for the parameters of this model.
	 * 
	 * @return <code>getESS()*Math.log( getNormalizationConstant() ) + Math.log( prior )</code>
	 * 
	 * @see NormalizableScoringFunction#getEss()
	 * @see NormalizableScoringFunction#getNormalizationConstant()
	 */
	public abstract double getLogPriorTerm();

	/**
	 * This method computes the gradient of <code>getLogPriorTerm()</code> for each parameter of this model. The
	 * results are added to the array <code>grad</code> beginning at index <code>start</code>.
	 * 
	 * @param grad
	 *            the gradient
	 * @param start
	 *            the start index in the <code>grad</code> array, where the partial derivations for the parameters of
	 *            this models shall be enter
	 * @throws Exception 
	 *            
	 * @see NormalizableScoringFunction#getLogPriorTerm()
	 */
	public abstract void addGradientOfLogPriorTerm( double[] grad, int start ) throws Exception;
	
	/**
	 * This method returns whether the implemented score is already normalized to 1. The standard implementation
	 * returns <code>false</code>.
	 * 
	 * @return whether the implemented score is already normalized to 1
	 */
	public boolean isNormalized();
	
	public double getInitialClassParam( double classProb );
}
