/*
 * This file is part of Jstacs.
 * 
 * Jstacs is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * Jstacs is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * Jstacs. If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.algorithms.optimization;

/**
 * This class is the framework for any (at least) one time differentiable
 * function f: R^n -> R.
 * 
 * @author Jens Keilwagen
 */
public abstract class DifferentiableFunction implements Function {

	/**
	 * Evaluates the gradient of a function at a certain vector (in mathematical
	 * sense) <code>x</code>.
	 * 
	 * @param x
	 *            the current vector
	 * 
	 * @return the evaluation of the gradient of a function; has dimension
	 *         {@link Function#getDimensionOfScope()}
	 * 
	 * @throws DimensionException
	 *             if dim(x) != n, with f: R^n -> R
	 * @throws EvaluationException
	 *             if there was something wrong while evaluating the gradient
	 * 
	 * @see Function#getDimensionOfScope()
	 */
	public abstract double[] evaluateGradientOfFunction( double[] x ) throws DimensionException, EvaluationException;

	/**
	 * This method is used to find an approximation of an one-dimensional
	 * subfunction. That means it will find an approximation of the minimum
	 * starting at point <code>current</code> and search in direction
	 * <code>d</code>.<br>
	 * <br>
	 * 
	 * \argmin_{alpha >= 0} f(current + alpha*d)<br>
	 * <br>
	 * 
	 * This method is a standard implementation. You are enabled to overwrite
	 * this method to be faster, if you know anything about the problem or if
	 * you just want to test other line search methods.
	 * 
	 * @param current
	 *            the start point
	 * @param d
	 *            the search direction
	 * @param alpha_0
	 *            the initial alpha
	 * @param fAlpha_0
	 *            the initial function value (this value is known in most cases
	 *            and does not have to be computed again)
	 * @param linEps
	 *            the tolerance for stopping this method
	 * @param startDistance
	 *            the initial distance for bracketing the minimum
	 * 
	 * @return <code>double[2] res = { alpha*, f(alpha*) }</code>
	 * 
	 * @throws DimensionException
	 *             if there is something wrong with the dimension
	 * @throws EvaluationException
	 *             if there was something wrong while evaluating the function
	 * 
	 * @see OneDimensionalFunction#findMin(double, double, double, double)
	 */
	protected double[] findOneDimensionalMin( double[] current, double[] d, double alpha_0, double fAlpha_0, double linEps,
			double startDistance ) throws DimensionException, EvaluationException {
		return ( new OneDimensionalSubFunction( this, current, d ) ).findMin( alpha_0, fAlpha_0, linEps, startDistance );
	}
}
