/*
 * This file is part of Jstacs.
 * 
 * Jstacs is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * Jstacs is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * Jstacs. If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.data.alphabets;

import de.jstacs.NonParsableException;

/**
 * This class implements the alphabet that is used for DNA.
 * 
 * @author Jan Grau, Jens Keilwagen
 */
public final class DNAAlphabet extends ComplementableDiscreteAlphabet {

	/**
	 * The constructor for the {@link de.jstacs.Storable} interface.
	 * 
	 * @param representation
	 *            the XML representation
	 * 
	 * @throws NonParsableException
	 *             if the stream is not parsable
	 */
	public DNAAlphabet( StringBuffer representation ) throws NonParsableException {
		super( representation );
	}

	/**
	 * The constructor for the {@link de.jstacs.InstantiableFromParameterSet}
	 * interface.
	 * 
	 * @param parameters
	 *            the {@link de.jstacs.parameters.ParameterSet}
	 * 
	 * @throws IllegalArgumentException
	 *             is thrown if space or tab will be used as symbols
	 * @throws DoubleSymbolException
	 *             is thrown if one of the symbols occurred more than once
	 */

	public DNAAlphabet( DNAAlphabetParameterSet parameters ) throws IllegalArgumentException, DoubleSymbolException {
		this();
		this.parameters = parameters;
	}

	/**
	 * The main constructor.
	 * 
	 * @throws DoubleSymbolException
	 *             is thrown if one of the symbols occurred more than once
	 * @throws IllegalArgumentException
	 *             is thrown if space or tab will be used as symbols
	 */
	public DNAAlphabet() throws DoubleSymbolException, IllegalArgumentException {
		super( DNAAlphabetParameterSet.dna, true );
	}

	/* (non-Javadoc)
	 * @see de.jstacs.data.alphabets.ComplementableDiscreteAlphabet#getComplementaryCode(int)
	 */
	@Override
	public int getComplementaryCode( int code ) {
		return 3 - code;
	}

	/* (non-Javadoc)
	 * @see de.jstacs.data.alphabets.ComplementableDiscreteAlphabet#getCurrentParameterSet()
	 */
	@Override
	public AlphabetParameterSet getCurrentParameterSet() throws Exception {
		if( this.parameters == null ) {
			return new DNAAlphabetParameterSet();
		} else {
			return this.parameters;
		}
	}

	/**
	 * The parameter set for a {@link DNAAlphabet}.
	 * 
	 * @author Jan Grau, Jens Keilwagen
	 */
	public static final class DNAAlphabetParameterSet extends AlphabetParameterSet {

		private static final String[] dna = { "A", "C", "G", "T" };

		/**
		 * Creates a new {@link DNAAlphabetParameterSet}.
		 * 
		 * @throws Exception
		 *             if something went wrong
		 */
		public DNAAlphabetParameterSet() throws Exception {
			super( DNAAlphabet.class );
		}

		/**
		 * Creates a new {@link DNAAlphabetParameterSet} from its XML
		 * representation.
		 * 
		 * @param representation
		 *            the XML representation
		 * @throws NonParsableException
		 *             is thrown if <code>representation</code> could not be
		 *             parsed
		 */
		public DNAAlphabetParameterSet( StringBuffer representation ) throws NonParsableException {
			super( representation );
		}

		/* (non-Javadoc)
		 * @see de.jstacs.parameters.ParameterSet#loadParameters()
		 */
		@Override
		protected void loadParameters() throws Exception {
			initParameterList();
		}

		/* (non-Javadoc)
		 * @see de.jstacs.parameters.InstanceParameterSet#getInstanceComment()
		 */
		@Override
		public String getInstanceComment() {
			return "An alphabet for DNA.";
		}
	}
}
