/*
 * This file is part of Jstacs.
 * 
 * Jstacs is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * Jstacs is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * Jstacs. If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.data.sequences.annotation;

import java.util.Collection;

import de.jstacs.NonParsableException;
import de.jstacs.io.XMLParser;
import de.jstacs.results.Result;

/**
 * Class for a {@link SequenceAnnotation} that has a position, a length, and an
 * orientation on the strand of a {@link de.jstacs.data.Sequence}.
 * 
 * @author Jan Grau
 */
public class StrandedLocatedSequenceAnnotationWithLength extends LocatedSequenceAnnotationWithLength {

	/**
	 * The possible orientations on the strands.
	 * 
	 * @author Jan Grau
	 */
	public enum Strand {
		/**
		 * Annotation is located on the forward strand
		 */
		FORWARD( "Forward strand" ),
		/**
		 * Annotation is located on the reverse strand
		 */
		REVERSE( "Reverse strand" ),
		/**
		 * The orientation of the annotation is not known
		 */
		UNKNOWN( "Unknown strand" ), ;

		private final String strandedness;

		Strand( String strandedness ) {
			this.strandedness = strandedness;
		}

		/**
		 * Returns the strandedness as a {@link String}.
		 * 
		 * @return the strandedness
		 */
		public String strandedness() {
			return strandedness;
		}
	};

	private String strandedness;

	/**
	 * Creates a new {@link StrandedLocatedSequenceAnnotationWithLength} of type
	 * <code>type</code>, with identifier <code>identifier</code>, and
	 * additional annotation (that does not fit the {@link SequenceAnnotation}
	 * definitions) <code>result</code>.
	 * 
	 * @param position
	 *            the position of the
	 *            {@link StrandedLocatedSequenceAnnotationWithLength} on the
	 *            sequence
	 * @param length
	 *            the length of the
	 *            {@link StrandedLocatedSequenceAnnotationWithLength}
	 * @param strandedness
	 *            the orientation on the strand
	 * @param type
	 *            the type of the annotation
	 * @param identifier
	 *            the identifier of the annotation
	 * @param results
	 *            the additional annotation
	 */
	public StrandedLocatedSequenceAnnotationWithLength( int position, int length, Strand strandedness, String type, String identifier,
														Result... results ) {
		super( position, length, type, identifier, results );
		this.strandedness = strandedness.strandedness();
	}

	/**
	 * Creates a new {@link StrandedLocatedSequenceAnnotationWithLength} of type
	 * <code>type</code>, with identifier <code>identifier</code>, and
	 * additional annotation (that does not fit the {@link SequenceAnnotation}
	 * definitions) <code>result</code>.
	 * 
	 * @param position
	 *            the position of the
	 *            {@link StrandedLocatedSequenceAnnotationWithLength} on the
	 *            sequence
	 * @param length
	 *            the length of the
	 *            {@link StrandedLocatedSequenceAnnotationWithLength}
	 * @param strandedness
	 *            the orientation on the strand
	 * @param type
	 *            the type of the annotation
	 * @param identifier
	 *            the identifier of the annotation
	 * @param results
	 *            the additional annotation
	 */
	public StrandedLocatedSequenceAnnotationWithLength( int position, int length, Strand strandedness, String type, String identifier,
														Collection<Result> results ) {
		super( position, length, type, identifier, results );
		this.strandedness = strandedness.strandedness();
	}

	/**
	 * Creates a new {@link StrandedLocatedSequenceAnnotationWithLength} of type
	 * <code>type</code>, with identifier <code>identifier</code>, and
	 * additional annotation (that does not fit the {@link SequenceAnnotation}
	 * definitions) <code>additionalAnnotations</code>, and sub-annotations.
	 * 
	 * @param position
	 *            the position of the
	 *            {@link StrandedLocatedSequenceAnnotationWithLength} on the
	 *            sequence
	 * @param length
	 *            the length of the
	 *            {@link StrandedLocatedSequenceAnnotationWithLength}
	 * @param strandedness
	 *            the orientation on the strand
	 * @param type
	 *            the type of the annotation
	 * @param identifier
	 *            the identifier of the annotation
	 * @param annotations
	 *            the sub-annotations
	 * @param additionalAnnotations
	 *            the additional annotation
	 */
	public StrandedLocatedSequenceAnnotationWithLength( int position, int length, Strand strandedness, String type, String identifier,
														SequenceAnnotation[] annotations, Result... additionalAnnotations ) {
		super( position, length, type, identifier, annotations, additionalAnnotations );
		this.strandedness = strandedness.strandedness();
	}

	/**
	 * Creates a new {@link StrandedLocatedSequenceAnnotationWithLength} of type
	 * <code>type</code>, with identifier <code>identifier</code>, and
	 * additional annotation (that does not fit the {@link SequenceAnnotation}
	 * definitions) <code>additionalAnnotations</code>, and sub-annotations. The
	 * position of the new {@link LocatedSequenceAnnotationWithLength} is the
	 * minimal position of all positions of <code>annotations</code> and the
	 * length is determined such that it is the maximum of these positions and
	 * (if applicable) the corresponding values of
	 * {@link StrandedLocatedSequenceAnnotationWithLength#getEnd()}.
	 * 
	 * @param strandedness
	 *            the orientation on the strand
	 * @param type
	 *            the type of the annotation
	 * @param identifier
	 *            the identifier of the annotation
	 * @param annotations
	 *            the sub-annotations
	 * @param additionalAnnotations
	 *            the additional annotation
	 */
	public StrandedLocatedSequenceAnnotationWithLength( String type, String identifier, Strand strandedness,
														LocatedSequenceAnnotation[] annotations, Result... additionalAnnotations ) {
		super( type, identifier, annotations, additionalAnnotations );
		this.strandedness = strandedness.strandedness();
	}

	/**
	 * Re-creates a {@link StrandedLocatedSequenceAnnotationWithLength} from its
	 * XML representation as returned by
	 * {@link StrandedLocatedSequenceAnnotationWithLength#toXML()}.
	 * 
	 * @param representation
	 *            the XML representation
	 * 
	 * @throws NonParsableException
	 *             is thrown if the XML code could not be parsed
	 */
	public StrandedLocatedSequenceAnnotationWithLength( StringBuffer representation ) throws NonParsableException {
		super( representation );
	}

	/**
	 * Returns the orientation/strandedness of this annotation.
	 * 
	 * @return the strandedness
	 */
	public String getStrandedness() {
		return strandedness;
	}

	/* (non-Javadoc)
	 * @see de.jstacs.data.sequences.annotation.LocatedSequenceAnnotationWithLength#fromXML(java.lang.StringBuffer)
	 */
	@Override
	protected void fromXML( StringBuffer representation ) throws NonParsableException {
		representation = XMLParser.extractForTag( representation, "strandedAnnotation" );
		super.fromXML( XMLParser.extractForTag( representation, "locatedSequenceAnnotationWithLength" ) );
		strandedness = XMLParser.extractStringForTag( representation, "strandedness" );
	}

	/* (non-Javadoc)
	 * @see de.jstacs.data.sequences.annotation.LocatedSequenceAnnotationWithLength#toString()
	 */
	@Override
	public String toString() {
		StringBuffer buf = new StringBuffer( super.toString() );
		buf.append( "strand: " );
		buf.append( strandedness );
		buf.append( "\n" );
		return buf.toString();
	}

	/* (non-Javadoc)
	 * @see de.jstacs.data.sequences.annotation.LocatedSequenceAnnotationWithLength#toXML()
	 */
	@Override
	public StringBuffer toXML() {
		StringBuffer buf = super.toXML();
		XMLParser.addTags( buf, "locatedSequenceAnnotationWithLength" );
		XMLParser.appendStringWithTags( buf, strandedness, "strandedness" );
		XMLParser.addTags( buf, "strandedAnnotation" );
		return buf;
	}

}
