/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.io;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Date;

/**
 * This class is for handling files. The most important methods of this class
 * are for writing a StringBuffer to or reading it from a file. This is useful
 * for all object that implement <code>Storable</code> and should be saved in
 * or loaded from a file.
 * 
 * @author Jens Keilwagen
 * 
 * @see de.jstacs.Storable
 * 
 */
public class FileManager
{
	private FileManager()
	{
	}

	/**
	 * This buffer is used to copy files.
	 */
	private static byte[] buffer = new byte[100000];

	/**
	 * This method copies all files and directories, if selected, from a
	 * <code>source</code> to a <code>target</code> that have been modified
	 * after a predefined <code>date</code>
	 * 
	 * @param source
	 *            the source directory
	 * @param target
	 *            the target directory
	 * @param date
	 *            the date
	 * @param recursive
	 *            a switch if this method should be used recursively in the
	 *            subdirectories
	 * @param filter
	 *            a filter for the files, taht enables the user to copy only
	 *            specific files
	 * 
	 * @return the number of copied files.
	 * 
	 * @throws IllegalArgumentException
	 * @throws IOException
	 */
	public static int copyDiff( File source, File target, Date date, boolean recursive, FileFilter filter ) throws IllegalArgumentException, IOException
	{
		if( !source.isDirectory() || !target.isDirectory() )
		{
			throw new IllegalArgumentException( "The source and the target have to be directories." );
		}
		File[] files = source.listFiles( filter );
		//System.out.println( "xxx " + files.length );
		int anz = 0, i;
		long time = date.getTime();
		boolean existsBefore;
		File help;
		for( File f : files )
		{			
			if( f.isDirectory() )
			{
				if( recursive )
				{
					help = new File( target.getAbsolutePath() + "/" + f.getName() );
					existsBefore = help.exists();
					if( !existsBefore )
					{
						help.mkdir();
					}
					i = copyDiff( f, help, date, recursive, filter);
					if( i == 0 )
					{
						if( !existsBefore )
						{
							help.delete();
						}
					}
					else
					{
						anz += i;
					}
				}					
			}
			else
			{
				if( f.lastModified() >= time )
				{
					//System.out.println( f.getName() + "\t" + target.getAbsolutePath() + "/" + f.getName() );
					
					copy( f.getAbsolutePath(), target.getAbsolutePath() + "/" + f.getName() );
					anz++;
				}
			}
		}
		return anz;
	}
	
	/**
	 * This method copies a file in a faster manner.
	 * 
	 * @param from
	 *            the file name of the original file
	 * @param to
	 *            the file name of the copied file
	 * 
	 * @throws IOException
	 *             if something went wrong
	 */
	public static void copy( String from, String to ) throws IOException
	{
		copy( from, to, buffer );
	}

	/**
	 * This method copies a file in a faster manner using a specified buffer.
	 * 
	 * @param from
	 *            the file name of the original file
	 * @param to
	 *            the file name of the copied file
	 * @param buffer
	 *            an array for reading the content of the original file, the size of the array determines how many byte
	 *            are read at ones.
	 * 
	 * @throws IOException
	 *             if something went wrong
	 */
	public static synchronized void copy( String from, String to, byte[] buffer ) throws IOException
	{
		FileInputStream in = null;
		FileOutputStream out = null;
		try
		{
			in = new FileInputStream( from );
			out = new FileOutputStream( to );
			int amountRead;
			while( (amountRead = in.read( buffer )) > -1 )
			{
				out.write( buffer, 0, amountRead );
			}
		}
		finally
		{
			if( in != null )
			{
				in.close();
			}
			if( out != null )
			{
				out.close();
			}
		}
	}

	/**
	 * This method reads a StringBuffer from a given file.
	 * 
	 * @param file
	 *            the file to be read
	 * 
	 * @return a Stringbuffer with the content of the file
	 * 
	 * @throws IOException
	 *             if something went wrong with the file
	 */
	public static StringBuffer readFile( File file ) throws IOException
	{
		BufferedReader r = new BufferedReader( new FileReader( file ), 100000 );
		StringBuffer res = new StringBuffer( 1000000 );
		String help;
		while( (help = r.readLine()) != null )
		{
			res.append( help + "\n" );
		}
		r.close();
		return res;
	}

	/**
	 * This method saves a StringBuffer to a given file.
	 * 
	 * @param outputFile
	 *            the file into which the output should be written
	 * @param buffer
	 *            the buffer to be written in a file
	 * @throws IOException
	 *             if something went wrong with the file
	 */
	public static void writeFile( File outputFile, StringBuffer buffer ) throws IOException
	{
		BufferedWriter b = new BufferedWriter( new FileWriter( outputFile ) );
		b.write( buffer.toString() );
		b.close();
	}
}
