/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.models.discrete.inhomogeneous;

import java.util.StringTokenizer;

import de.jstacs.NonParsableException;
import de.jstacs.NotTrainedException;
import de.jstacs.data.Sample;
import de.jstacs.io.XMLParser;
import de.jstacs.models.Model;
import de.jstacs.models.discrete.DGMParameterSet;
import de.jstacs.models.discrete.inhomogeneous.parameters.FSDAGMParameterSet;

/**
 * This class can be used for any discrete fixed structure DAG model (FSDAGModel).
 * 
 * @author Jens Keilwagen
 */
public class FSDAGModel extends DAGModel
{
	/**
	 * This is the main constructor. It creates an new instance from the given parameter set.
	 * 
	 * @param params the parameter set
	 * 
	 * @throws CloneNotSupportedException if the parameter set could not be cloned
	 * @throws IllegalArgumentException if the parameter set is not instantiated
	 * @throws NonParsableException if the parameter set is not parsable
	 */
	public FSDAGModel( FSDAGMParameterSet params ) throws CloneNotSupportedException, IllegalArgumentException,
			NonParsableException
	{
		super( params );
	}

	/**
	 * This is the constructor for {@link de.jstacs.Storable}.
	 * 
	 * @param xml the xml representation
	 * 
	 * @throws NonParsableException if the representation could not be parsed.
	 */
	public FSDAGModel( StringBuffer xml ) throws NonParsableException
	{
		super( xml );
	}

	public String getInstanceName()
	{
		return "fixed structure directed acyclic graphical model";
	}

	public byte getMaximalMarkovOrder()
	{
		int max = constraints[0].getMarginalOrder();
		for( int i = 1; i < length; i++ )
		{
			if( max < constraints[i].getMarginalOrder() )
			{
				max = constraints[i].getMarginalOrder();
			}
		}
		return (byte) (max - 1);
	}

	private static final String XML_TAG = "FSDAGModel";

	protected String getXMLTag()
	{
		return XML_TAG;
	}

	public void train( Sample data, double[] weights ) throws Exception
	{
		estimateParameters( data, weights );
	}

	/**
	 * Computes the model with structure <code>graph</code>
	 * 
	 * @param data
	 *            the sample
	 * @param weights
	 *            the weights for the sequences in the sample
	 * @param graph
	 *            the graph
	 * 
	 * @throws Exception
	 *             if something went wrong
	 */
	public void train( Sample data, double[] weights, int[][] graph ) throws Exception
	{
		// check
		if( !checkAcyclic( length, graph ) )
		{
			throw new IllegalArgumentException( "the graph is not acyclic" );
		}
		trainUnchecked( data, weights, graph );
	}

	/**
	 * This method draws the parameters of the model from the a posteriori density. For drawing from the prior you
	 * have to set the data and their weights to <code>null</code>. Furthermore this method enables you to specify a
	 * new graph structure.
	 * 
	 * @param data
	 *            a sample or <code>null</code>
	 * @param weights
	 *            the (positive) weights for each sequence of the sample or <code>null</code>
	 * @param graph
	 *            the graph or <code>null</code> for the current graph
	 * 
	 * @throws Exception
	 */
	public void drawParameters( Sample data, double[] weights, int[][] graph ) throws Exception
	{
		if( graph != null )
		{
			// check
			if( !checkAcyclic( length, graph ) )
			{
				throw new IllegalArgumentException( "the graph is not acyclic" );
			}
			params.getParameterAt( 2 ).setValue( FSDAGMParameterSet.encode( graph ) );
			// create
			createConstraints( graph );
		}
		// draw
		drawParameters( data, weights );
	}

	/**
	 * This method trains the model with given graph structure on the given data.
	 * Be careful if you use this method, since it does not check if the graph is acyclic
	 *  
	 * @param data the data
	 * @param weights the weights for the data
	 * @param graph the given graph
	 * 
	 * @throws Exception if something went wrong
	 * 
	 * @see DAGModel#checkAcyclic(int, int[][])
	 */
	private void trainUnchecked( Sample data, double[] weights, int[][] graph ) throws Exception
	{
		// set parameter
		params.getParameterAt( 2 ).setValue( FSDAGMParameterSet.encode( graph ) );
		// create
		createConstraints( graph );
		// estimate
		estimateParameters( data, weights );
	}

	/**
	 * Computes the models with structure <code>graph</code>
	 * 
	 * @param models
	 *            an array of AbstractModels containing only instances of FSDAGModel
	 * @param data
	 *            the sample
	 * @param weights
	 *            the weights for the sequences in the sample
	 * @param graph
	 *            the graph
	 * 
	 * @throws Exception
	 *             if something went wrong
	 */
	public static void train( Model[] models, int[][] graph, double[][] weights, Sample... data )
			throws Exception
	{
		// check
		if( !checkAcyclic( graph.length, graph ) )
		{
			throw new IllegalArgumentException( "the graph is not acyclic" );
		}
		if( data.length == 1 )
		{
			for( int i = 0; i < models.length; i++ )
			{
				((FSDAGModel) models[i]).trainUnchecked( data[0], weights[i], graph );
			}
		}
		else
		{
			for( int i = 0; i < models.length; i++ )
			{
				((FSDAGModel) models[i]).trainUnchecked( data[i], weights[i], graph );
			}
		}
	}

	protected void set( DGMParameterSet params, boolean trained ) throws CloneNotSupportedException,
			NonParsableException
	{
		super.set( params, trained );
		byte i = 0, j;
		// extract
		StringBuffer help, xml = new StringBuffer( (String) params.getParameterAt( 2 ).getValue() );
		StringTokenizer t;
		int[][] graph = new int[length][];
		for( i = 0; i < length; i++ )
		{
			help = XMLParser.extractForTag( xml, "parents node=" + i, "parents" );
			if( help == null )
			{
				graph[i] = new int[]{ i };
			}
			else
			{
				t = new StringTokenizer( help.toString(), "," );
				graph[i] = new int[t.countTokens() + 1];
				for( j = 0; t.hasMoreTokens(); j++ )
				{
					graph[i][j] = Integer.parseInt( t.nextToken().trim() );
					if( graph[i][j] < 0 || graph[i][j] >= length || graph[i][j] == i )
					{
						throw new IllegalArgumentException( "The graph was not correct encoded. See parents from node "
								+ i );
					}
				}
				graph[i][j] = i;
				if( XMLParser.extractForTag( xml, "parents node=" + i, "parents" ) != null )
				{
					throw new IllegalArgumentException(
							"The graph was not correct encoded. There are at least 2 sets of parents for node " + i );
				}
			}
		}
		// check
		if( !checkAcyclic( length, graph ) )
		{
			throw new IllegalArgumentException( "the graph is not acyclic" );
		}
		// create
		createConstraints( graph );
	}
	
	public String getStructure()
	{
		if( trained )
		{
			try
			{
				return super.getStructure();
			}
			catch( NotTrainedException e )
			{
				// will never happen
				RuntimeException r = new RuntimeException(e.getMessage());
				throw r;
			}
		}
		
		StringBuffer all = new StringBuffer( 500 );
		for( int counter1 = 0; counter1 < constraints.length; counter1++ )
		{
			all.append( constraints[counter1].toString() + "\n" );
		}
		return all.toString();
	}
}
