/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.models.discrete.inhomogeneous.parameters;

import de.jstacs.DataType;
import de.jstacs.NonParsableException;
import de.jstacs.data.AlphabetContainer;
import de.jstacs.io.XMLParser;
import de.jstacs.models.discrete.inhomogeneous.FSDAGModel;
import de.jstacs.parameters.SimpleParameter;

/**
 * The class for the parameters of a {@link FSDAGModel}.
 * 
 * @author Jens Keilwagen
 */
public class FSDAGMParameterSet extends IDGMParameterSet
{
	/**
	 * The constructor for the {@link de.jstacs.Storable} interface.
	 * 
	 * @param s
	 *            the StringBuffer
	 * 
	 * @throws NonParsableException
	 *             if the StringBuffer is not parsable
	 */
	public FSDAGMParameterSet( StringBuffer s ) throws NonParsableException
	{
		super( s );
	}

	/**
	 * This is the constructor that creates an empty parameter set for a {@link FSDAGModel}.
	 */
	public FSDAGMParameterSet( )
	{
		this( FSDAGModel.class );
	}

	/**
	 * This is the constructor that creates a filled parameter set.
	 * 
	 * @param alphabet
	 *            the alphabet container that is used in the model
	 * @param length
	 *            the length of the model (has to be positive)
	 * @param ess
	 *            the equivalent sample size (has to be positive)
	 * @param description
	 *            a short description of the model (used for a better handling of the object by the user)
	 * @param graph
	 *            the graph description string, encodes in XML-like manner the parents of each node &quot;&lt;parents
	 *            node=i&gt;j,k,l&lt;/parents&gt;&quot;
	 * 
	 * @throws Exception
	 *             if something went wrong
	 * 
	 * @see FSDAGMParameterSet#encode(int[][])
	 */
	public FSDAGMParameterSet( AlphabetContainer alphabet, int length, double ess,
			String description, String graph ) throws Exception
	{
		this( FSDAGModel.class, alphabet, length, ess, description, graph );
	}
	
	/**
	 * This is the constructor that creates an empty parameter set.
	 * 
	 * @param clazz
	 *            the class of the object that will be created with this parameter set
	 * 
	 * @see de.jstacs.models.discrete.inhomogeneous.FSDAGModel
	 * @see de.jstacs.models.mixture.gibbssampling.FSDAGModelForGibbsSampling
	 */
	protected FSDAGMParameterSet( Class<? extends FSDAGModel> clazz )
	{
		super( clazz );
	}

	/**
	 * This is the constructor that creates a filled parameter set.
	 * 
	 * @param clazz
	 *            the class of the object that will be created with this parameter set
	 * @param alphabet
	 *            the alphabet container that is used in the model
	 * @param length
	 *            the length of the model (has to be positive)
	 * @param ess
	 *            the equivalent sample size (has to be positive)
	 * @param description
	 *            a short description of the model (used for a better handling of the object by the user)
	 * @param graph
	 *            the graph description string, encodes in XML-like manner the parents of each node &quot;&lt;parents
	 *            node=i&gt;j,k,l&lt;/parents&gt;&quot;
	 * 
	 * @throws Exception
	 *             if something went wrong
	 * 
	 * @see FSDAGMParameterSet#encode(int[][])
	 */
	protected FSDAGMParameterSet( Class<? extends FSDAGModel> clazz, AlphabetContainer alphabet, int length, double ess,
			String description, String graph ) throws Exception
	{
		super( clazz, alphabet, length, ess, description );
		parameters.get( 2 ).setValue( graph );
	}

	protected void loadParameters() throws Exception
	{
		super.loadParameters();
		parameters
				.add( new SimpleParameter(
						DataType.STRING,
						"graph structure",
						"the graph structure for the model (has to be acyclic)&lt;br&gt;"
								+ "The graph has to be given in a semi-XML-format, i.e. if 1, 2 and 5 are the parents from node 0 enter&lt;br&gt;"
								+ "&lt;parents node=0&gt;1,2,5&lt;/parents&gt;&lt;br&gt;"
								+ "the root nodes do not have to be insert explicitly", false ) );
	}

	public String getInstanceComment()
	{
		return "a directed acyclic graphical (DAG) model with fixed structure (FS)";
	}

	/**
	 * This method can be used to encode adjacency list to a graph description string.
	 * 
	 * @param graph
	 *            <code>graph[i]</code> contains the parents of node i
	 * 
	 * @return the graph description string
	 */
	public static String encode( int[][] graph )
	{
		StringBuffer help, encoded = new StringBuffer( 1000 );
		int i = 0, j;
		for( ; i < graph.length; i++ )
		{
			if( graph[i] != null && graph[i].length > 0 )
			{
				help = new StringBuffer( 100 );
				help.append( graph[i][0] );
				for( j = 1; j < graph[i].length; j++ )
				{
					help.append( "," + graph[i][j] );
				}
				XMLParser.addTags( help, "parents node=" + i, "parents" );
				encoded.append( help );
			}
		}
		return encoded.toString();
	}

}
