package de.jstacs.models.mixture.motif.positionprior;

import de.jstacs.NonParsableException;
import de.jstacs.io.XMLParser;

/**
 * This class implements a gaussian like discrete truncated prior.
 * 
 * @author Jens Keilwagen
 */
public class GaussianLikePositionPrior extends PositionPrior
{
	private int length;
	private double max, prec, z;
	
	/**
	 * This constructor creates an instance with given sequence length, maximal value and sigma for the Gaussian.
	 * 
	 * @param length the sequence length
	 * @param max the maximal position
	 * @param sigma the standard deviation (sigma) of the Gaussian 
	 */
	public GaussianLikePositionPrior( int length, double max, double sigma )
	{
		super();
		if( length < 1 )
		{
			throw new IllegalArgumentException( "The length has to be positive." );
		}
		this.length = length;
		if( max < 0 )
		{
			throw new IllegalArgumentException( "The maximum has to be non-negative." );
		}
		this.max = max;
		if( sigma <= 0 )
		{
			throw new IllegalArgumentException( "The sigma has to be non-negative." );
		}
		this.prec = 1d/(2d*sigma*sigma);
	}

	/**
	 * The standard constructor for the interface {@link de.jstacs.Storable}.
	 * 
	 * @param rep
	 *            the StringBuffer containing the model
	 * 
	 * @throws NonParsableException
	 *             if the StringBuffer can not be parsed
	 */
	public GaussianLikePositionPrior( StringBuffer rep ) throws NonParsableException
	{
		super( rep );
	}

	public void setMotifLength( int motifLength )
	{
		super.setMotifLength( motifLength );
		z = 0;
		for( int m = length-motifLength, i = 0; i<= m; i++ )
		{
			z += Math.exp( -prec*(max-i)*(max-i) ); 
		}
		z = Math.log( z );
	}
	/**
	 * Returns only the important part and leaving the logarithm of the normalization constant out.
	 */
	public double getLogPriorForPositions( int seqLength, int... starts ) throws IllegalArgumentException
	{
		if( seqLength == length && starts[0] <= length-motifLength )
		{
			return -prec*(max-starts[0])*(max-starts[0]) - z;
		}
		else
		{
			throw new IllegalArgumentException( "This sequence length could not be modeled." );
		}
	}

	public int getLength()
	{
		return length;
	}

	public String getInstanceName()
	{
		return getClass().getSimpleName();
	}

	protected StringBuffer getAdditionalInformation()
	{
		StringBuffer xml = new StringBuffer( 1000 );
		XMLParser.appendIntWithTags( xml, length, "length" );
		XMLParser.appendDoubleWithTags( xml, max, "max" );
		XMLParser.appendDoubleWithTags( xml, prec, "prec" );
		return xml;
	}

	protected void extractAdditionalInformation( StringBuffer xml ) throws NonParsableException
	{
		length = XMLParser.extractIntForTag( xml, "length" );
		max = XMLParser.extractDoubleForTag( xml, "max" );
		prec = XMLParser.extractDoubleForTag( xml, "prec" );
	}

}
