/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.parameters;

import de.jstacs.DataType;
import de.jstacs.NonParsableException;
import de.jstacs.Storable;
import de.jstacs.io.XMLParser;
import de.jstacs.parameters.SimpleParameter.IllegalValueException;
import de.jstacs.parameters.validation.ParameterValidator;

/**
 * Class for a parameter that represents a local file.
 * 
 * @author Jan Grau
 *
 */
public class FileParameter extends Parameter {

	/**
	 * The name of the parameter
	 */
	private String name;
	/**
	 * The comment on the parameter
	 */
	private String comment;
	/**
	 * <code>true</code> if the parameter is required
	 */
	private boolean required;
	/**
	 * The MIME-type of accepted files
	 */
	private String mime;
	/**
	 * The file
	 */
	private FileRepresentation value;
	/**
	 * The default value
	 */
	private FileRepresentation defaultValue;
	/**
	 * True if a file is set as value
	 */
	private boolean isSet;
	/**
	 * The error message, <code>null</code> if not error occurred
	 */
	private String errorMessage;
	
	/**
	 * The parameter validator
	 */
	private ParameterValidator valid;
	
	
	public FileParameter clone() throws CloneNotSupportedException{
		FileParameter clone = (FileParameter) super.clone();
		clone.value = value == null ? null : value.clone();
		clone.defaultValue = defaultValue == null ? null : defaultValue.clone();
		return clone;
	}
	
	/**
	 * Restores a file parameter from an XML representation.
	 * @param buf the XML representation
	 * @throws NonParsableException if the XML code could not be parsed, an Exception is thrown
	 */
	public FileParameter(StringBuffer buf) throws NonParsableException{
		fromXML(buf);
	}
	
	/**
	 * Creates a file parameter.
	 * @param name the name of the parameter
	 * @param comment a comment on the parameter 
	 * @param mime the mime type of allowed files
	 * @param required true if the parameter is required to continue
	 */
	public FileParameter(String name, String comment, String mime, boolean required) {
		this.name = name;
		this.comment = comment;
		this.mime = mime;
		this.required = required;
	}
	
	/**
	 * Constructs a file parameter.
	 * @param name the name of the parameter
	 * @param comment a comment on the parameter
	 * @param mime the mime-type of allowed files
	 * @param required true if parameter is required
	 * @param validator a validator that validates e.g. the contents of the file
	 */
	public FileParameter(String name, String comment, String mime, boolean required, ParameterValidator validator){
		this(name,comment,mime,required);
		this.valid = validator;
	}


	public String getName() {
		return name;
	}
	
	public boolean isAtomic(){
		return true;
	}

	public DataType getDatatype() {
		return DataType.FILE;
	}

	public String getComment() {
		return comment;
	}

	public boolean isRequired() {
		return required;
	}
	
	/**
	 * Resets the file parameter to its original state.
	 *
	 */
	public void reset(){
		this.value = null;
		this.isSet = false;
		this.errorMessage = null;
	}
	
	/**
	 * Returns the content of the file
	 * @return the content
	 */
	public FileRepresentation getFileContents(){
		return value;
	}
	
	public String getErrorMessage(){
		return errorMessage;
	}

	public boolean checkValue(Object value) {
		if(valid != null){
			if(valid.checkValue(value)){
				errorMessage = null;
				return true;
			}else{
				errorMessage = valid.getErrorMessage();
				return false;
			}
		}else if(value != null && value instanceof FileRepresentation){
			FileRepresentation f = (FileRepresentation) value;
			if( f.getFilename() != null &&
					f.getFilename().length() != 0 &&
					f.getContent() != null &&
					f.getContent().length() != 0){
				errorMessage = null;
				return true;
			}else{
				errorMessage = "No file specified or file is empty.";
				return false;
			}
		}else{
			errorMessage = "Value is no file or null.";
			return false;
		}
	}
	
	public void setDefault(Object defaultValue) throws IllegalValueException{
		if(checkValue(defaultValue)){
			this.defaultValue = (FileRepresentation) defaultValue;
			setValue(defaultValue);
		}else{
			throw new IllegalValueException(errorMessage);
		}
	}
	
	public void simplify(){
		
	}

	public void setValue(Object value) throws IllegalValueException {
		if(!checkValue(value)){
			throw new IllegalValueException(errorMessage);
		}
		this.value = (FileRepresentation) value;
		this.isSet = true;
	}

	public Object getValue() {
		if(value == null){
			return null;
		}else{
			return value.getFilename();
		}
	}

	public boolean hasDefaultOrIsSet(){
		return isSet();
	}
	
	public boolean isSet() {
		return isSet;
	}

	public StringBuffer toXML() {
		StringBuffer buf = super.toXML();
		XMLParser.addTags( buf, "superParameter" );
		XMLParser.appendStringWithTags(buf,name,"name");
		XMLParser.appendStringWithTags(buf,comment,"comment");
		XMLParser.appendStringWithTags(buf,mime,"mime");
		XMLParser.appendBooleanWithTags(buf,required,"required");
		XMLParser.appendBooleanWithTags(buf,isSet,"isSet");
		XMLParser.appendStringWithTags(buf,errorMessage,"errorMessage");
		if(value == null){
			XMLParser.appendStringWithTags(buf,"null","value");
		}else{
			XMLParser.appendStringWithTags(buf,value.toXML().toString(),"value");
		}
		if(valid == null){
			XMLParser.appendStringWithTags(buf,"null","validator");
		}else{
			StringBuffer buf2 = new StringBuffer();
			XMLParser.appendStringWithTags(buf2,valid.getClass().getName(),"className");
			buf2.append(valid.toXML());
			XMLParser.appendStringWithTags(buf,buf2.toString(),"validator");
		}
		XMLParser.addTags(buf,"fileParameter");
		
		return buf;
	}

	protected void fromXML(StringBuffer representation)
			throws NonParsableException {
		StringBuffer buf = XMLParser.extractForTag(representation,"fileParameter");
		super.fromXML( XMLParser.extractForTag( representation, "superParameter" ) );
		name = XMLParser.extractStringForTag(buf,"name");
		comment = XMLParser.extractStringForTag(buf,"comment");
		mime = XMLParser.extractStringForTag(buf,"mime");
		required = XMLParser.extractBooleanForTag(buf,"required");
		isSet = XMLParser.extractBooleanForTag(buf,"isSet");
		errorMessage = XMLParser.extractStringForTag(buf,"errorMessage");
        if(errorMessage != null && errorMessage.equals("null")){
            errorMessage = null;
        }
		String val = XMLParser.extractStringForTag(buf,"value");
		if(val.equals("null")){
			value = null;
		}else{
			value = new FileRepresentation( new StringBuffer(val) );
		}
		val = XMLParser.extractStringForTag(buf,"validator");
		if(val.equals("null")){
			valid = null;
		}else{
			StringBuffer buf2 = new StringBuffer(val);
			String className = XMLParser.extractStringForTag(buf2,"className");
			try{
				valid = (ParameterValidator) Class.forName(className).getConstructor(new Class[]{StringBuffer.class}).newInstance(buf2);
			}catch(Exception e){
				throw new NonParsableException(e.getMessage());
			}
		}

	}

	/**
	 * Returns the mime type of the allowed files
	 * @return the mime type
	 */
	public String getAcceptedMimeType() {
		return mime;
	}

	/**
	 * Class that represents a file
	 * @author Jan Grau
	 *
	 */
	public static class FileRepresentation implements Storable, Cloneable{
		
		/**
		 * The name of the file
		 */
		private String filename;
		/**
		 * The contents of the file
		 */
		private String content;
		
		/**
		 * Created a file representation out of the filename and the file's contents
		 * @param filename the filename
		 * @param content the contents
		 */
		public FileRepresentation(String filename, String content){
			this.filename = filename;
			this.content = content;
		}
		
		/**
		 * Restores the file representation from an XML representation
		 * @param buf the XML representation
		 * @throws NonParsableException if the representation could not be parsed, an Exception is thrown
		 */
		public FileRepresentation(StringBuffer buf) throws NonParsableException{
			fromXML(buf);
		}
		
		public FileRepresentation clone() throws CloneNotSupportedException{
			return (FileRepresentation) super.clone();
		}
		
		/**
		 * Returns the filename.
		 * @return the filename
		 */
		public String getFilename(){
			return filename;
		}
		
		/**
		 * Returns the content of the file
		 * @return the content
		 */
		public String getContent(){
			return content;
		}

		public StringBuffer toXML() {
			StringBuffer buf = new StringBuffer();
			XMLParser.appendStringWithTags(buf,filename,"filename");
			XMLParser.appendStringWithTags(buf,content,"content");
			XMLParser.addTags(buf,"fileRepresentation");
			
			return buf;
		}

		
		private void fromXML(StringBuffer representation) throws NonParsableException {
			representation = XMLParser.extractForTag(representation,"fileRepresentation");
			filename = XMLParser.extractStringForTag(representation,"filename");
			content = XMLParser.extractStringForTag(representation,"content");
		}
		
		
		
	}
	
}
