/*
 * This file is part of Jstacs.
 * 
 * Jstacs is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * Jstacs is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * Jstacs. If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.models.mixture.gibbssampling;

import java.io.IOException;

import de.jstacs.data.Sample;

/**
 * This is the interface that any {@link de.jstacs.models.AbstractModel} has to implement if it
 * should be used in a Gibbs Sampling.
 * 
 * <br>
 * <br>
 * 
 * Before starting a series of samplings the method
 * {@link GibbsSamplingComponent#initModelForSampling(int)} is called. <br>
 * Then before starting a specific sampling the method
 * {@link GibbsSamplingComponent#extendSampling(int, boolean)} is called. <br>
 * In the sampling the method
 * {@link GibbsSamplingComponent#drawParameters(Sample, double[] )} is used for
 * drawing the parameters from the posterior (and writing them to a file). <br>
 * After finishing the sampling the method
 * {@link GibbsSamplingComponent#samplingStopped()} is called. <br>
 * After a sampling the methods
 * {@link GibbsSamplingComponent#parseParameterSet(int, int)} and
 * {@link GibbsSamplingComponent#parseNextParameterSet()} will be used for
 * computing for instance the (log-) likelihoods. <br>
 * 
 * <br>
 * <br>
 * 
 * The method {@link GibbsSamplingComponent#isInSamplingMode()} can be used to
 * check whether an object is currently used for sampling. If the object is in
 * sampling mode, it should not support any other method for changing the
 * parameters than
 * {@link GibbsSamplingComponent#drawParameters(Sample, double[] )} and
 * {@link GibbsSamplingComponent#parseParameterSet(int, int)}. Furthermore it is
 * legal to throw an {@link Exception} when the object is in sampling mode and a
 * method for saving, cloning or training the parameters is called.
 * 
 * @author Berit Haldemann, Jens Keilwagen
 * 
 * @see de.jstacs.models.AbstractModel
 * @see de.jstacs.models.mixture.AbstractMixtureModel
 * @see de.jstacs.models.mixture.AbstractMixtureModel.Algorithm#GIBBS_SAMPLING
 */
public interface GibbsSamplingComponent {

	/**
	 * This method allows the user to parse the set of parameters with index
	 * <code>n</code> of a certain <code>sampling</code> (from a file). The
	 * internal numbering should start with 0. The parameter set with index 0 is
	 * the initial (random) parameter set. It is recommended that a series of
	 * parameter sets is accessed by the following lines:
	 * 
	 * <p>
	 * <code>
	 * for( sampling = 0; sampling < numSampling; sampling++ )<br>
	 * {
	 * <dir>
	 *     boolean b = parseParameterSet( sampling, n );<br>
	 *     while( b )<br>
	 *     {<br>
	 *        //do something<br>
	 *        b = parseNextParameterSet();<br>
	 *     }
	 * </dir>
	 * }<br>
	 * </code>
	 * </p>
	 * 
	 * @param sampling
	 *            the index of the sampling
	 * @param n
	 *            the index of the parameter set
	 * 
	 * @return <code>true</code> if the parameter set could be parsed
	 * 
	 * @throws Exception
	 *             if there is a problem with parsing the parameters
	 * 
	 * @see GibbsSamplingComponent#parseNextParameterSet()
	 */
	public boolean parseParameterSet( int sampling, int n ) throws Exception;

	/**
	 * This method allows the user to parse the next set of parameters (from a
	 * file).
	 * 
	 * @return <code>true</code> if the parameters could be parsed, otherwise
	 *         <code>false</code>
	 * 
	 * @see GibbsSamplingComponent#parseParameterSet(int, int)
	 */
	public boolean parseNextParameterSet();

	/**
	 * This method initializes the model for the sampling. For instance this
	 * method can be used to create new files where all parameter sets are
	 * stored.
	 * 
	 * @param starts
	 *            the number of different sampling starts that will be done
	 * 
	 * @throws IOException
	 *             if something went wrong
	 * 
	 * @see java.io.File#createTempFile(String, String, java.io.File )
	 */
	public void initModelForSampling( int starts ) throws IOException;

	/**
	 * This method allows to extend a sampling.
	 * 
	 * @param sampling
	 *            the index of the sampling
	 * @param append
	 *            whether to append the sampled parameters to an existing file
	 *            or to overwrite the file
	 * 
	 * @throws IOException
	 *             if the file could not be handled correctly
	 */
	public void extendSampling( int sampling, boolean append ) throws IOException;

	/**
	 * This method is the opposite of the method
	 * {@link GibbsSamplingComponent#extendSampling(int, boolean)}. It can be
	 * used for closing any streams of writer, ...
	 * 
	 * @throws IOException
	 *             if something went wrong
	 * 
	 * @see GibbsSamplingComponent#extendSampling(int, boolean)
	 */
	public void samplingStopped() throws IOException;

	/**
	 * This method draws the parameters of the model from the a posteriori
	 * density. It is recommended that the parameters are written to a specific
	 * file so that they can later be parsed using the methods of the interface.
	 * 
	 * <br>
	 * <br>
	 * 
	 * Before using this method the method
	 * {@link GibbsSamplingComponent#initModelForSampling(int)} should be
	 * called.
	 * 
	 * @param data
	 *            a sample
	 * @param weights
	 *            the (non-negative) weights for each sequence of the sample
	 * 
	 * @throws Exception
	 *             if there is a problem with drawing the parameters, the model
	 *             is not initialized, ...
	 * 
	 * @see GibbsSamplingComponent#initModelForSampling(int)
	 * @see GibbsSamplingComponent#parseParameterSet(int, int)
	 * @see GibbsSamplingComponent#parseNextParameterSet()
	 */
	public void drawParameters( Sample data, double[] weights ) throws Exception;

	/**
	 * This method returns <code>true</code> if the object is currently used in
	 * a sampling, otherwise <code>false</code>.
	 * 
	 * @return <code>true</code> if the object is currently used in a sampling,
	 *         otherwise <code>false</code>
	 */
	public boolean isInSamplingMode();
}
