package de.jstacs.motifDiscovery;

import java.io.OutputStream;
import java.util.Arrays;

import de.jstacs.algorithms.optimization.NegativeDifferentiableFunction;
import de.jstacs.algorithms.optimization.Optimizer;
import de.jstacs.algorithms.optimization.StartDistanceForecaster;
import de.jstacs.algorithms.optimization.Optimizer.TerminationCondition;
import de.jstacs.classifier.scoringFunctionBased.OptimizableFunction;
import de.jstacs.classifier.scoringFunctionBased.OptimizableFunction.KindOfParameter;
import de.jstacs.data.DiscreteSequenceEnumerator;
import de.jstacs.data.EmptySampleException;
import de.jstacs.data.Sample;
import de.jstacs.data.Sequence;
import de.jstacs.io.ArrayHandler;
import de.jstacs.motifDiscovery.history.History;
import de.jstacs.scoringFunctions.NormalizableScoringFunction;
import de.jstacs.scoringFunctions.ScoringFunction;
import de.jstacs.scoringFunctions.mix.motifSearch.HiddenMotifsMixture;
import de.jstacs.utils.ComparableElement;
import de.jstacs.utils.SafeOutputStream;

/**
 * This class contains some important methods for the initiation and optimization of {@link MutableMotifDiscoverer}.
 * 
 * @author Jan Grau, Jens Keilwagen
 */
public final class MutableMotifDiscovererToolbox extends MotifDiscovererToolBox {
	
	/**
	 * This method allows to enumerate all possible seeds for a motif in the {@link HiddenMotifsMixture} of a specific class.
	 * 
	 * @param data the data
	 * @param funs the {@link ScoringFunction}s
	 * @param classIndex the index of the class 
	 * @param motifIndex the index of the motif in the {@link HiddenMotifsMixture}
	 * @param weight the weight of the seed {@link Sequence}
	 * @param opt the objective function
	 * @param out a stream that allows to write some output if necessary
	 * 
	 * @return the best {@link Sequence} with respect to the {@link OptimizableFunction} 
	 * 
	 * @throws Exception if something went wrong 
	 */
	public static Sequence enumerate( Sample[] data, ScoringFunction[] funs, int classIndex, int motifIndex, double weight, OptimizableFunction opt, OutputStream out ) throws Exception
	{
		HiddenMotifsMixture hmm = (HiddenMotifsMixture) funs[classIndex];
		int len = hmm.getMotifLength( motifIndex );
		DiscreteSequenceEnumerator si = new DiscreteSequenceEnumerator( funs[classIndex].getAlphabetContainer(), len, true );
		Sequence seq, bestSeq = null;
		double curr, best = Double.NEGATIVE_INFINITY;
		double[] pars, weights = {weight};
		Sample s = null;
		while( si.hasMoreElements() )
		{
			seq = si.nextElement();
			try {
				s = new Sample( null, seq );
			} catch ( IllegalArgumentException doesNotHappen ) {
			} catch ( EmptySampleException doesNotHappen ) {
			}
			
			hmm.initializeHiddenUniformly();
			
			hmm.initializeMotif( motifIndex, s, weights );
			hmm.adjustHiddenParameters( data[classIndex], null );
			
			opt.reset(funs);
			pars = opt.getParameters( KindOfParameter.PLUGIN );
			
			curr = opt.evaluateFunction(pars);
			out.write( (seq + "\t" + curr + "\n").getBytes() );
			
			if( curr > best ) {
				best = curr;
				bestSeq = seq;
			}
		}
		out.write( ( "best: " + bestSeq + " " + best + "\n" ).getBytes() );
		hmm.initializeHiddenUniformly();
		hmm.initializeMotif(motifIndex, new Sample( null, bestSeq ), weights );
		hmm.adjustHiddenParameters( data[classIndex], null );
		return bestSeq;
	}
	
	/**
	 * This enum defines some constants for the method {@link MutableMotifDiscovererToolbox#getSortedInitialParameters(Sample[], ScoringFunction[], InitMethodForScoringFunction[], OptimizableFunction, int, SafeOutputStream)}.
	 * These constants define how to initialize the {@link ScoringFunction}s. 
	 * 
	 * @author Jens Keilwagen
	 */
	public static enum InitMethodForScoringFunction {
		/**
		 * This constants indicates that the {@link ScoringFunction} should be initialized using {@link ScoringFunction#initializeFunction(int, boolean, Sample[], double[][])}.
		 */
		PLUG_IN,
		/**
		 * This constants indicates that the {@link ScoringFunction} should be initialized using {@link ScoringFunction#initializeFunctionRandomly(boolean)}.
		 */
		RANDOMLY,
		/**
		 * This constants indicates that the {@link ScoringFunction} should not be initialized, i.e. the instance is not changed and uses the current parameters.
		 */
		NOTHING;
	}
	
	/**
	 * This method allows to initialize the {@link ScoringFunction} using different {@link InitMethodForScoringFunction}. It returns an array of {@link ComparableElement}s that contain the parameters and the
	 * 
	 * @param data the data
	 * @param funs the {@link ScoringFunction}
	 * @param init the specific {@link InitMethodForScoringFunction}, the entries correspond one to one to those of <code>fun</code>
	 * @param opt the objective function
	 * @param n the number of initializations
	 * @param stream a stream that allows to write some output if necessary
	 * 
	 * @return a sorted array containing {@link ComparableElement}s of parameter arrays and corresponding values of the {@link OptimizableFunction}
	 * 
	 * @throws Exception if something went wrong
	 */
	@SuppressWarnings("unchecked")
	public static ComparableElement<double[],Double>[] getSortedInitialParameters( Sample[] data, ScoringFunction[] funs, InitMethodForScoringFunction[] init, OptimizableFunction opt, int n, SafeOutputStream stream ) throws Exception
	{
		ComparableElement<double[],Double>[] erg = new ComparableElement[n];
		double[] params;
		double c;
		for( int j, i = 0; i < n; i++ )
		{
			for( j = 0; j < funs.length; j++ )
			{
				switch( init[j] )
				{
					case PLUG_IN: 
						funs[j].initializeFunction( j, false, data, null );
						break;
					case RANDOMLY: 
						funs[j].initializeFunctionRandomly( false );
						break;
					case NOTHING:
						//leave the parameters as they are
						break;
				}
				
			}
			opt.reset( funs );
			params = opt.getParameters( KindOfParameter.PLUGIN );
			c = opt.evaluateFunction( params );
			stream.writeln( i + "\t" + c );
			erg[i] = new ComparableElement<double[],Double>( params, c );			
		}
		Arrays.sort( erg );
		stream.writeln( "[" + erg[0].getWeight() + " .. " + erg[n-1].getWeight() + "]" );
		return erg;
	}
	
	/**
	 * This method creates a minimalNewLength-array that can be used in an optimization.
	 * 
	 * @param funs the ScoringFunctions used in an optimization
	 * 
	 * @return an minimalNewLength-array for the given ScoringFunctions
	 */
	public static int[][] createMinimalNewLengthArray( ScoringFunction[] funs ) {
		int[][] minimalNewLength = new int[funs.length][];
		MutableMotifDiscoverer disc;
		for( int i, j = 0; j < funs.length; j++ )
		{
			if( funs[j] instanceof MutableMotifDiscoverer )
			{
				disc = (MutableMotifDiscoverer)funs[j];
				minimalNewLength[j] = new int[disc.getNumberOfMotifs()];
				for( i = 0; i < minimalNewLength[j].length; i++ )
				{
					minimalNewLength[j][i] = disc.getMotifLength( i );
				}
			}
		}
		return minimalNewLength;
	}
	
	/**
	 * This method creates a History-array that can be used in an optimization.
	 * 
	 * @param funs the ScoringFunctions used in an optimization
	 * @param template the template history instance
	 * 
	 * @return an History-array for the given ScoringFunctions
	 * 
	 * @throws CloneNotSupportedException if the t<code>template</code> could not be cloned
	 */
	public static History[][] createHistoryArray( ScoringFunction[] funs, History template ) throws CloneNotSupportedException{
		History[][] history = new History[funs.length][];
		for( int i, j = 0; j < funs.length; j++ )
		{
			if( funs[j] instanceof MutableMotifDiscoverer )
			{
				history[j] = new History[((MutableMotifDiscoverer)funs[j]).getNumberOfMotifs()];
				for( i = 0; i < history[j].length; i++ )
				{
					history[j][i] = template.clone();
				}
			}
		}
		return history;
	}
	
	/**
	 * This method clears all elements of an History-array, so that it can be used again.
	 * 
	 * @param history the array
	 */
	public static void clearHistoryArray( History[][] history ) {
		for( int i, j = 0; j < history.length; j++ )
		{
			if( history[j] != null )
			{
				for( i = 0; i < history[j].length; i++ )
				{
					history[j][i].clear();
				}
			}
		}
	}
	
	/**
	 * This method tries to optimize the problem at hand as good as possible. If the optimization uses {@link MutableMotifDiscoverer}s it tries to perform modify operations as long as they seem to be promising.
	 * 
	 * @param funs the {@link ScoringFunction}
	 * @param opt the {@link OptimizableFunction}
	 * @param algorithm used for the optimization
	 * @param eps used for the optimization
	 * @param linEps used for the optimization
	 * @param startDistance used for the optimization
	 * @param out an stream that allows to obtain some information while optimization
	 * @param breakOnChanged a switch that decides whether a new optimization should be started after one successful modify or after all motifs have been tried to modify.
	 * @param template a history instance used to build an array with this instance
	 * @param plugIn a switch whether to take the internal parameters or not
	 * 
	 * @return the optimized value (res[0][0]) and the array for the class parameters (res[1])
	 * 
	 * @throws Exception if something went wrong while optimization
	 *
	 * @see MutableMotifDiscovererToolbox#clearHistoryArray(de.jstacs.motifDiscovery.history.History[][])
	 * @see MutableMotifDiscovererToolbox#optimize(de.jstacs.scoringFunctions.ScoringFunction[], de.jstacs.classifier.scoringFunctionBased.OptimizableFunction, byte, double, double, de.jstacs.algorithms.optimization.StartDistanceForecaster, de.jstacs.utils.SafeOutputStream, boolean, de.jstacs.motifDiscovery.history.History[][], int[][], de.jstacs.classifier.scoringFunctionBased.OptimizableFunction.KindOfParameter)
	 */
	public static double[][] optimize( ScoringFunction[] funs, OptimizableFunction opt, byte algorithm, double eps, double linEps, StartDistanceForecaster startDistance, SafeOutputStream out, boolean breakOnChanged, History template, KindOfParameter plugIn ) throws Exception {
		return optimize( funs, opt, algorithm, eps, linEps, startDistance, out, breakOnChanged, createHistoryArray( funs, template ), createMinimalNewLengthArray( funs ), plugIn );
	}
	
	/**
	 * This method tries to optimize the problem at hand as good as possible. If the optimization uses {@link MutableMotifDiscoverer}s it tries to perform modify operations as long as they seem to be promising.
	 * 
	 * @param funs the {@link ScoringFunction}
	 * @param opt the {@link OptimizableFunction}
	 * @param algorithm used for the optimization
	 * @param eps used for the optimization
	 * @param linEps used for the optimization
	 * @param startDistance used for the optimization
	 * @param out an stream that allows to obtain some information while optimization
	 * @param breakOnChanged a switch that decides whether a new optimization should be started after one successful modify or after all motifs have been tried to modify.
	 * @param hist an array that is used to check whether a modify-operation can be performed
	 * @param minimalNewLength the minimal new length for each motif in each class, that will be used in an expand if the motif was shortened before
	 * @param plugIn a switch whether to take the internal parameters or not
	 * 
	 * @return the optimized value (res[0][0]) and the array for the class parameters (res[1])
	 * 
	 * @throws Exception if something went wrong while optimization
	 */
	public static double[][] optimize( ScoringFunction[] funs, OptimizableFunction opt, byte algorithm, double eps, double linEps, StartDistanceForecaster startDistance, SafeOutputStream out, boolean breakOnChanged, History[][] hist, int[][] minimalNewLength, KindOfParameter plugIn ) throws Exception {
		NegativeDifferentiableFunction neg = new NegativeDifferentiableFunction(opt);
		int k;
		ScoringFunction[] best = null;
		double[] params, classParams = null;
		Sample[] data = opt.getData();
		double[][] weights = opt.getSequenceWeights();
		double bestVal = Double.NEGATIVE_INFINITY, current;
		
		do{
			opt.reset( funs );
			startDistance.reset();
			params = opt.getParameters( plugIn );
			plugIn = KindOfParameter.LAST;
			Optimizer.optimize( algorithm, neg, params, TerminationCondition.SMALL_DIFFERENCE_OF_FUNCTION_EVALUATIONS, eps, linEps, startDistance, out );
			current = opt.evaluateFunction( params );
			if( current > bestVal )
			{
				best = null;
				System.gc();
				best = ArrayHandler.clone( funs );
				bestVal = current;
				classParams = opt.getClassParams(params);
			}
 				
		}while( doHeuristicSteps( funs, data, weights, opt, out, breakOnChanged, hist, minimalNewLength ) );
		for( k = 0; k < funs.length; k++ )
		{
			funs[k] = best[k];
		}
		return new double[][]{{bestVal}, classParams};
	}
		
	/**
	 * This method tries to make some heuristic step if at least one {@link ScoringFunction} is a {@link MutableMotifDiscoverer}.
	 * These heuristic steps include shift, shrink, and expand as far as the user allows those operations by the {@link History} array.
	 * 
	 * @param funs the {@link ScoringFunction}
	 * @param data array of {@link Sample} containing the data for each class
	 * @param weights the weights corresponding to the {@link Sequence}s in <code>data</code>
	 * @param opt the {@link OptimizableFunction}
	 * @param out an stream that allows to obtain some information while optimization
	 * @param breakOnChanged a switch that decides whether a new optimization should be started after one successful modify or after all motifs have been tried to modify.
	 * @param hist an array that is used to check whether a modify-operation can be performed
	 * @param minimalNewLength the minimal new length for each motif in each class, that will be used in an expand if the motif was shortened before
	 * 
	 * @return <code>true</code> if some heuristic steps has been performed otherwise <code>false</code>
	 * 
	 * @throws Exception if something went wrong
	 */
	public static boolean doHeuristicSteps( ScoringFunction[] funs, Sample[] data, double[][] weights, OptimizableFunction opt,  SafeOutputStream out, boolean breakOnChanged, History[][] hist, int[][] minimalNewLength ) throws Exception {
		boolean changed = false, changedThisOne;
		double normOld, normNew;
		NormalizableScoringFunction nsf;
		for( int k = 0; k < funs.length && ( !changed || !breakOnChanged ); k++ )
		{
			if( funs[k] instanceof MutableMotifDiscoverer )
			{
				out.writeln( "MutableMotifDiscoverer " + k + ":\n" + funs[k].toString() );
				if( funs[k] instanceof NormalizableScoringFunction ) {
					nsf = (NormalizableScoringFunction) funs[k];
					normOld = nsf.getNormalizationConstant();
				} else {
					nsf = null;
					normOld = -1;
				}
				MutableMotifDiscoverer currMD = (MutableMotifDiscoverer) funs[k];
				int numMotifs = currMD.getNumberOfMotifs();
				changedThisOne = false;
				for( int l = 0 ; l < numMotifs; l++ ){
					changedThisOne |= modify( currMD, k, l, data, weights, hist[k][l], minimalNewLength[k][l], out );
				}
				if( changedThisOne ) {
					changed = true;
					if( nsf != null ) {
						normNew = nsf.getNormalizationConstant();
						//set new class parameter
						opt.addTermToClassParameter( k, Math.log( normOld/normNew ) );
					}
				}
			}
		}
		return changed;
	}
	
	private static boolean modify( MutableMotifDiscoverer md, int clazz, int motif, Sample[] data, double[][] weights, History hist, int minimalNewLength, SafeOutputStream out ) throws Exception
	{
		boolean modified = false;
		
		int[] notSignif = md.determineNotSignificantPositionsFor( motif, data, weights, clazz );
		int sum = notSignif[0]+notSignif[1], ml = md.getMotifLength( motif );

		int[] modification = new int[2];
		
		if( sum > 0 ) { // at least one side is not significant
			if( sum < ml ) {
				//try to shift from the side with the most not significant positions
				if( notSignif[0] > notSignif[1] ) {
					// (more) not significant positions at the left side
					modification[0] = notSignif[0];
				} else {
					// (more) not significant positions at the right side
					modification[0] = -notSignif[1];
				}
				modification[1] = modification[0];
				if( hist.operationAllowed( modification ) ){
					modified = md.modifyMotif( motif, modification[0], modification[1] );
				}
			}
			
			if( !modified ) {
				//shrink
				if( sum < ml ) {
					// shrink is possible
					modification[0] = notSignif[0];
					modification[1] = -notSignif[1];
				} else {
					// shrink to size 1
					modification[0] = 0;
					modification[1] = 1-md.getMotifLength( motif );
				}
				if( modification[0] != modification[1] // modification [0,0] forbidden
				        && hist.operationAllowed( modification ) ){
					modified = md.modifyMotif( motif, modification[0], modification[1] );
				}
			}
		} else {
			//expand
			if( md.getMotifLength( motif ) < minimalNewLength ) {
				// expand to minimal length
				modification[0] = 0;
				modification[1] = minimalNewLength - md.getMotifLength( motif );
			} else {
				// expand 1 to the left and right
				modification[0] = -1;
				modification[1] = 1;
			}
			
			if( hist.operationAllowed( modification ) ){
				modified = md.modifyMotif( motif, modification[0], modification[1] );
			}
		}
		
		if( modified ){
			hist.operationPerfomed( modification );
			out.writeln( "class " + clazz + ": modified motif " + motif + " => " + Arrays.toString( modification ) );
		}
		return modified;
	}
}
