/*
 * This file is part of Jstacs.
 * 
 * Jstacs is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * Jstacs is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * Jstacs. If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.io;

import de.jstacs.NonParsableException;
import de.jstacs.Storable;

/**
 * Class for parsing standard data types and arrays in and out of an XML
 * {@link java.io.File}. The methods with prefix <code>append</code> or
 * <code>add</code> are for encoding, while methods with prefix
 * <code>extract</code> are for decoding.
 * 
 * @author Jan Grau, Jens Keilwagen
 */
public class XMLParser {

	/**
	 * Extracts the code between equal start and end tags and returns the
	 * substring between the tags as new {@link StringBuffer}.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the substring shall be taken (start tag
	 *            should be equal to end tag)
	 * 
	 * @return the substring between the tags as {@link StringBuffer}
	 * 
	 * @throws NonParsableException
	 *             if the substring could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, String)
	 */
	public static StringBuffer extractForTag( StringBuffer source, String tag ) throws NonParsableException {
		return extractForTag( source, tag, tag );
	}

	/**
	 * Extracts the code between start and end tag and returns the substring
	 * between the tags as new {@link StringBuffer}.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @return the substring between start and end tag as {@link StringBuffer}
	 * 
	 * @throws NonParsableException
	 *             if the substring could not be parsed
	 */
	public static StringBuffer extractForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException {
		// System.out.println(startTag);
		if( source == null || source.length() == 0 ) {
			return null;
		}
		int start = source.indexOf( "<" + startTag + ">" );
		if( start == -1 ) {
			return null;
		}
		start += startTag.length() + 2;
		int end = start;
		int pos = start;
		int open = 1;
		while( ( pos = source.indexOf( endTag, pos + 1 ) ) > 0 ) {
			if( source.charAt( pos - 1 ) == '/' && source.charAt( pos - 2 ) == '<' ) {
				open--;
			} else if( source.charAt( pos - 1 ) == '<' ) {
				open++;
			}
			// System.out.print(open+" ");
			if( open == 0 ) {
				end = pos - 2;
				break;
			}
		}
		if( pos == -1 ) {
			throw new NonParsableException( "Missing end tag for: " + endTag );
		}
		StringBuffer ex = new StringBuffer( source.substring( start, end ) );
		source.delete( start - startTag.length() - 2, end + endTag.length() + 3 );
		return ex;
	}

	/**
	 * Returns the value between equal start and end tags as
	 * <code>boolean</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the tags between which the value shall be taken (start tag
	 *            should be equal to end tag)
	 * 
	 * @return the value between the tags as <code>boolean</code>
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractBooleanForTag(StringBuffer, String, String)
	 */
	public static boolean extractBooleanForTag( StringBuffer source, String startTag ) throws NonParsableException {
		return extractBooleanForTag( source, startTag, startTag );
	}

	/**
	 * Returns the value between start and end tag as <code>boolean</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @return the value between start and end tag as <code>boolean</code>
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, String)
	 */
	public static boolean extractBooleanForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException {
		return Boolean.valueOf( extractForTag( source, startTag, endTag ).toString().trim() ).booleanValue();
	}

	/**
	 * Returns the value between equal start and end tags as <code>byte</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the tags between which the value shall be taken (start tag
	 *            should be equal to end tag)
	 * 
	 * @return the value between the tags as <code>byte</code>
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractByteForTag(StringBuffer, String, String)
	 */
	public static byte extractByteForTag( StringBuffer source, String startTag ) throws NonParsableException {
		return extractByteForTag( source, startTag, startTag );
	}

	/**
	 * Returns the value between start and end tag as <code>byte</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * @return the value between start and end tag as <code>byte</code>
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, String)
	 */
	public static byte extractByteForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, startTag, endTag );
		try {
			return Byte.parseByte( ex.toString() );
		} catch ( NumberFormatException e ) {
			throw new NonParsableException( "Could not parse \"" + startTag + "\" to int." );
		} catch ( NullPointerException e ) {
			throw new NonParsableException( "Could not find \"" + startTag + "\"." );
		}
	}

	/**
	 * Returns the enumeration between equal start and end tags as {@link Enum}.
	 * 
	 * @param <T>
	 *            the type of the {@link Enum} objects
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the enumeration shall be taken (start
	 *            tag should be equal to end tag)
	 * 
	 * @return the enumeration between the tags as {@link Enum}
	 * 
	 * @throws NonParsableException
	 *             if the enumeration could not be parsed
	 * 
	 * @see XMLParser#extractEnumForTag(StringBuffer, String, String)
	 */
	public static <T extends Enum<T>> T extractEnumForTag( StringBuffer source, String tag ) throws NonParsableException {
		return extractEnumForTag( source, tag, tag );
	}

	/**
	 * Returns the enumeration between start and end tag as {@link Enum}.
	 * 
	 * @param <T>
	 *            the type of the {@link Enum} objects
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @return the enumeration between start and end tag as {@link Enum}
	 * 
	 * @throws NonParsableException
	 *             if the enumeration could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, String)
	 */
	@SuppressWarnings( "unchecked" )
	public static <T extends Enum<T>> T extractEnumForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, startTag, endTag );
		if( ex == null ) {
			throw new NonParsableException( "Could not find \"" + startTag + "\"." );
		}
		String enumName = XMLParser.extractStringForTag( ex, "enumName" );
		T erg;
		try {
			erg = Enum.valueOf( (Class<T>)Class.forName( enumName ), XMLParser.extractStringForTag( ex, "name" ) );
		} catch ( Exception e ) {
			throw getNonParsableException( "problem at " + enumName + ": " + e.getClass().getSimpleName() + ": " + e.getCause().toString(),
					e );
		}
		return erg;
	}

	/**
	 * Returns the value between equal start and end tags as <code>int</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the tags between which the value shall be taken (start tag
	 *            should be equal to end tag)
	 * 
	 * @return the value between the tags as <code>int</code>
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractIntForTag(StringBuffer, String, String)
	 */
	public static int extractIntForTag( StringBuffer source, String startTag ) throws NonParsableException {
		return extractIntForTag( source, startTag, startTag );
	}

	/**
	 * Returns the value between start and end tag as <code>int</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @return the value between start and end tag as <code>int</code>
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, String)
	 */
	public static int extractIntForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, startTag, endTag );
		try {
			return Integer.parseInt( ex.toString() );
		} catch ( NumberFormatException e ) {
			throw new NonParsableException( "Could not parse \"" + startTag + "\" to int." );
		} catch ( NullPointerException e ) {
			throw new NonParsableException( "Could not find \"" + startTag + "\"." );
		}
	}

	/**
	 * Returns the value between equal start and end tags as <code>long</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the tags between which the value shall be taken (start tag
	 *            should be equal to end tag)
	 * 
	 * @return the value between the tags as <code>long</code>
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractLongForTag(StringBuffer, String, String)
	 */
	public static long extractLongForTag( StringBuffer source, String startTag ) throws NonParsableException {
		return extractLongForTag( source, startTag, startTag );
	}

	/**
	 * Returns the value between start and end tag as <code>long</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @return the value between start and end tag as <code>long</code>
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, String)
	 */
	public static long extractLongForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, startTag, endTag );
		try {
			return Long.parseLong( ex.toString() );
		} catch ( NumberFormatException e ) {
			throw new NonParsableException( "Could not parse \"" + startTag + "\" to int." );
		} catch ( NullPointerException e ) {
			throw new NonParsableException( "Could not find \"" + startTag + "\"." );
		}
	}

	/**
	 * Returns the value between equal start and end tags as <code>double</code>
	 * .
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the tags between which the value shall be taken (start tag
	 *            should be equal to end tag)
	 * 
	 * @return the value between the tags as <code>double</code>
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractDoubleForTag(StringBuffer, String, String)
	 */
	public static double extractDoubleForTag( StringBuffer source, String startTag ) throws NonParsableException {
		return extractDoubleForTag( source, startTag, startTag );
	}

	/**
	 * Returns the value between start and end tag as <code>double</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @return the value between start and end tag as <code>double</code>
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, String)
	 */
	public static double extractDoubleForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, startTag, endTag );
		try {
			return Double.parseDouble( ex.toString() );
		} catch ( NumberFormatException e ) {
			throw new NonParsableException( "Could not parse \"" + startTag + "\" to double." );
		} catch ( NullPointerException e ) {
			throw new NonParsableException( "Could not find \"" + startTag + "\"." );
		}
	}

	/**
	 * Returns the value between equal start and end tags as {@link Storable}.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the tags between which the value shall be taken (start tag
	 *            should be equal to end tag)
	 * 
	 * @return the value between the tags as {@link Storable}
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractStorableForTag(StringBuffer, String, String)
	 */
	public static Storable extractStorableForTag( StringBuffer source, String startTag ) throws NonParsableException {
		return extractStorableForTag( source, startTag, startTag );
	}

	/**
	 * Returns the value between start and end tag as {@link Storable}.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @return the value between start and end tag as {@link Storable}
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, String)
	 */
	public static Storable extractStorableForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, startTag, endTag );
		if( ex == null ) {
			throw new NonParsableException( "Could not find \"" + startTag + "\"." );
		}
		String className = XMLParser.extractStringForTag( ex, "className" );
		Storable erg;
		try {
			erg = (Storable)Class.forName( className ).getConstructor( new Class[]{ StringBuffer.class } ).newInstance( ex );
		} catch ( NoSuchMethodException e ) {
			throw getNonParsableException( "You must provide a constructor " + className + "(StringBuffer).", e );
		} catch ( Exception e ) {
			throw getNonParsableException( "problem at " + className + ": " + e.getClass().getSimpleName() + ": " + e.getCause().toString(),
					e );
		}
		return erg;
	}

	/**
	 * Returns the value between equal start and end tags as {@link Storable} or
	 * <code>null</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken (start tag
	 *            should be equal to end tag)
	 * 
	 * @return the value between the tags as {@link Storable} or
	 *         <code>null</code>
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractStorableOrNullForTag(StringBuffer, String, String)
	 */
	public static Storable extractStorableOrNullForTag( StringBuffer source, String tag ) throws NonParsableException {
		return extractStorableOrNullForTag( source, tag, tag );
	}

	/**
	 * Returns the value between start and end tag as {@link Storable} or
	 * <code>null</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @return the value between start and end tag as {@link Storable} or
	 *         <code>null</code>
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, String)
	 */
	public static Storable extractStorableOrNullForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, startTag, endTag );
		if( ex == null ) {
			throw new NonParsableException( "Could not find \"" + startTag + "\"." );
		}
		if( ex.toString().trim().equals( "null" ) ) {
			return null;
		}
		String className = XMLParser.extractStringForTag( ex, "className" );

		Storable erg;
		try {
			erg = (Storable)Class.forName( className ).getConstructor( new Class[]{ StringBuffer.class } ).newInstance( ex );
		} catch ( NoSuchMethodException e ) {
			throw getNonParsableException( "You must provide a constructor " + className + "(StringBuffer).", e );
		} catch ( Exception e ) {
			throw getNonParsableException( "problem at " + className + ": " + e.getClass().getSimpleName() + ": " + e.getCause().toString(),
					e );
		}
		return erg;
	}

	/**
	 * The method returns a {@link NonParsableException} with a given error
	 * message for an {@link Exception}.
	 * 
	 * @param s
	 *            the error message of the {@link NonParsableException}
	 * @param e
	 *            the {@link Exception}
	 * 
	 * @return a {@link NonParsableException} with given error message
	 */
	private static final NonParsableException getNonParsableException( String s, Exception e ) {
		NonParsableException n = new NonParsableException( s );
		n.setStackTrace( e.getStackTrace() );
		return n;
	}

	/**
	 * Returns the value between equal start and end tags as a {@link String}.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the tags between which the value shall be taken (start tag
	 *            should be equal to end tag)
	 * 
	 * @return the value between the tags as a {@link String}
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractStringForTag(StringBuffer, String, String)
	 */
	public static String extractStringForTag( StringBuffer source, String startTag ) throws NonParsableException {
		return extractStringForTag( source, startTag, startTag );
	}

	/**
	 * Returns the value between start and end tag as a {@link String}.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @return the value between start and end tag as a {@link String}
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, String)
	 */
	public static String extractStringForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, startTag, endTag );
		if( ex == null ) {
			throw new NonParsableException( "Could not find \"" + startTag + "\"." );
		}
		return ex.toString();
	}

	/**
	 * Returns the value between equal start and end tags as a
	 * <code>boolean</code> array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken (start tag
	 *            should be equal to end tag)
	 * 
	 * @return the value between the tags as a <code>boolean</code> array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractBooleanArrayForTag(StringBuffer, String, String)
	 */
	public static boolean[] extractBooleanArrayForTag( StringBuffer source, String tag ) throws NonParsableException {
		return extractBooleanArrayForTag( source, tag, tag );
	}

	/**
	 * Returns the value between start and end tag as a <code>boolean</code>
	 * array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @return the value between start and end tag as a <code>boolean</code>
	 *         array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, String)
	 * @see XMLParser#extractBooleanForTag(StringBuffer, String, String)
	 */
	public static boolean[] extractBooleanArrayForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, startTag, endTag );
		int l = extractIntForTag( ex, "length" );
		boolean[] b = new boolean[l];
		for( int i = 0; i < l; i++ ) {
			b[i] = extractBooleanForTag( ex, "pos val=\"" + i + "\"", "pos" );
		}
		return b;
	}

	/**
	 * Returns the value between equal start and end tags as a <code>byte</code>
	 * array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the tags between which the value shall be taken (start tag
	 *            should be equal to end tag)
	 * 
	 * @return the value between the tags as a <code>byte</code> array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractByteArrayForTag(StringBuffer, String, String)
	 */
	public static byte[] extractByteArrayForTag( StringBuffer source, String startTag ) throws NonParsableException {
		return extractByteArrayForTag( source, startTag, startTag );
	}

	/**
	 * Returns the value between start and end tag as a <code>byte</code> array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @return the value between start and end tag as a <code>byte</code> array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, String)
	 * @see XMLParser#extractByteForTag(StringBuffer, String, String)
	 */
	public static byte[] extractByteArrayForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, startTag, endTag );
		int l = extractIntForTag( ex, "length" );
		byte[] array = new byte[l];
		for( int i = 0; i < l; i++ ) {
			array[i] = extractByteForTag( ex, "pos val=\"" + i + "\"", "pos" );
		}
		return array;
	}

	/**
	 * Returns the value between equal start and end tags as an <code>int</code>
	 * array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the tags between which the value shall be taken (start tag
	 *            should be equal to end tag)
	 * 
	 * @return the value between the tags as an <code>int</code> array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractIntArrayForTag(StringBuffer, String, String)
	 */
	public static int[] extractIntArrayForTag( StringBuffer source, String startTag ) throws NonParsableException {
		return extractIntArrayForTag( source, startTag, startTag );
	}

	/**
	 * Returns the value between start and end tag as an <code>int</code> array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer}
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @return the value between start and end tag as an <code>int</code> array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, String)
	 * @see XMLParser#extractIntForTag(StringBuffer, String, String)
	 */
	public static int[] extractIntArrayForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, startTag, endTag );
		int l = extractIntForTag( ex, "length" );
		int[] array = new int[l];
		for( int i = 0; i < l; i++ ) {
			array[i] = extractIntForTag( ex, "pos val=\"" + i + "\"", "pos" );
		}
		return array;
	}

	/**
	 * Returns the value between equal start and end tags as a
	 * <code>double</code> array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the tags between which the value shall be taken (start tag
	 *            should be equal to end tag)
	 * 
	 * @return the value between the tags as a <code>double</code> array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractDoubleArrayForTag(StringBuffer, String, String)
	 */
	public static double[] extractDoubleArrayForTag( StringBuffer source, String startTag ) throws NonParsableException {
		return extractDoubleArrayForTag( source, startTag, startTag );
	}

	/**
	 * Returns the value between start and end tag as a <code>double</code>
	 * array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @return the value between start and end tag as a <code>double</code>
	 *         array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, String)
	 * @see XMLParser#extractDoubleForTag(StringBuffer, String, String)
	 */
	public static double[] extractDoubleArrayForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, startTag, endTag );
		int l = extractIntForTag( ex, "length" );
		double[] array = new double[l];
		for( int i = 0; i < array.length; i++ ) {
			array[i] = extractDoubleForTag( ex, "pos val=\"" + i + "\"", "pos" );
		}
		return array;
	}

	/**
	 * Returns the value between equal start and end tags as a {@link Storable}
	 * array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the tags between which the value shall be taken (start tag
	 *            should be equal to end tag)
	 * 
	 * @return the value between the tags as a {@link Storable} array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractStorableArrayForTag(StringBuffer, String, String)
	 * @see ArrayHandler#cast(Object...)
	 */
	public static Storable[] extractStorableArrayForTag( StringBuffer source, String startTag ) throws NonParsableException {
		return extractStorableArrayForTag( source, startTag, startTag );
	}

	/**
	 * Returns the value between start and end tag as a {@link Storable} array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @return the value between start and end tag as a {@link Storable} array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, String)
	 * @see XMLParser#extractStorableForTag(StringBuffer, String, String)
	 * @see ArrayHandler#cast(Object...)
	 */
	public static Storable[] extractStorableArrayForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, startTag, endTag );
		int l = extractIntForTag( ex, "length" );
		Storable[] array = new Storable[l];
		for( int i = 0; i < array.length; i++ ) {
			array[i] = extractStorableForTag( ex, "pos val=\"" + i + "\"", "pos" );
		}
		return array;
	}

	/**
	 * Returns the value between equal start and end tags as a {@link String}
	 * array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the tags between which the value shall be taken (start tag
	 *            should be equal to end tag)
	 * 
	 * @return the value between the tags as a {@link String} array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractStringArrayForTag(StringBuffer, String, String)
	 */
	public static String[] extractStringArrayForTag( StringBuffer source, String startTag ) throws NonParsableException {
		return extractStringArrayForTag( source, startTag, startTag );
	}

	/**
	 * Returns the value between start and end tag as a {@link String} array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @return the value between start and end tag as a {@link String} array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, String)
	 * @see XMLParser#extractStringForTag(StringBuffer, String, String)
	 */
	public static String[] extractStringArrayForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, startTag, endTag );
		int l = extractIntForTag( ex, "length" );
		String[] array = new String[l];
		for( int i = 0; i < array.length; i++ ) {
			array[i] = extractStringForTag( ex, "pos val=\"" + i + "\"", "pos" );
		}
		return array;
	}

	/**
	 * Returns the value between equal start and end tags as a two dimensional
	 * {@link Storable} array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken (start tag
	 *            should be equal to end tag)
	 * 
	 * @return the value between the tags as a two dimensional {@link Storable}
	 *         array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractStorable2ArrayForTag(StringBuffer, String, String)
	 */
	public static Storable[][] extractStorable2ArrayForTag( StringBuffer source, String tag ) throws NonParsableException {
		return extractStorable2ArrayForTag( source, tag, tag );
	}

	/**
	 * Returns the value between start and end tag as a two dimensional
	 * {@link Storable} array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @return the value between start and end tag as a two dimensional
	 *         {@link Storable} array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, String)
	 * @see XMLParser#extractStorableArrayForTag(StringBuffer, String, String)
	 */
	public static Storable[][] extractStorable2ArrayForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, startTag, endTag );
		int l = extractIntForTag( ex, "length" );
		Storable[][] array = new Storable[l][];
		for( int i = 0; i < l; i++ ) {
			array[i] = extractStorableArrayForTag( ex, "pos val=\"" + i + "\"", "pos" );
		}
		return array;
	}

	/**
	 * Returns the value between equal start and end tags as a three dimensional
	 * {@link Storable} array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken (start tag
	 *            should be equal to end tag)
	 * 
	 * @return the value between the tags as a three dimensional
	 *         {@link Storable} array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractStorable3ArrayForTag(StringBuffer, String, String)
	 */
	public static Storable[][][] extractStorable3ArrayForTag( StringBuffer source, String tag ) throws NonParsableException {
		return extractStorable3ArrayForTag( source, tag, tag );
	}

	/**
	 * Returns the value between start and end tag as a three dimensional
	 * {@link Storable} array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @return the value between start and end tag as a three dimensional
	 *         {@link Storable} array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, String)
	 * @see XMLParser#extractStorable2ArrayForTag(StringBuffer, String, String)
	 */
	public static Storable[][][] extractStorable3ArrayForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, startTag, endTag );
		int l = extractIntForTag( ex, "length" );
		Storable[][][] array = new Storable[l][][];
		for( int i = 0; i < l; i++ ) {
			array[i] = extractStorable2ArrayForTag( ex, "pos val=\"" + i + "\"", "pos" );
		}
		return array;
	}

	/**
	 * Returns the value between equal start and end tags as a two dimensional
	 * <code>boolean</code> array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken (start tag
	 *            should be equal to end tag)
	 * 
	 * @return the value between the tags as a two dimensional
	 *         <code>boolean</code> array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractBoolean2ArrayForTag(StringBuffer, String, String)
	 */
	public static boolean[][] extractBoolean2ArrayForTag( StringBuffer source, String tag ) throws NonParsableException {
		return extractBoolean2ArrayForTag( source, tag, tag );
	}

	/**
	 * Returns the value between start and end tag as a two dimensional
	 * <code>boolean</code> array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @return the value between start and end tag as a two dimensional
	 *         <code>boolean</code> array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, String)
	 * @see XMLParser#extractBooleanArrayForTag(StringBuffer, String, String)
	 */
	public static boolean[][] extractBoolean2ArrayForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, startTag, endTag );
		boolean[][] array = new boolean[extractIntForTag( ex, "length" )][];
		for( int i = 0; i < array.length; i++ ) {
			array[i] = extractBooleanArrayForTag( ex, "pos val=\"" + i + "\"", "pos" );
		}
		return array;
	}

	/**
	 * Returns the value between equal start and end tags as a two dimensional
	 * <code>byte</code> array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the tags between which the value shall be taken (start tag
	 *            should be equal to end tag)
	 * 
	 * @return the value between the tags as a two dimensional <code>byte</code>
	 *         array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractByte2ArrayForTag(StringBuffer, String, String)
	 */
	public static byte[][] extractByte2ArrayForTag( StringBuffer source, String startTag ) throws NonParsableException {
		return extractByte2ArrayForTag( source, startTag, startTag );
	}

	/**
	 * Returns the value between start and end tag as a two dimensional
	 * <code>byte</code> array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @return the value between start and end tag as a two dimensional
	 *         <code>byte</code> array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, String)
	 * @see XMLParser#extractByteArrayForTag(StringBuffer, String, String)
	 */
	public static byte[][] extractByte2ArrayForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, startTag, endTag );
		byte[][] array = new byte[extractIntForTag( ex, "length" )][];
		for( int i = 0; i < array.length; i++ ) {
			array[i] = extractByteArrayForTag( ex, "pos val=\"" + i + "\"", "pos" );
		}
		return array;
	}

	/**
	 * Returns the value between equal start and end tags as a two dimensional
	 * <code>int</code> array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the tags between which the value shall be taken (start tag
	 *            should be equal to end tag)
	 * 
	 * @return the value between the tags as a two dimensional <code>int</code>
	 *         array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractInt2ArrayForTag(StringBuffer, String, String)
	 */
	public static int[][] extractInt2ArrayForTag( StringBuffer source, String startTag ) throws NonParsableException {
		return extractInt2ArrayForTag( source, startTag, startTag );
	}

	/**
	 * Returns the value between start and end tag as a two dimensional
	 * <code>int</code> array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @return the value between start and end tag as a two dimensional
	 *         <code>int</code> array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, String)
	 * @see XMLParser#extractIntArrayForTag(StringBuffer, String, String)
	 */
	public static int[][] extractInt2ArrayForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, startTag, endTag );
		int l = extractIntForTag( ex, "length" );
		int[][] array = new int[l][];
		for( int i = 0; i < l; i++ ) {
			array[i] = extractIntArrayForTag( ex, "pos val=\"" + i + "\"", "pos" );
		}
		return array;
	}

	/**
	 * Returns the value between equal start and end tags as a two dimensional
	 * <code>double</code> array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the tags between which the value shall be taken (start tag
	 *            should be equal to end tag)
	 * 
	 * @return the value between the tags as a two dimensional
	 *         <code>double</code> array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractDouble2ArrayForTag(StringBuffer, String, String)
	 */
	public static double[][] extractDouble2ArrayForTag( StringBuffer source, String startTag ) throws NonParsableException {
		return extractDouble2ArrayForTag( source, startTag, startTag );
	}

	/**
	 * Returns the value between start and end tag as a two dimensional
	 * <code>double</code> array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @return the value between start and end tag as a two dimensional
	 *         <code>double</code> array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, String)
	 * @see XMLParser#extractDoubleArrayForTag(StringBuffer, String, String)
	 */
	public static double[][] extractDouble2ArrayForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, startTag, endTag );
		int i, l = extractIntForTag( ex, "length" );
		double[][] array = new double[l][];
		for( i = 0; i < l; i++ ) {
			array[i] = extractDoubleArrayForTag( ex, "pos val=\"" + i + "\"", "pos" );
		}
		return array;
	}

	/**
	 * Returns the value between equal start and end tags as a two dimensional
	 * {@link String} array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param tag
	 *            the tags between which the value shall be taken (start tag
	 *            should be equal to end tag)
	 * 
	 * @return the value between the tags as a two dimensional {@link String}
	 *         array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractString2ArrayForTag(StringBuffer, String, String)
	 */
	public static String[][] extractString2ArrayForTag( StringBuffer source, String tag ) throws NonParsableException {
		return extractString2ArrayForTag( source, tag, tag );
	}

	/**
	 * Returns the value between start and end tag as a two dimensional
	 * {@link String} array.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be decoded from
	 *            XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @return the value between start and end tag as a two dimensional
	 *         {@link String} array
	 * 
	 * @throws NonParsableException
	 *             if the value could not be parsed
	 * 
	 * @see XMLParser#extractForTag(StringBuffer, String, String)
	 * @see XMLParser#extractStringArrayForTag(StringBuffer, String, String)
	 */
	public static String[][] extractString2ArrayForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException {
		StringBuffer ex = extractForTag( source, startTag, endTag );
		int i, l = extractIntForTag( ex, "length" );
		String[][] array = new String[l][];
		for( i = 0; i < l; i++ ) {
			array[i] = extractStringArrayForTag( ex, "pos val=\"" + i + "\"", "pos" );
		}
		return array;
	}

	/**
	 * Frames the {@link StringBuffer} <code>source</code> with equal tags
	 * &quot;&lt; <code>tag</code>&gt;&quot; and &quot;&lt;<code>/tag</code>
	 * &gt;&quot;.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param tag
	 *            the tags by which the {@link StringBuffer} should be framed
	 *            (start tag should be equal to end tag)
	 * 
	 * @see XMLParser#addTags(StringBuffer, String, String)
	 */
	public static void addTags( StringBuffer source, String tag ) {
		addTags( source, tag, tag );
	}

	/**
	 * Frames the {@link StringBuffer} <code>source</code> with &quot;&lt;
	 * <code>startTag</code>&gt;&quot; and &quot;&lt;<code>/endTag</code>
	 * &gt;&quot;.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 */
	public static void addTags( StringBuffer source, String startTag, String endTag ) {
		source.insert( 0, "<" + startTag + ">\n" );
		source.insert( source.length(), "</" + endTag + ">\n" );
	}

	/**
	 * Appends a <code>boolean</code> value with equal tags to the
	 * {@link StringBuffer} <code>source</code> in the following way: &quot;&lt;
	 * <code>tag</code>&gt;<code>b</code>&lt;<code>/tag</code>&gt;&quot;.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param b
	 *            the value that should be framed by the tags and appended to
	 *            the source
	 * @param tag
	 *            the tags by which the value should be framed (start tag should
	 *            be equal to end tag)
	 * 
	 * @see XMLParser#appendBooleanWithTags(StringBuffer, boolean, String,
	 *      String)
	 */
	public static void appendBooleanWithTags( StringBuffer source, boolean b, String tag ) {
		appendBooleanWithTags( source, b, tag, tag );
	}

	/**
	 * Appends a <code>boolean</code> value with start and end tag to the
	 * {@link StringBuffer} <code>source</code> in the following way: &quot;&lt;
	 * <code>startTag</code>&gt;<code>b</code>&lt;<code>/endTag</code>
	 * &gt;&quot;.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param b
	 *            the value that should be framed by start and end tag and
	 *            appended to the source
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 */
	public static void appendBooleanWithTags( StringBuffer source, boolean b, String startTag, String endTag ) {
		source.append( "<" + startTag + ">" + b + "</" + endTag + ">\n" );
	}

	/**
	 * Appends a <code>byte</code> value with equal tags to the
	 * {@link StringBuffer} <code>source</code> in the following way: &quot;&lt;
	 * <code>tag</code>&gt;<code>b</code>&lt;<code>/tag</code>&gt;&quot;.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param b
	 *            the value that should be framed by the tags and appended to
	 *            the source
	 * @param tag
	 *            the tags by which the value should be framed (start tag should
	 *            be equal to end tag)
	 * 
	 * @see XMLParser#appendByteWithTags(StringBuffer, byte, String, String)
	 */
	public static void appendByteWithTags( StringBuffer source, byte b, String tag ) {
		appendByteWithTags( source, b, tag, tag );
	}

	/**
	 * Appends a <code>byte</code> value with start and end tag to the
	 * {@link StringBuffer} <code>source</code> in the following way: &quot;&lt;
	 * <code>startTag</code>&gt;<code>b</code>&lt;<code>/endTag</code>
	 * &gt;&quot;.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param b
	 *            the value that should be framed by start and end tag and
	 *            appended to the source
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 */
	public static void appendByteWithTags( StringBuffer source, byte b, String startTag, String endTag ) {
		source.append( "<" + startTag + ">" + b + "</" + endTag + ">\n" );
	}

	/**
	 * Appends an {@link Enum} object with equal tags to the
	 * {@link StringBuffer} <code>source</code>.
	 * 
	 * @param <T>
	 *            the type of the {@link Enum} objects
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param e
	 *            the enumeration that should be framed by the tags and appended
	 *            to the source
	 * @param tag
	 *            the tags by which the enumeration should be framed (start tag
	 *            should be equal to end tag)
	 * 
	 * @see XMLParser#appendEnumWithTags(StringBuffer, Enum, String, String)
	 */
	public static <T extends Enum<T>> void appendEnumWithTags( StringBuffer source, Enum<T> e, String tag ) {
		appendEnumWithTags( source, e, tag, tag );
	}

	/**
	 * Appends an {@link Enum} object with start and end tag to the
	 * {@link StringBuffer} <code>source</code>.
	 * 
	 * @param <T>
	 *            the type of the {@link Enum} objects
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param e
	 *            the enumeration that should be framed by start and end tag and
	 *            appended to the source
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @see XMLParser#appendStringWithTags(StringBuffer, String, String, String)
	 */
	public static <T extends Enum<T>> void appendEnumWithTags( StringBuffer source, Enum<T> e, String startTag, String endTag ) {
		StringBuffer help = new StringBuffer( 1000 );
		help.append( "<" + startTag + ">\n" );
		appendStringWithTags( help, e.getClass().getName(), "enumName" );
		appendStringWithTags( help, e.name(), "name" );
		help.append( "</" + endTag + ">\n" );
		source.append( help );
	}

	/**
	 * Appends an <code>int</code> value with equal tags to the
	 * {@link StringBuffer} <code>source</code> in the following way: &quot;&lt;
	 * <code>tag</code>&gt;<code>i</code>&lt;<code>/tag</code>&gt;&quot;
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param i
	 *            the value that should be framed by the tags and appended to
	 *            the source
	 * @param tag
	 *            the tags by which the value should be framed (start tag should
	 *            be equal to end tag)
	 * 
	 * @see XMLParser#appendIntWithTags(StringBuffer, int, String, String)
	 */
	public static void appendIntWithTags( StringBuffer source, int i, String tag ) {
		appendIntWithTags( source, i, tag, tag );
	}

	/**
	 * Appends an <code>int</code> value with start and end tag to the
	 * {@link StringBuffer} <code>source</code> in the following way: &quot;&lt;
	 * <code>startTag</code>&gt;<code>i</code>&lt;<code>/endTag</code>
	 * &gt;&quot;.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param i
	 *            the value that should be framed by start and end tag and
	 *            appended to the source
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 */
	public static void appendIntWithTags( StringBuffer source, int i, String startTag, String endTag ) {
		source.append( "<" + startTag + ">" + i + "</" + endTag + ">\n" );
	}

	/**
	 * Appends a <code>long</code> value with equal tags to the
	 * {@link StringBuffer} <code>source</code> in the following way: &quot;&lt;
	 * <code>tag</code>&gt;<code>i</code>&lt;<code>/tag</code>&gt;&quot;.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param i
	 *            the value that should be framed by the tags and appended to
	 *            the source
	 * @param tag
	 *            the tags by which the value should be framed (start tag should
	 *            be equal to end tag)
	 * 
	 * @see XMLParser#appendLongWithTags(StringBuffer, long, String, String)
	 */
	public static void appendLongWithTags( StringBuffer source, long i, String tag ) {
		appendLongWithTags( source, i, tag, tag );
	}

	/**
	 * Appends a <code>long</code> value with start and end tag to the
	 * {@link StringBuffer} <code>source</code> in the following way: &quot;&lt;
	 * <code>startTag</code>&gt;<code>i</code>&lt;<code>/endTag</code>
	 * &gt;&quot;.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param i
	 *            the value that should be framed by start and end tag and
	 *            appended to the source
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 */
	public static void appendLongWithTags( StringBuffer source, long i, String startTag, String endTag ) {
		source.append( "<" + startTag + ">" + i + "</" + endTag + ">\n" );
	}

	/**
	 * Appends a <code>double</code> value with equal tags to the
	 * {@link StringBuffer} <code>source</code> in the following way: &quot;&lt;
	 * <code>tag</code>&gt;<code>d</code>&lt;<code>/tag</code>&gt;&quot;.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param d
	 *            the value that should be framed by the tags and appended to
	 *            the source
	 * @param startTag
	 *            the tags by which the value should be framed (start tag should
	 *            be equal to end tag)
	 * 
	 * @see XMLParser#appendDoubleWithTags(StringBuffer, double, String, String)
	 */
	public static void appendDoubleWithTags( StringBuffer source, double d, String startTag ) {
		appendDoubleWithTags( source, d, startTag, startTag );
	}

	/**
	 * Appends a <code>double</code> value with start and end tag to the
	 * {@link StringBuffer} <code>source</code> in the following way: &quot;&lt;
	 * <code>startTag</code>&gt;<code>d</code>&lt;<code>/endTag</code>
	 * &gt;&quot;.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param d
	 *            the value that should be framed by start and end tag and
	 *            appended to the source
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 */
	public static void appendDoubleWithTags( StringBuffer source, double d, String startTag, String endTag ) {
		source.append( "<" + startTag + ">" + d + "</" + endTag + ">\n" );
	}

	/**
	 * Appends a {@link Storable} object with equal tags to the
	 * {@link StringBuffer} <code>source</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param s
	 *            the object that should be framed by the tags and appended to
	 *            the source
	 * @param startTag
	 *            the tags by which the object should be framed (start tag
	 *            should be equal to end tag)
	 * 
	 * @see XMLParser#appendStorableWithTags(StringBuffer, Storable, String,
	 *      String)
	 */
	public static void appendStorableWithTags( StringBuffer source, Storable s, String startTag ) {
		appendStorableWithTags( source, s, startTag, startTag );
	}

	/**
	 * Appends a {@link Storable} object with start and end tag to the
	 * {@link StringBuffer} <code>source</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param s
	 *            the object that should be framed by start and end tag and
	 *            appended to the source
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 */
	public static void appendStorableWithTags( StringBuffer source, Storable s, String startTag, String endTag ) {
		StringBuffer help = new StringBuffer( 1000 );
		help.append( "<" + startTag + ">\n" );
		appendStringWithTags( help, s.getClass().getName(), "className" );
		help.append( s.toXML() );
		help.append( "\t</" + endTag + ">\n" );
		source.append( help );
	}

	/**
	 * Appends a {@link Storable} object or &quot;null&quot; with equal tags to
	 * the {@link StringBuffer} <code>source</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param s
	 *            the object that should be framed by the tags and appended to
	 *            the source, if <code>null</code> then &quot;null&quot; framed
	 *            by the tags is appended to the source
	 * @param tag
	 *            the tags by which the object or &quot;null&quot; should be
	 *            framed (start tag should be equal to end tag)
	 * 
	 * @see XMLParser#appendStorableOrNullWithTags(StringBuffer, Storable,
	 *      String, String)
	 */
	public static void appendStorableOrNullWithTags( StringBuffer source, Storable s, String tag ) {
		appendStorableOrNullWithTags( source, s, tag, tag );
	}

	/**
	 * Appends a {@link Storable} object or &quot;null&quot; with start and end
	 * tag to the {@link StringBuffer} <code>source</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param s
	 *            the object that should be framed by start and end tag and
	 *            appended to the source, if <code>null</code> then
	 *            &quot;null&quot; framed by start and end tag is appended to
	 *            the source
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 */
	public static void appendStorableOrNullWithTags( StringBuffer source, Storable s, String startTag, String endTag ) {
		StringBuffer help = new StringBuffer( 1000 );
		help.append( "<" + startTag + ">\n" );
		if( s != null ) {
			appendStringWithTags( help, s.getClass().getName(), "className" );
			help.append( s.toXML() );
		} else {
			help.append( "null" );
		}
		help.append( "\t</" + endTag + ">\n" );
		source.append( help );
	}

	/**
	 * Appends a {@link String} with equal tags to the {@link StringBuffer}
	 * <code>source</code> in the following way: &quot;&lt;<code>tag</code>&gt;
	 * <code>s</code>&lt;<code>/tag</code> &gt;&quot;.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param s
	 *            the {@link String} that should be framed by the tags and
	 *            appended to the source
	 * @param startTag
	 *            the tags by which the {@link String} should be framed (start
	 *            tag should be equal to end tag)
	 * 
	 * @see XMLParser#appendStringWithTags(StringBuffer, String, String, String)
	 */
	public static void appendStringWithTags( StringBuffer source, String s, String startTag ) {
		appendStringWithTags( source, s, startTag, startTag );
	}

	/**
	 * Appends a {@link String} with start and end tag to the
	 * {@link StringBuffer} <code>source</code> in the following way: &quot;&lt;
	 * <code>startTag</code>&gt;<code>s</code>&lt;<code>/endTag</code>
	 * &gt;&quot;.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param s
	 *            the value that should be framed by start and end tag and
	 *            appended to the source
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 */
	public static void appendStringWithTags( StringBuffer source, String s, String startTag, String endTag ) {
		source.append( "<" + startTag + ">" + s + "</" + endTag + ">\n" );
	}

	/**
	 * Encodes a <code>byte</code> array. The encoding is the following way: <br>
	 * &lt;length&gt;b.length&lt;/length&gt; <br>
	 * &lt;pos val=&quot;0&quot;&gt;b[0]&lt;/pos&gt; <br>
	 * &lt;pos val=&quot;1&quot;&gt;b[1]&lt;/pos&gt; <br>
	 * ... <br>
	 * It is necessary to include such an encoded array in tags or another
	 * array.
	 * 
	 * @param b
	 *            the array that should be encoded
	 * 
	 * @return a {@link StringBuffer} representing the encoded array
	 * 
	 * @see XMLParser#appendIntWithTags(StringBuffer, int, String)
	 * @see XMLParser#appendByteWithTags(StringBuffer, byte, String, String)
	 */
	protected static StringBuffer ByteArrayWithTags( byte[] b ) {
		int l = b.length;
		StringBuffer source = new StringBuffer( 25 + l * 30 );
		source.append( "\t" );
		appendIntWithTags( source, l, "length" );
		for( int j = 0; j < l; j++ ) {
			source.append( "\t" );
			appendByteWithTags( source, b[j], "pos val=\"" + j + "\"", "pos" );
		}
		return source;
	}

	/**
	 * Encodes an <code>int</code> array. The encoding is like in
	 * {@link #ByteArrayWithTags(byte[])}.
	 * 
	 * @param i
	 *            the array that should be encoded
	 * 
	 * @return a {@link StringBuffer} representing the encoded array
	 * 
	 * @see XMLParser#ByteArrayWithTags(byte[])
	 * @see XMLParser#appendIntWithTags(StringBuffer, int, String)
	 * @see XMLParser#appendIntWithTags(StringBuffer, int, String, String)
	 */
	protected static StringBuffer IntArrayWithTags( int[] i ) {
		int l = i.length;
		StringBuffer source = new StringBuffer( 25 + l * 30 );
		source.append( "\t" );
		appendIntWithTags( source, l, "length" );
		for( int j = 0; j < l; j++ ) {
			source.append( "\t" );
			appendIntWithTags( source, i[j], "pos val=\"" + j + "\"", "pos" );
		}
		return source;
	}

	/**
	 * Encodes a <code>double</code> array. The encoding is like in
	 * {@link #ByteArrayWithTags(byte[])}.
	 * 
	 * @param d
	 *            the array that should be encoded
	 * 
	 * @return a {@link StringBuffer} representing the encoded array
	 * 
	 * @see XMLParser#ByteArrayWithTags(byte[])
	 * @see XMLParser#appendIntWithTags(StringBuffer, int, String)
	 * @see XMLParser#appendDoubleWithTags(StringBuffer, double, String, String)
	 */
	protected static StringBuffer DoubleArrayWithTags( double[] d ) {
		int l = d.length;
		StringBuffer source = new StringBuffer( 25 + l * 40 );
		source.append( "\t" );
		appendIntWithTags( source, l, "length" );
		for( int j = 0; j < l; j++ ) {
			source.append( "\t" );
			appendDoubleWithTags( source, d[j], "pos val=\"" + j + "\"", "pos" );
		}
		return source;
	}

	/**
	 * Encodes a <code>boolean</code> array. The encoding is like in
	 * {@link #ByteArrayWithTags(byte[])}.
	 * 
	 * @param b
	 *            the array that should be encoded
	 * 
	 * @return a {@link StringBuffer} representing the encoded array
	 * 
	 * @see XMLParser#ByteArrayWithTags(byte[])
	 * @see XMLParser#appendIntWithTags(StringBuffer, int, String)
	 * @see XMLParser#appendBooleanWithTags(StringBuffer, boolean, String,
	 *      String)
	 */
	protected static StringBuffer BooleanArrayWithTags( boolean[] b ) {
		int l = b.length;
		StringBuffer source = new StringBuffer( 25 + l * 40 );
		source.append( "\t" );
		appendIntWithTags( source, l, "length" );
		for( int j = 0; j < l; j++ ) {
			source.append( "\t" );
			appendBooleanWithTags( source, b[j], "pos val=\"" + j + "\"", "pos" );
		}
		return source;
	}

	/**
	 * Encodes a {@link String} array. The encoding is like in
	 * {@link #ByteArrayWithTags(byte[])}.
	 * 
	 * @param s
	 *            the array that should be encoded
	 * 
	 * @return a {@link StringBuffer} representing the encoded array
	 * 
	 * @see XMLParser#ByteArrayWithTags(byte[])
	 * @see XMLParser#appendIntWithTags(StringBuffer, int, String)
	 * @see XMLParser#appendStringWithTags(StringBuffer, String, String, String)
	 */
	protected static StringBuffer StringArrayWithTags( String[] s ) {
		int l = s.length;
		StringBuffer source = new StringBuffer( 25 + l * 30 );
		source.append( "\t" );
		appendIntWithTags( source, l, "length" );
		for( int j = 0; j < l; j++ ) {
			source.append( "\t" );
			appendStringWithTags( source, s[j], "pos val=\"" + j + "\"", "pos" );
		}
		return source;
	}

	/**
	 * Encodes a {@link Storable} array. The encoding is like in
	 * {@link #ByteArrayWithTags(byte[])}.
	 * 
	 * @param s
	 *            the {@link Storable} object that should be encoded
	 * 
	 * @return a {@link StringBuffer} representing the encoded {@link Storable}
	 *         object
	 * 
	 * @see XMLParser#ByteArrayWithTags(byte[])
	 * @see XMLParser#appendIntWithTags(StringBuffer, int, String)
	 * @see XMLParser#appendStorableWithTags(StringBuffer, Storable, String,
	 *      String)
	 */
	protected static StringBuffer StorableArrayWithTags( Storable[] s ) {
		int l = s.length;
		StringBuffer source = new StringBuffer( 25 + l * 1030 );
		source.append( "\t" );
		appendIntWithTags( source, l, "length" );
		for( int j = 0; j < l; j++ ) {
			source.append( "\t" );
			appendStorableWithTags( source, s[j], "pos val=\"" + j + "\"", "pos" );
		}
		return source;
	}

	/**
	 * Appends an encoded <code>boolean</code> array with equal tags to the
	 * {@link StringBuffer} <code>source</code>. The encoding is like in
	 * {@link #appendByteArrayWithTags(StringBuffer, byte[], String, String)}.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param b
	 *            the array that should be encoded and appended to the source
	 *            framed by the tags
	 * @param tag
	 *            the tags by which the encoded array should be framed (start
	 *            tag should be equal to end tag)
	 * 
	 * @see XMLParser#appendByteArrayWithTags(StringBuffer, byte[], String,
	 *      String)
	 * @see XMLParser#appendBooleanArrayWithTags(StringBuffer, boolean[],
	 *      String, String)
	 */
	public static void appendBooleanArrayWithTags( StringBuffer source, boolean[] b, String tag ) {
		appendBooleanArrayWithTags( source, b, tag, tag );
	}

	/**
	 * Appends an encoded <code>boolean</code> array with start and end tag to
	 * the {@link StringBuffer} <code>source</code>. The encoding is like in
	 * {@link #appendByteArrayWithTags(StringBuffer, byte[], String, String)}.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param b
	 *            the array that should be encoded and appended to the source
	 *            framed by start and end tag
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @see XMLParser#appendByteArrayWithTags(StringBuffer, byte[], String,
	 *      String)
	 * @see XMLParser#BooleanArrayWithTags(boolean[])
	 * @see XMLParser#addTags(StringBuffer, String, String)
	 */
	public static void appendBooleanArrayWithTags( StringBuffer source, boolean[] b, String startTag, String endTag ) {
		StringBuffer help = BooleanArrayWithTags( b );
		addTags( help, startTag, endTag );
		source.append( help );
	}

	/**
	 * Appends an encoded <code>byte</code> array with equal tags to the
	 * {@link StringBuffer} <code>source</code>. The encoding is like in
	 * {@link #appendByteArrayWithTags(StringBuffer, byte[], String, String)}.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param b
	 *            the array that should be encoded and appended to the source
	 *            framed by the tags
	 * @param tag
	 *            the tags by which the encoded array should be framed (start
	 *            tag should be equal to end tag)
	 * 
	 * @see XMLParser#appendByteArrayWithTags(StringBuffer, byte[], String,
	 *      String)
	 */
	public static void appendByteArrayWithTags( StringBuffer source, byte[] b, String tag ) {
		appendByteArrayWithTags( source, b, tag, tag );
	}

	/**
	 * Appends an encoded <code>byte</code> array with start and end tag to the
	 * {@link StringBuffer} <code>source</code>. The encoding is this way: <br>
	 * &lt;<code>startTag</code>&gt; &lt;length&gt;b.length&lt;/length&gt; <br>
	 * &lt;pos val=&quot;0&quot;&gt;b[0]&lt;/pos&gt; <br>
	 * &lt;pos val=&quot;1&quot;&gt;b[1]&lt;/pos&gt; <br>
	 * ... <br>
	 * &lt;<code>/endTag</code>&gt;.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param b
	 *            the array that should be encoded and appended to the source
	 *            framed by start and end tag
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @see XMLParser#ByteArrayWithTags(byte[])
	 * @see XMLParser#addTags(StringBuffer, String, String)
	 */
	public static void appendByteArrayWithTags( StringBuffer source, byte[] b, String startTag, String endTag ) {
		StringBuffer help = ByteArrayWithTags( b );
		addTags( help, startTag, endTag );
		source.append( help );
	}

	/**
	 * Appends an encoded <code>int</code> array with equal tags to the
	 * {@link StringBuffer} <code>source</code>. The encoding is like in
	 * {@link #appendByteArrayWithTags(StringBuffer, byte[], String, String)}.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param i
	 *            the array that should be encoded and appended to the source
	 *            framed by the tags
	 * @param tag
	 *            the tags by which the encoded array should be framed (start
	 *            tag should be equal to end tag)
	 * 
	 * @see XMLParser#appendByteArrayWithTags(StringBuffer, byte[], String,
	 *      String)
	 * @see XMLParser#appendIntArrayWithTags(StringBuffer, int[], String,
	 *      String)
	 */
	public static void appendIntArrayWithTags( StringBuffer source, int[] i, String tag ) {
		appendIntArrayWithTags( source, i, tag, tag );
	}

	/**
	 * Appends an encoded <code>int</code> array with start and end tag to the
	 * {@link StringBuffer} <code>source</code>. The encoding is like in
	 * {@link #appendByteArrayWithTags(StringBuffer, byte[], String, String)}.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param i
	 *            the array that should be encoded and appended to the source
	 *            framed by start and end tag
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @see XMLParser#appendByteArrayWithTags(StringBuffer, byte[], String,
	 *      String)
	 * @see XMLParser#IntArrayWithTags(int[])
	 * @see XMLParser#addTags(StringBuffer, String, String)
	 */
	public static void appendIntArrayWithTags( StringBuffer source, int[] i, String startTag, String endTag ) {
		StringBuffer help = IntArrayWithTags( i );
		addTags( help, startTag, endTag );
		source.append( help );
	}

	/**
	 * Appends an encoded <code>double</code> array with equal tags to the
	 * {@link StringBuffer} <code>source</code>. The encoding is like in like in
	 * {@link #appendByteArrayWithTags(StringBuffer, byte[], String, String)}.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param d
	 *            the array that should be encoded and appended to the source
	 *            framed by the tags
	 * @param tag
	 *            the tags by which the encoded array should be framed (start
	 *            tag should be equal to end tag)
	 * 
	 * @see XMLParser#appendByteArrayWithTags(StringBuffer, byte[], String,
	 *      String)
	 * @see XMLParser#appendDoubleArrayWithTags(StringBuffer, double[], String,
	 *      String)
	 */
	public static void appendDoubleArrayWithTags( StringBuffer source, double[] d, String tag ) {
		appendDoubleArrayWithTags( source, d, tag, tag );
	}

	/**
	 * Appends an encoded <code>double</code> array with start and end tag to
	 * the {@link StringBuffer} <code>source</code>. The encoding is like in
	 * like in
	 * {@link #appendByteArrayWithTags(StringBuffer, byte[], String, String)}.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param d
	 *            the array that should be encoded and appended to the source
	 *            framed by start and end tag
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @see XMLParser#appendByteArrayWithTags(StringBuffer, byte[], String,
	 *      String)
	 * @see XMLParser#DoubleArrayWithTags(double[])
	 * @see XMLParser#addTags(StringBuffer, String, String)
	 */
	public static void appendDoubleArrayWithTags( StringBuffer source, double[] d, String startTag, String endTag ) {
		StringBuffer help = DoubleArrayWithTags( d );
		addTags( help, startTag, endTag );
		source.append( help );
	}

	/**
	 * Appends an encoded {@link Storable} array with equal tags to the
	 * {@link StringBuffer} <code>source</code>. The encoding is like in like in
	 * {@link #appendByteArrayWithTags(StringBuffer, byte[], String, String)}.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param s
	 *            the {@link Storable} array that should be encoded and appended
	 *            to the source framed by the tags
	 * @param tag
	 *            the tags by which the encoded {@link Storable} array should be
	 *            framed (start tag should be equal to end tag)
	 * 
	 * @see XMLParser#appendByteArrayWithTags(StringBuffer, byte[], String,
	 *      String)
	 * @see XMLParser#appendStorableArrayWithTags(StringBuffer, Storable[],
	 *      String, String)
	 */
	public static void appendStorableArrayWithTags( StringBuffer source, Storable[] s, String tag ) {
		appendStorableArrayWithTags( source, s, tag, tag );
	}

	/**
	 * Appends an encoded {@link Storable} array with start and end tag to the
	 * {@link StringBuffer} <code>source</code>. The encoding is like in like in
	 * {@link #appendByteArrayWithTags(StringBuffer, byte[], String, String)}.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param s
	 *            the {@link Storable} array that should be encoded and appended
	 *            to the source framed by start and end tag
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @see XMLParser#appendByteArrayWithTags(StringBuffer, byte[], String,
	 *      String)
	 * @see XMLParser#StorableArrayWithTags(Storable[])
	 * @see XMLParser#addTags(StringBuffer, String, String)
	 */
	public static void appendStorableArrayWithTags( StringBuffer source, Storable[] s, String startTag, String endTag ) {
		StringBuffer help = StorableArrayWithTags( s );
		addTags( help, startTag, endTag );
		source.append( help );
	}

	/**
	 * Appends an encoded {@link String} array with equal tags to the
	 * {@link StringBuffer} <code>source</code>. The encoding is like in
	 * {@link #appendByteArrayWithTags(StringBuffer, byte[], String, String)}.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param s
	 *            the array that should be encoded and appended to the source
	 *            framed by the tags
	 * @param tag
	 *            the tags by which the encoded array should be framed (start
	 *            tag should be equal to end tag)
	 * 
	 * @see XMLParser#appendByteArrayWithTags(StringBuffer, byte[], String,
	 *      String)
	 * @see XMLParser#appendStringArrayWithTags(StringBuffer, String[], String,
	 *      String)
	 */
	public static void appendStringArrayWithTags( StringBuffer source, String[] s, String tag ) {
		appendStringArrayWithTags( source, s, tag, tag );
	}

	/**
	 * Appends an encoded {@link String} array with start and end tag to the
	 * {@link StringBuffer} <code>source</code>. The encoding is like in
	 * {@link #appendByteArrayWithTags(StringBuffer, byte[], String, String)}.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param s
	 *            the array that should be encoded and appended to the source
	 *            framed by start and end tag
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @see XMLParser#appendByteArrayWithTags(StringBuffer, byte[], String,
	 *      String)
	 * @see XMLParser#StringArrayWithTags(String[])
	 * @see XMLParser#addTags(StringBuffer, String, String)
	 */
	public static void appendStringArrayWithTags( StringBuffer source, String[] s, String startTag, String endTag ) {
		StringBuffer help = StringArrayWithTags( s );
		addTags( help, startTag, endTag );
		source.append( help );
	}

	/**
	 * Appends an encoded two dimensional <code>boolean</code> array with equal
	 * tags to the {@link StringBuffer} <code>source</code>. The encoding is
	 * like in
	 * {@link #appendByte2ArrayWithTags(StringBuffer, byte[][], String, String)}
	 * .
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param i
	 *            the two dimensional array that should be encoded and appended
	 *            to the source framed by the tags
	 * @param tag
	 *            the tags by which the encoded two dimensional array should be
	 *            framed (start tag should be equal to end tag)
	 * 
	 * @see XMLParser#appendByte2ArrayWithTags(StringBuffer, byte[][], String,
	 *      String)
	 * @see XMLParser#appendBoolean2ArrayWithTags(StringBuffer, boolean[][],
	 *      String, String)
	 */
	public static void appendBoolean2ArrayWithTags( StringBuffer source, boolean[][] i, String tag ) {
		appendBoolean2ArrayWithTags( source, i, tag, tag );
	}

	/**
	 * Appends an encoded two dimensional <code>boolean</code> array with start
	 * and end tag to the {@link StringBuffer} <code>source</code>. The encoding
	 * is is like in
	 * {@link #appendByte2ArrayWithTags(StringBuffer, byte[][], String, String)}
	 * .
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param i
	 *            the two dimensional array that should be encoded and appended
	 *            to the source framed by start and end tag
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @see XMLParser#appendByte2ArrayWithTags(StringBuffer, byte[][], String,
	 *      String)
	 * @see XMLParser#appendIntWithTags(StringBuffer, int, String)
	 * @see XMLParser#BooleanArrayWithTags(boolean[])
	 * @see XMLParser#addTags(StringBuffer, String, String)
	 */
	public static void appendBoolean2ArrayWithTags( StringBuffer source, boolean[][] i, String startTag, String endTag ) {
		int l = i.length;
		StringBuffer help1 = new StringBuffer( 2500 ), help2;
		help1.append( "\t" );
		appendIntWithTags( help1, l, "length" );
		for( int j = 0; j < l; j++ ) {
			help2 = BooleanArrayWithTags( i[j] );
			help2.append( "\t" );
			addTags( help2, "pos val=\"" + j + "\"", "pos" );
			help2.insert( 0, "\n\t" );
			help1.append( help2 );
		}
		addTags( help1, startTag, endTag );
		source.append( help1 );
	}

	/**
	 * Appends an encoded two dimensional <code>byte</code> array with equal
	 * tags to the {@link StringBuffer} <code>source</code>. The encoding is
	 * like in
	 * {@link #appendByte2ArrayWithTags(StringBuffer, byte[][], String, String)}
	 * .
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param i
	 *            the two dimensional array that should be encoded and appended
	 *            to the source framed by the tags
	 * @param tag
	 *            the tags by which the encoded two dimensional array should be
	 *            framed (start tag should be equal to end tag)
	 * 
	 * @see XMLParser#appendByte2ArrayWithTags(StringBuffer, byte[][], String,
	 *      String)
	 */
	public static void appendByte2ArrayWithTags( StringBuffer source, byte[][] i, String tag ) {
		appendByte2ArrayWithTags( source, i, tag, tag );
	}

	/**
	 * Appends an encoded two dimensional <code>byte</code> array with start and
	 * end tag to the {@link StringBuffer} <code>source</code>. The encoding is
	 * the following way: <br>
	 * &lt;startTag&gt; &lt;length&gt;i.length&lt;/length&gt; <br>
	 * &lt;pos val=&quot;0&quot;&gt; &lt;length&gt;i[0].length&lt;/length&gt; <br>
	 * &lt;pos val=&quot;0&quot;&gt;i[0][0]&lt;/pos&gt; <br>
	 * &lt;pos val=&quot;1&quot;&gt;i[0][1]&lt;/pos&gt; <br>
	 * ... <br>
	 * &lt;/pos&gt; <br>
	 * &lt;pos val=&quot;1&quot;&gt; &lt;length&gt;i[1].length&lt;/length&gt; <br>
	 * ... <br>
	 * &lt;/pos&gt; <br>
	 * ... <br>
	 * &lt;/endTag&gt;.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param i
	 *            the two dimensional array that should be encoded and appended
	 *            to the source framed by start and end tag
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @see XMLParser#appendIntWithTags(StringBuffer, int, String)
	 * @see XMLParser#ByteArrayWithTags(byte[])
	 * @see XMLParser#addTags(StringBuffer, String, String)
	 */
	public static void appendByte2ArrayWithTags( StringBuffer source, byte[][] i, String startTag, String endTag ) {
		int l = i.length;
		StringBuffer help1 = new StringBuffer( 2500 ), help2;
		help1.append( "\t" );
		appendIntWithTags( help1, l, "length" );
		for( int j = 0; j < l; j++ ) {
			help2 = ByteArrayWithTags( i[j] );
			help2.append( "\t" );
			addTags( help2, "pos val=\"" + j + "\"", "pos" );
			help2.insert( 0, "\n\t" );
			help1.append( help2 );
		}
		addTags( help1, startTag, endTag );
		source.append( help1 );
	}

	/**
	 * Appends an encoded two dimensional <code>int</code> array with equal tags
	 * to the {@link StringBuffer} <code>source</code>. The encoding is like in
	 * {@link #appendByte2ArrayWithTags(StringBuffer, byte[][], String, String)}
	 * .
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param i
	 *            the two dimensional array that should be encoded and appended
	 *            to the source framed by the tags
	 * @param tag
	 *            the tags by which the encoded two dimensional array should be
	 *            framed (start tag should be equal to end tag)
	 * 
	 * @see XMLParser#appendByte2ArrayWithTags(StringBuffer, byte[][], String,
	 *      String)
	 * @see XMLParser#appendInt2ArrayWithTags(StringBuffer, int[][], String,
	 *      String)
	 */
	public static void appendInt2ArrayWithTags( StringBuffer source, int[][] i, String tag ) {
		appendInt2ArrayWithTags( source, i, tag, tag );
	}

	/**
	 * Appends an encoded two dimensional <code>int</code> array with start and
	 * end tag to the {@link StringBuffer} <code>source</code>. The encoding is
	 * like in
	 * {@link #appendByte2ArrayWithTags(StringBuffer, byte[][], String, String)}
	 * .
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param i
	 *            the two dimensional array that should be encoded and appended
	 *            to the source framed by start and end tag
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @see XMLParser#appendByte2ArrayWithTags(StringBuffer, byte[][], String,
	 *      String)
	 * @see XMLParser#appendIntWithTags(StringBuffer, int, String)
	 * @see XMLParser#IntArrayWithTags(int[])
	 * @see XMLParser#addTags(StringBuffer, String, String)
	 */
	public static void appendInt2ArrayWithTags( StringBuffer source, int[][] i, String startTag, String endTag ) {
		int l = i.length;
		StringBuffer help1 = new StringBuffer( 2500 ), help2;
		help1.append( "\t" );
		appendIntWithTags( help1, l, "length" );
		for( int j = 0; j < l; j++ ) {
			help2 = IntArrayWithTags( i[j] );
			help2.append( "\t" );
			addTags( help2, "pos val=\"" + j + "\"", "pos" );
			help2.insert( 0, "\n\t" );
			help1.append( help2 );
		}
		addTags( help1, startTag, endTag );
		source.append( help1 );
	}

	/**
	 * Appends an encoded two dimensional <code>double</code> array with equal
	 * tags to the {@link StringBuffer} <code>source</code>. The encoding is
	 * like in
	 * {@link #appendByte2ArrayWithTags(StringBuffer, byte[][], String, String)}
	 * .
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param d
	 *            the two dimensional array that should be encoded and appended
	 *            to the source framed by the tags
	 * @param tag
	 *            the tags by which the encoded two dimensional array should be
	 *            framed (start tag should be equal to end tag)
	 * 
	 * @see XMLParser#appendByte2ArrayWithTags(StringBuffer, byte[][], String,
	 *      String)
	 * @see XMLParser#appendDouble2ArrayWithTags(StringBuffer, double[][],
	 *      String, String)
	 */
	public static void appendDouble2ArrayWithTags( StringBuffer source, double[][] d, String tag ) {
		appendDouble2ArrayWithTags( source, d, tag, tag );
	}

	/**
	 * Appends an encoded two dimensional <code>double</code> array with start
	 * and end tag to the {@link StringBuffer} <code>source</code>. The encoding
	 * is like in
	 * {@link #appendByte2ArrayWithTags(StringBuffer, byte[][], String, String)}
	 * .
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param d
	 *            the two dimensional array that should be encoded and appended
	 *            to the source framed by start and end tag
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @see XMLParser#appendByte2ArrayWithTags(StringBuffer, byte[][], String,
	 *      String)
	 * @see XMLParser#appendIntWithTags(StringBuffer, int, String)
	 * @see XMLParser#DoubleArrayWithTags(double[])
	 * @see XMLParser#addTags(StringBuffer, String, String)
	 */
	public static void appendDouble2ArrayWithTags( StringBuffer source, double[][] d, String startTag, String endTag ) {
		int l = d.length;
		StringBuffer help1 = new StringBuffer( 2500 ), help2;
		help1.append( "\t" );
		appendIntWithTags( help1, l, "length" );
		for( int j = 0; j < l; j++ ) {
			help2 = DoubleArrayWithTags( d[j] );
			help2.append( "\t" );
			addTags( help2, "pos val=\"" + j + "\"", "pos" );
			help2.insert( 0, "\n\t" );
			help1.append( help2 );
		}
		addTags( help1, startTag, endTag );
		source.append( help1 );
	}

	/**
	 * Appends an encoded two dimensional {@link Storable} array with equal tags
	 * to the {@link StringBuffer} <code>source</code>. The encoding is like in
	 * {@link #appendByte2ArrayWithTags(StringBuffer, byte[][], String, String)}
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param s
	 *            the two dimensional {@link Storable} array that should be
	 *            encoded and appended to the source framed by the tags
	 * @param tag
	 *            the tags by which the encoded two dimensional {@link Storable}
	 *            array should be framed (start tag should be equal to end tag)
	 * 
	 * @see XMLParser#appendByte2ArrayWithTags(StringBuffer, byte[][], String,
	 *      String)
	 * @see XMLParser#appendStorable2ArrayWithTags(StringBuffer, Storable[][],
	 *      String, String)
	 */
	public static void appendStorable2ArrayWithTags( StringBuffer source, Storable[][] s, String tag ) {
		appendStorable2ArrayWithTags( source, s, tag, tag );
	}

	/**
	 * Appends an encoded two dimensional {@link Storable} array with start and
	 * end tag to the {@link StringBuffer} <code>source</code>. The encoding is
	 * like in
	 * {@link #appendByte2ArrayWithTags(StringBuffer, byte[][], String, String)}
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param s
	 *            the two dimensional {@link Storable} array that should be
	 *            encoded and appended to the source framed by start and end tag
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @see XMLParser#appendByte2ArrayWithTags(StringBuffer, byte[][], String,
	 *      String)
	 * @see XMLParser#appendIntWithTags(StringBuffer, int, String)
	 * @see XMLParser#StorableArrayWithTags(Storable[])
	 * @see XMLParser#addTags(StringBuffer, String, String)
	 */
	public static void appendStorable2ArrayWithTags( StringBuffer source, Storable[][] s, String startTag, String endTag ) {
		int l = s.length;
		StringBuffer help1 = new StringBuffer( 2500 ), help2;
		help1.append( "\t" );
		appendIntWithTags( help1, l, "length" );
		for( int j = 0; j < l; j++ ) {
			help2 = StorableArrayWithTags( s[j] );
			help2.append( "\t" );
			addTags( help2, "pos val=\"" + j + "\"", "pos" );
			help2.insert( 0, "\n\t" );
			help1.append( help2 );
		}
		addTags( help1, startTag, endTag );
		source.append( help1 );
	}

	/**
	 * Appends an encoded three dimensional {@link Storable} array with equal
	 * tags to the {@link StringBuffer} <code>source</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param s
	 *            the three dimensional {@link Storable} array that should be
	 *            encoded and appended to the source framed by the tags
	 * @param tag
	 *            the tags by which the encoded three dimensional
	 *            {@link Storable} array should be framed (start tag should be
	 *            equal to end tag)
	 * 
	 * @see XMLParser#appendStorable3ArrayWithTags(StringBuffer, Storable[][][],
	 *      String, String)
	 */
	public static void appendStorable3ArrayWithTags( StringBuffer source, Storable[][][] s, String tag ) {
		appendStorable3ArrayWithTags( source, s, tag, tag );
	}

	/**
	 * Appends an encoded three dimensional {@link Storable} array with start
	 * and end tag to the {@link StringBuffer} <code>source</code>.
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param s
	 *            the three dimensional {@link Storable} array that should be
	 *            encoded and appended to the source framed by start and end tag
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @see XMLParser#appendIntWithTags(StringBuffer, int, String)
	 * @see XMLParser#appendStorable2ArrayWithTags(StringBuffer, Storable[][],
	 *      String, String)
	 * @see XMLParser#addTags(StringBuffer, String, String)
	 */
	public static void appendStorable3ArrayWithTags( StringBuffer source, Storable[][][] s, String startTag, String endTag ) {
		int l = s.length;
		StringBuffer help1 = new StringBuffer( 2500 );
		help1.append( "\t" );
		appendIntWithTags( help1, l, "length" );
		for( int j = 0; j < l; j++ ) {
			appendStorable2ArrayWithTags( help1, s[j], "pos val=\"" + j + "\"", "pos" );
		}
		addTags( help1, startTag, endTag );
		source.append( help1 );
	}

	/**
	 * Appends an encoded two dimensional {@link String} array with equal tags
	 * to the {@link StringBuffer} <code>source</code>. The encoding is like in
	 * {@link #appendByte2ArrayWithTags(StringBuffer, byte[][], String, String)}
	 * .
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param s
	 *            the two dimensional array that should be encoded and appended
	 *            to the source framed by the tags
	 * @param tag
	 *            the tags by which the encoded two dimensional array should be
	 *            framed (start tag should be equal to end tag)
	 * 
	 * @see XMLParser#appendByte2ArrayWithTags(StringBuffer, byte[][], String,
	 *      String)
	 * @see XMLParser#appendString2ArrayWithTags(StringBuffer, String[][],
	 *      String, String)
	 */
	public static void appendString2ArrayWithTags( StringBuffer source, String[][] s, String tag ) {
		appendString2ArrayWithTags( source, s, tag, tag );
	}

	/**
	 * Appends an encoded two dimensional {@link String} array with start and
	 * end tag to the {@link StringBuffer} <code>source</code>. The encoding is
	 * like in
	 * {@link #appendByte2ArrayWithTags(StringBuffer, byte[][], String, String)}
	 * .
	 * 
	 * @param source
	 *            the source {@link StringBuffer} that should be encoded in XML
	 * @param s
	 *            the two dimensional array that should be encoded and appended
	 *            to the source framed by start and end tag
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * 
	 * @see XMLParser#appendByte2ArrayWithTags(StringBuffer, byte[][], String,
	 *      String)
	 * @see XMLParser#appendIntWithTags(StringBuffer, int, String)
	 * @see XMLParser#StringArrayWithTags(String[])
	 * @see XMLParser#addTags(StringBuffer, String, String)
	 */
	public static void appendString2ArrayWithTags( StringBuffer source, String[][] s, String startTag, String endTag ) {
		int l = s.length;
		StringBuffer help1 = new StringBuffer( 2500 ), help2;
		help1.append( "\t" );
		appendIntWithTags( help1, l, "length" );
		for( int j = 0; j < l; j++ ) {
			help2 = StringArrayWithTags( s[j] );
			help2.append( "\t" );
			addTags( help2, "pos val=\"" + j + "\"", "pos" );
			help2.insert( 0, "\n\t" );
			help1.append( help2 );
		}
		addTags( help1, startTag, endTag );
		source.append( help1 );
	}
}