/*
 * This file is part of Jstacs.
 * 
 * Jstacs is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * Jstacs is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * Jstacs. If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.scoringFunctions;

import java.util.Arrays;

import de.jstacs.NonParsableException;
import de.jstacs.data.AlphabetContainer;
import de.jstacs.data.Sample;
import de.jstacs.data.Sequence;
import de.jstacs.data.sequences.annotation.StrandedLocatedSequenceAnnotationWithLength.Strand;
import de.jstacs.io.ArrayHandler;
import de.jstacs.io.XMLParser;
import de.jstacs.motifDiscovery.MotifDiscoverer;
import de.jstacs.motifDiscovery.MutableMotifDiscoverer;
import de.jstacs.scoringFunctions.mix.AbstractMixtureScoringFunction;
import de.jstacs.utils.DoubleList;
import de.jstacs.utils.IntList;

/**
 * This class enables the user to model parts of a sequence independent of each
 * other. The first part of the sequence is modeled by the first
 * {@link NormalizableScoringFunction} and has the length of the first
 * {@link NormalizableScoringFunction}, the second part starts directly after
 * the first part, is modeled by the second {@link NormalizableScoringFunction}
 * ... etc.
 * 
 * @author Jens Keilwagen
 */
public class IndependentProductScoringFunction extends AbstractNormalizableScoringFunction implements MutableMotifDiscoverer {

	private NormalizableScoringFunction[] score;

	private int[] start, partialLength, params, motifs, motifIndexArray, components, componentIndexArray;

	private boolean[] isVariable;

	private IntList partIList;

	private final static AlphabetContainer getAlphabetContainer( NormalizableScoringFunction[] functions, int length[] ) throws IllegalArgumentException {
		AlphabetContainer[] cons = new AlphabetContainer[functions.length];
		int[] lengths = new int[functions.length];
		for( int i = 0; i < functions.length; i++ ) {
			cons[i] = functions[i].getAlphabetContainer();
			lengths[i] = length[i];
		}
		return new AlphabetContainer( cons, lengths );
	}

	private final static int sum( int[] length ) throws IllegalArgumentException {
		int res = 0, i = 0;
		while( i < length.length && length[i] > 0 ) {
			res += length[i++];
		}
		if( i != length.length ) {
			throw new IllegalArgumentException( "The length with index " + i + " is 0." );
		}
		return res;
	}

	private final static int[] getLengthArray( NormalizableScoringFunction... functions ) throws IllegalArgumentException {
		int i = 0;
		int[] res = new int[functions.length];
		while( i < functions.length && functions[i].getLength() > 0 ) {
			res[i] = functions[i].getLength();
			i++;
		}
		if( i != functions.length ) {
			throw new IllegalArgumentException( "The NormalizableScoringFunction with index " + i + " has a length 0." );
		}
		return res;
	}

	/**
	 * This constructor creates an instance of an
	 * {@link IndependentProductScoringFunction} from a given series of
	 * independent {@link NormalizableScoringFunction}s. The length that is
	 * modeled by each component is determined by
	 * {@link ScoringFunction#getLength()}. So the length should not be 0.
	 * 
	 * @param functions
	 *            the components, i.e. the given series of independent
	 *            {@link NormalizableScoringFunction}s
	 * 
	 * @throws CloneNotSupportedException
	 *             if at least one element of <code>functions</code> could not
	 *             be cloned
	 * @throws IllegalArgumentException
	 *             if at least one component has length 0 or the components do
	 *             not have the same equivalent sample size (ess)
	 * 
	 * @see IndependentProductScoringFunction#IndependentProductScoringFunction(NormalizableScoringFunction[],
	 *      int[])
	 */
	public IndependentProductScoringFunction( NormalizableScoringFunction... functions ) throws CloneNotSupportedException,
																						IllegalArgumentException {
		this( functions, getLengthArray( functions ) );
	}

	/**
	 * This constructor creates an instance of an
	 * {@link IndependentProductScoringFunction} from given series of
	 * independent {@link NormalizableScoringFunction}s and lengths.
	 * 
	 * @param functions
	 *            the components, i.e. the given series of independent
	 *            {@link NormalizableScoringFunction}s
	 * @param length
	 *            the lengths, one for each component
	 * 
	 * @throws CloneNotSupportedException
	 *             if at least one component could not be cloned
	 * @throws IllegalArgumentException
	 *             if the lengths and the components are not matching or the
	 *             components do not have the same equivalent sample size (ess)
	 * 
	 * @see IndependentProductScoringFunction#IndependentProductScoringFunction(NormalizableScoringFunction[],
	 *      int[])
	 */
	public IndependentProductScoringFunction( NormalizableScoringFunction[] functions, int[] length ) throws CloneNotSupportedException,
																										IllegalArgumentException {
		super( getAlphabetContainer( functions, length ), sum( length ) );
		score = ArrayHandler.clone( functions );
		setStartsAndLengths( length );
		setParamsStarts();
		// test ESS
		double ess = score[0].getEss();
		int i = 1;
		while( i < score.length && score[i].getEss() == ess ) {
			i++;
		}
		if( i != score.length ) {
			throw new IllegalArgumentException( "All NormalizableScoringFunction have to use the same ESS. Violated at index " + i + "." );
		}
		prepare();
	}

	private void prepare() {
		motifs = new int[score.length + 1];
		components = new int[score.length];
		for( int i = 0, c = 1; i < score.length; i++ ) {
			motifs[i + 1] = motifs[i];
			if( score[i] instanceof MotifDiscoverer ) {
				motifs[i + 1] += ( (MotifDiscoverer)score[i] ).getNumberOfMotifs();
			}
			components[i] = c;
			if( score[i] instanceof AbstractMixtureScoringFunction ) {
				components[i] *= ( (AbstractMixtureScoringFunction)score[i] ).getNumberOfComponents();
			} else if( score[i] instanceof MotifDiscoverer ) {
				components[i] *= ( (MotifDiscoverer)score[i] ).getNumberOfComponents();
			}
		}
		motifIndexArray = new int[3];
		componentIndexArray = new int[score.length];
	}

	/**
	 * This is the constructor for the interface {@link de.jstacs.Storable}.
	 * Creates a new {@link IndependentProductScoringFunction} out of a
	 * {@link StringBuffer} as returned by {@link #toXML()}.
	 * 
	 * @param source
	 *            the XML representation as {@link StringBuffer}
	 * 
	 * @throws NonParsableException
	 *             if the XML representation could not be parsed
	 */
	public IndependentProductScoringFunction( StringBuffer source ) throws NonParsableException {
		super( source );
	}

	private void setStartsAndLengths( int[] length ) throws IllegalArgumentException {
		int oldStart = 0;
		start = new int[score.length];
		partialLength = new int[score.length];
		isVariable = new boolean[score.length];
		for( int i = 0; i < score.length; i++ ) {
			start[i] = oldStart;
			partialLength[i] = length[i];
			isVariable[i] = score[i] instanceof VariableLengthScoringFunction;
			if( !isVariable[i] && score[i].getLength() != partialLength[i] ) {
				throw new IllegalArgumentException( "Could not use length " + partialLength[i] + " for component " + i + "." );
			}
			oldStart += length[i];
		}
		partIList = new IntList();
	}

	private void setParamsStarts() {
		if( params == null ) {
			params = new int[score.length + 1];
		}
		for( int n, i = 0; i < score.length; i++ ) {
			n = score[i].getNumberOfParameters();
			if( n == UNKNOWN ) {
				params = null;
				break;
			} else {
				params[i + 1] = params[i] + n;
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.scoringFunctions.AbstractNormalizableScoringFunction#clone()
	 */
	public IndependentProductScoringFunction clone() throws CloneNotSupportedException {
		IndependentProductScoringFunction clone = (IndependentProductScoringFunction)super.clone();
		clone.score = ArrayHandler.clone( score );
		clone.setStartsAndLengths( partialLength );
		clone.params = null;
		clone.setParamsStarts();
		clone.motifs = motifs.clone();
		clone.motifIndexArray = motifIndexArray.clone();
		clone.components = components.clone();
		clone.componentIndexArray = componentIndexArray.clone();
		return clone;
	}

	private int getSFIndex( int index ) {
		int i = 1;
		while( i < params.length && index >= params[i] ) {
			i++;
		}
		return i - 1;
	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.scoringFunctions.NormalizableScoringFunction#getSizeOfEventSpaceForRandomVariablesOfParameter(int)
	 */
	public int getSizeOfEventSpaceForRandomVariablesOfParameter( int index ) {
		int i = getSFIndex( index );
		return score[i].getSizeOfEventSpaceForRandomVariablesOfParameter( index - params[i] );
	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.scoringFunctions.NormalizableScoringFunction#getNormalizationConstant()
	 */
	public double getNormalizationConstant() {
		double norm = 1;
		for( int i = 0; i < score.length; i++ ) {
			norm *= score[i].getNormalizationConstant();
		}
		return norm;
	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.scoringFunctions.NormalizableScoringFunction#getPartialNormalizationConstant(int)
	 */
	public double getPartialNormalizationConstant( int parameterIndex ) throws Exception {
		int i = 0, j = getSFIndex( parameterIndex );
		double partNorm = 1;
		for( ; i < score.length; i++ ) {
			if( i == j ) {
				partNorm *= score[i].getPartialNormalizationConstant( parameterIndex - params[i] );
			} else {
				partNorm *= score[i].getNormalizationConstant();
			}
		}
		return partNorm;

	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.scoringFunctions.NormalizableScoringFunction#getEss()
	 */
	public double getEss() {
		return score[0].getEss();
	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.scoringFunctions.ScoringFunction#initializeFunction(int, boolean, de.jstacs.data.Sample[], double[][])
	 */
	public void initializeFunction( int index, boolean freeParams, Sample[] data, double[][] weights ) throws Exception {
		Sample[] part = new Sample[data.length];
		for( int j, i = 0; i < score.length; i++ ) {
			for( j = 0; j < data.length; j++ ) {
				part[j] = data[j].getInfixSample( start[i], partialLength[i] );
			}
			score[i].initializeFunction( index, freeParams, part, weights );
		}
		setParamsStarts();
	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.scoringFunctions.AbstractNormalizableScoringFunction#fromXML(java.lang.StringBuffer)
	 */
	protected void fromXML( StringBuffer rep ) throws NonParsableException {
		StringBuffer xml = XMLParser.extractForTag( rep, getInstanceName() );
		alphabets = (AlphabetContainer)XMLParser.extractStorableForTag( xml, "AlphabetContainer" );
		length = XMLParser.extractIntForTag( xml, "length" );
		score = (NormalizableScoringFunction[])ArrayHandler.cast( XMLParser.extractStorableArrayForTag( xml, "ScoringFunctions" ) );
		setStartsAndLengths( XMLParser.extractIntArrayForTag( xml, "partialLength" ) );
		setParamsStarts();
		prepare();
	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.scoringFunctions.ScoringFunction#getInstanceName()
	 */
	public String getInstanceName() {
		return getClass().getSimpleName();
	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.scoringFunctions.ScoringFunction#getCurrentParameterValues()
	 */
	public double[] getCurrentParameterValues() throws Exception {
		int numPars = this.getNumberOfParameters();
		double[] pars = new double[numPars], help;
		for( int j, k = 0, i = 0; i < score.length; i++ ) {
			help = score[i].getCurrentParameterValues();
			for( j = 0; j < help.length; j++ ) {
				pars[k++] = help[j];
			}
		}
		return pars;
	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.scoringFunctions.ScoringFunction#getLogScore(de.jstacs.data.Sequence, int)
	 */
	public double getLogScore( Sequence seq, int start ) {
		double s = 0;
		for( int i = 0; i < score.length; i++ ) {
			if( isVariable[i] ) {
				s += ( (VariableLengthScoringFunction)score[i] ).getLogScore( seq, start + this.start[i], partialLength[i] );
			} else {
				s += score[i].getLogScore( seq, start + this.start[i] );
			}
		}
		return s;
	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.scoringFunctions.ScoringFunction#getLogScoreAndPartialDerivation(de.jstacs.data.Sequence, int, de.jstacs.utils.IntList, de.jstacs.utils.DoubleList)
	 */
	public double getLogScoreAndPartialDerivation( Sequence seq, int start, IntList indices, DoubleList partialDer ) {
		double s = 0;
		for( int j, i = 0; i < score.length; i++ ) {
			partIList.clear();
			if( isVariable[i] ) {
				s += ( (VariableLengthScoringFunction)score[i] ).getLogScoreAndPartialDerivation( seq,
						start + this.start[i],
						partialLength[i],
						partIList,
						partialDer );
			} else {
				s += score[i].getLogScoreAndPartialDerivation( seq, start + this.start[i], partIList, partialDer );
			}
			for( j = 0; j < partIList.length(); j++ ) {
				indices.add( partIList.get( j ) + this.params[i] );
			}
		}
		return s;
	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.scoringFunctions.ScoringFunction#getNumberOfParameters()
	 */
	public int getNumberOfParameters() {
		if( params == null ) {
			return UNKNOWN;
		} else {
			return params[score.length];
		}
	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.scoringFunctions.AbstractNormalizableScoringFunction#getNumberOfRecommendedStarts()
	 */
	public int getNumberOfRecommendedStarts() {
		int max = score[0].getNumberOfRecommendedStarts();
		for( int i = 1; i < score.length; i++ ) {
			max = Math.max( max, score[i].getNumberOfRecommendedStarts() );
		}
		return max;
	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.scoringFunctions.ScoringFunction#setParameters(double[], int)
	 */
	public void setParameters( double[] params, int start ) {
		for( int i = 0; i < score.length; i++ ) {
			score[i].setParameters( params, start + this.params[i] );
		}
	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.Storable#toXML()
	 */
	public StringBuffer toXML() {
		StringBuffer xml = new StringBuffer( 10000 );
		XMLParser.appendStorableWithTags( xml, alphabets, "AlphabetContainer" );
		XMLParser.appendIntWithTags( xml, length, "length" );
		XMLParser.appendStorableArrayWithTags( xml, score, "ScoringFunctions" );
		XMLParser.appendIntArrayWithTags( xml, partialLength, "partialLength" );
		XMLParser.addTags( xml, getInstanceName() );
		return xml;
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		StringBuffer sb = new StringBuffer( 100000 );
		for( int i = 0; i < score.length; i++ ) {
			sb.append( "sequence part beginning at " + start[i] + " with length " + partialLength[i] + "\n" );
			sb.append( score[i].toString() + "\n" );
		}
		return sb.toString();
	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.scoringFunctions.NormalizableScoringFunction#getLogPriorTerm()
	 */
	public double getLogPriorTerm() {
		double val = 0;
		for( int i = 0; i < score.length; i++ ) {
			val += score[i].getLogPriorTerm();
		}
		return val;
	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.scoringFunctions.NormalizableScoringFunction#addGradientOfLogPriorTerm(double[], int)
	 */
	public void addGradientOfLogPriorTerm( double[] grad, int start ) throws Exception {
		for( int i = 0; i < score.length; i++ ) {
			score[i].addGradientOfLogPriorTerm( grad, start + params[i] );
		}
	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.scoringFunctions.ScoringFunction#isInitialized()
	 */
	public boolean isInitialized() {
		int i = 0;
		while( i < score.length && score[i].isInitialized() ) {
			i++;
		}
		return i == score.length;
	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.scoringFunctions.ScoringFunction#initializeFunctionRandomly(boolean)
	 */
	public void initializeFunctionRandomly( boolean freeParams ) throws Exception {
		for( int i = 0; i < score.length; i++ ) {
			score[i].initializeFunctionRandomly( freeParams );
		}
		setParamsStarts();
	}

	private void fillMotifIndexArray( int motifIndex ) throws IndexOutOfBoundsException {
		motifIndexArray[0] = -1;
		for( int i = 1; i < motifs.length; i++ ) {
			if( score[i - 1] instanceof MotifDiscoverer &&
					( motifIndex >= motifs[i-1] && motifIndex < motifs[i] ) ) {
				motifIndexArray[0] = i - 1;
				motifIndexArray[1] = motifIndex - motifs[motifIndexArray[0]];
				motifIndexArray[2] = score[motifIndexArray[0]] instanceof MutableMotifDiscoverer ? 1 : -1;
				return;
			}
		}
		throw new IndexOutOfBoundsException();
	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.motifDiscovery.MutableMotifDiscoverer#initializeMotif(int, de.jstacs.data.Sample, double[])
	 */
	public void initializeMotif( int motifIndex, Sample data, double[] weights ) throws Exception {
		fillMotifIndexArray( motifIndex );
		if( motifIndexArray[2] >= 0 ) {
			( (MutableMotifDiscoverer)score[motifIndexArray[0]] ).initializeMotif( motifIndexArray[1], data, weights );
			setParamsStarts();
		} else {
			System.out.println( "warning: not possible" );
		}
	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.motifDiscovery.MutableMotifDiscoverer#modifyMotif(int, double[], double[], double[][][][], double[][][][], int, int)
	 */
	public boolean modifyMotif( int motifIndex, double[] weightsLeft, double[] weightsRight, double[][][][] replacementLeft,
			double[][][][] replacementRight, int offsetLeft, int offsetRight ) throws Exception {
		fillMotifIndexArray( motifIndex );
		if( motifIndexArray[2] >= 0 ) {
			boolean b = ( (MutableMotifDiscoverer)score[motifIndexArray[0]] ).modifyMotif( motifIndexArray[1],
					weightsLeft,
					weightsRight,
					replacementLeft,
					replacementRight,
					offsetLeft,
					offsetRight );
			setParamsStarts();
			return b;
		} else {
			return false;
		}
	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.motifDiscovery.MutableMotifDiscoverer#modifyMotif(int, int, int)
	 */
	public boolean modifyMotif( int motifIndex, int offsetLeft, int offsetRight ) throws Exception {
		fillMotifIndexArray( motifIndex );
		if( motifIndexArray[2] >= 0 ) {
			boolean b = ( (MutableMotifDiscoverer)score[motifIndexArray[0]] ).modifyMotif( motifIndexArray[1], offsetLeft, offsetRight );
			setParamsStarts();
			return b;
		} else {
			return false;
		}
	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.motifDiscovery.MutableMotifDiscoverer#determineNotSignificantPositionsFor(int, de.jstacs.data.Sample[], double[][], int)
	 */
	public int[] determineNotSignificantPositionsFor( int motif, Sample[] data, double[][] weights, int classIdx ) {
		fillMotifIndexArray( motif );
		if( motifIndexArray[2] >= 0 ) {
			Sample[] part = new Sample[data.length];
			for( int j = 0; j < part.length; j++ ) {
				part[j] = data[j].getInfixSample( start[motifIndexArray[0]], partialLength[motifIndexArray[0]] );
			}
			return ( (MutableMotifDiscoverer)score[motifIndexArray[0]] ).determineNotSignificantPositionsFor( motifIndexArray[1],
					part,
					weights,
					classIdx );
		} else {
			return new int[]{ 0, 0 };
		}
	}

	private void fillComponentIndexArray( int componentIndex ) {
		for( int i = score.length - 1; i >= 0; i-- ) {
			componentIndexArray[i] = componentIndex / components[i];
			componentIndex %= components[i];
		}
	}

	private void fillMotifIndexArrayForComponent( int componentIndex, int motifIndex ) throws IndexOutOfBoundsException {
		fillComponentIndexArray( componentIndex );
		int i = 0, n;
		for( ; i < score.length; i++ ) {
			if( score[i] instanceof MotifDiscoverer ) {
				n = ( (MotifDiscoverer)score[i] ).getNumberOfMotifsInComponent( componentIndexArray[i] );
				if( motifIndex < n ) {
					motifIndexArray[0] = i;
					motifIndexArray[1] = motifIndex;
					motifIndexArray[2] = motifIndexArray[2] = score[motifIndexArray[0]] instanceof MutableMotifDiscoverer ? 1 : -1;
				} else {
					motifIndex -= n;
				}
			}
		}
		throw new IndexOutOfBoundsException();
	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.motifDiscovery.MotifDiscoverer#getGlobalIndexOfMotifInComponent(int, int)
	 */
	public int getGlobalIndexOfMotifInComponent( int component, int motif ) {
		fillMotifIndexArrayForComponent( component, motif );
		return motifs[motifIndexArray[0]] + ( (MotifDiscoverer)score[motifIndexArray[0]] ).getGlobalIndexOfMotifInComponent( componentIndexArray[motifIndexArray[0]],
						motifIndexArray[1] );
	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.motifDiscovery.MotifDiscoverer#getIndexOfMaximalComponentFor(de.jstacs.data.Sequence)
	 */
	public int getIndexOfMaximalComponentFor( Sequence sequence ) throws Exception {
		int c = 0, old = 1, idx;
		for( int i = 0; i < score.length; i++ ) {
			if( score[i] instanceof AbstractMixtureScoringFunction ) {
				idx = ( (AbstractMixtureScoringFunction)score[i] ).getIndexOfMaximalComponentFor( sequence, start[i] );
			} else if( score[i] instanceof MotifDiscoverer ) {
				idx = ( (MotifDiscoverer)score[i] ).getIndexOfMaximalComponentFor( sequence.getSubSequence( score[i].getAlphabetContainer(),
						start[i],
						partialLength[i] ) );
			} else {
				idx = 0;
			}
			c += old * idx;
			old = components[i];
		}
		return c;
	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.motifDiscovery.MotifDiscoverer#getMotifLength(int)
	 */
	public int getMotifLength( int motif ) {
		fillMotifIndexArray( motif );
		return ( (MotifDiscoverer)score[motifIndexArray[0]] ).getMotifLength( motifIndexArray[1] );
	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.motifDiscovery.MotifDiscoverer#getNumberOfComponents()
	 */
	public int getNumberOfComponents() {
		return components[score.length - 1];
	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.motifDiscovery.MotifDiscoverer#getNumberOfMotifs()
	 */
	public int getNumberOfMotifs() {
		return motifs[score.length];
	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.motifDiscovery.MotifDiscoverer#getNumberOfMotifsInComponent(int)
	 */
	public int getNumberOfMotifsInComponent( int component ) {
		fillComponentIndexArray( component );
		int i = 0, anz = 0;
		for( ; i < score.length; i++ ) {
			if( score[i] instanceof MotifDiscoverer ) {
				anz += ( (MotifDiscoverer)score[i] ).getNumberOfMotifsInComponent( componentIndexArray[i] );
			}
		}
		return anz;
	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.motifDiscovery.MotifDiscoverer#getProfileOfScoresFor(int, int, de.jstacs.data.Sequence, int, de.jstacs.motifDiscovery.MotifDiscoverer.KindOfProfile)
	 */
	public double[] getProfileOfScoresFor( int component, int motif, Sequence sequence, int startpos, KindOfProfile dist ) throws Exception {
		fillMotifIndexArrayForComponent( component, motif );
		int w = ( (MotifDiscoverer)score[motifIndexArray[0]] ).getMotifLength( motifIndexArray[1] );
		double[] erg = new double[length - w + 1];
		Arrays.fill( erg, Double.NEGATIVE_INFINITY );
		double[] part = ( (MotifDiscoverer)score[motifIndexArray[0]] ).getProfileOfScoresFor( componentIndexArray[motifIndexArray[0]],
				motifIndexArray[1],
				sequence,
				start[motifIndexArray[0]] + startpos,
				dist );
		System.arraycopy( part, 0, erg, start[motifIndexArray[0]], partialLength[motifIndexArray[0]] - w + 1 );
		return erg;
	}

	/*
	 * (non-Javadoc)
	 * @see de.jstacs.motifDiscovery.MotifDiscoverer#getStrandFor(int, int, de.jstacs.data.Sequence, int)
	 */
	public Strand getStrandFor( int component, int motif, Sequence sequence, int startpos ) throws Exception {
		fillMotifIndexArrayForComponent( component, motif );
		return ( (MotifDiscoverer)score[motifIndexArray[0]] ).getStrandFor( componentIndexArray[motifIndexArray[0]],
				motifIndexArray[1],
				sequence,
				startpos );
	}
}
