/*
 * This file is part of Jstacs.
 * 
 * Jstacs is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * Jstacs is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * Jstacs. If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.classifier.scoringFunctionBased.cll;

import de.jstacs.NonParsableException;
import de.jstacs.classifier.scoringFunctionBased.ScoreClassifier;
import de.jstacs.classifier.scoringFunctionBased.logPrior.DoesNothingLogPrior;
import de.jstacs.classifier.scoringFunctionBased.logPrior.LogPrior;
import de.jstacs.data.Sample;
import de.jstacs.io.XMLParser;
import de.jstacs.scoringFunctions.ScoringFunction;

/**
 * This class implements the conditional log likelihood (CLL) classifier.
 * 
 * @author Jens Keilwagen, Jan Grau
 */
public class CLLClassifier extends ScoreClassifier {

	/**
	 * The prior that is used in this instance.
	 */
	protected LogPrior prior;

	/**
	 * The default constructor that creates a new {@link CLLClassifier} from a
	 * given parameter set and {@link ScoringFunction}s for the classes.
	 * 
	 * @param params
	 *            the parameter set for the classifier
	 * @param score
	 *            the {@link ScoringFunction}s for the classes
	 * 
	 * @throws CloneNotSupportedException
	 *             if at least one {@link ScoringFunction} could not be cloned
	 * 
	 * @see CLLClassifier#CLLClassifier(CLLClassifierParameterSet, LogPrior,
	 *      ScoringFunction...)
	 */
	public CLLClassifier( CLLClassifierParameterSet params, ScoringFunction... score ) throws CloneNotSupportedException {
		this( params, null, score );
	}

	/**
	 * The default constructor that creates a new {@link CLLClassifier} from a
	 * given parameter set, a prior and {@link ScoringFunction}s for the
	 * classes.
	 * 
	 * @param params
	 *            the parameter set for the classifier
	 * @param prior
	 *            the prior that shall be used
	 * @param score
	 *            the {@link ScoringFunction}s for the classes
	 * 
	 * @throws CloneNotSupportedException
	 *             if at least one {@link ScoringFunction} could not be cloned
	 * 
	 * @see ScoreClassifier#ScoreClassifier(de.jstacs.classifier.scoringFunctionBased.ScoreClassifierParameterSet,
	 *      ScoringFunction...)
	 * @see CLLClassifier#setPrior(LogPrior)
	 */
	public CLLClassifier( CLLClassifierParameterSet params, LogPrior prior, ScoringFunction... score ) throws CloneNotSupportedException {
		super( params, score );
		setPrior( prior );
	}

	/* (non-Javadoc)
	 * @see de.jstacs.classifier.scoringFunctionBased.ScoreClassifier#clone()
	 */
	@Override
	public CLLClassifier clone() throws CloneNotSupportedException {
		CLLClassifier clone = (CLLClassifier)super.clone();
		clone.prior = prior.getNewInstance();
		return clone;
	}

	/**
	 * The standard constructor for the interface {@link de.jstacs.Storable}.
	 * Constructs a {@link CLLClassifier} out of its XML representation.
	 * 
	 * @param xml
	 *            the XML representation as {@link StringBuffer}
	 * 
	 * @throws NonParsableException
	 *             if the {@link CLLClassifier} could not be reconstructed out
	 *             of the XML representation (the {@link StringBuffer} could not
	 *             be parsed)
	 * 
	 * @see ScoreClassifier#ScoreClassifier(StringBuffer)
	 * @see de.jstacs.Storable
	 */
	public CLLClassifier( StringBuffer xml ) throws NonParsableException {
		super( xml );
	}

	/* (non-Javadoc)
	 * @see de.jstacs.classifier.scoringFunctionBased.ScoreClassifier#getFunction(de.jstacs.data.Sample[], double[][])
	 */
	@Override
	protected NormConditionalLogLikelihood getFunction( Sample[] data, double[][] weights ) throws Exception {
		CLLClassifierParameterSet p = (CLLClassifierParameterSet)params;
		return new NormConditionalLogLikelihood( score, data, weights, prior, p.shouldBeNormalized(), p.useOnlyFreeParameter() );
	}

	/**
	 * This method sets a new prior that should be used for optimization. Since
	 * it could not be ensured that the classifier is optimal now
	 * {@link ScoreClassifier#hasBeenOptimized()} will return <code>false</code>
	 * after invoking this method.
	 * 
	 * @param prior
	 *            the new prior
	 * 
	 * @see ScoreClassifier#hasBeenOptimized()
	 */
	public void setPrior( LogPrior prior ) {
		if( prior != null ) {
			this.prior = prior;
		} else {
			this.prior = DoesNothingLogPrior.defaultInstance;
		}
		hasBeenOptimized = false;
	}

	private static final String XML_TAG = "cll-classifier";

	/* (non-Javadoc)
	 * @see de.jstacs.classifier.scoringFunctionBased.ScoreClassifier#getXMLTag()
	 */
	@Override
	protected String getXMLTag() {
		return XML_TAG;
	}

	/* (non-Javadoc)
	 * @see de.jstacs.classifier.scoringFunctionBased.ScoreClassifier#getFurtherClassifierInfos()
	 */
	@Override
	protected StringBuffer getFurtherClassifierInfos() {
		StringBuffer xml = super.getFurtherClassifierInfos();
		if( !( prior instanceof DoesNothingLogPrior ) ) {
			StringBuffer pr = new StringBuffer( 1000 );
			pr.append( "<prior>\n" );
			XMLParser.appendStringWithTags( pr, prior.getClass().getName(), "className" );
			pr.append( prior.toXML() );
			pr.append( "\t</prior>\n" );
			xml.append( pr );
		}
		return xml;
	}

	/* (non-Javadoc)
	 * @see de.jstacs.classifier.scoringFunctionBased.ScoreClassifier#extractFurtherClassifierInfosFromXML(java.lang.StringBuffer)
	 */
	@Override
	protected void extractFurtherClassifierInfosFromXML( StringBuffer xml ) throws NonParsableException {
		super.extractFurtherClassifierInfosFromXML( xml );
		StringBuffer pr = XMLParser.extractForTag( xml, "prior" );
		if( pr != null ) {
			String className = XMLParser.extractStringForTag( pr, "className" );
			try {
				prior = (LogPrior)Class.forName( className ).getConstructor( new Class[]{ StringBuffer.class } ).newInstance( pr );
			} catch ( NoSuchMethodException e ) {
				NonParsableException n = new NonParsableException( "You must provide a constructor " + className + "(StringBuffer)." );
				n.setStackTrace( e.getStackTrace() );
				throw n;
			} catch ( Exception e ) {
				NonParsableException n = new NonParsableException( "problem at " + className + ": " + e.getMessage() );
				n.setStackTrace( e.getStackTrace() );
				throw n;
			}
		} else {
			prior = DoesNothingLogPrior.defaultInstance;
		}
		try {
			prior.set( ( (CLLClassifierParameterSet)params ).useOnlyFreeParameter(), score );
		} catch ( Exception e ) {
			NonParsableException n = new NonParsableException( "problem when setting the kind of parameter: " + e.getMessage() );
			n.setStackTrace( e.getStackTrace() );
			throw n;
		}
	}

	/**
	 * This method creates an array of {@link CLLClassifier}s by using the
	 * cross-product of the given {@link ScoringFunction}s.
	 * 
	 * @param params
	 *            the parameters that will be used in all classifiers
	 * @param prior
	 *            the prior that will be used in all classifiers
	 * @param functions
	 *            the {@link ScoringFunction}s
	 *            <ol>
	 *            <li> <code>functions[i]</code> are the {@link ScoringFunction}s
	 *            that can be used for class i
	 *            <li> <code>functions.length</code> has to be at least 2
	 *            </ol>
	 * 
	 * @return an array of {@link CLLClassifier}s
	 * 
	 * @throws CloneNotSupportedException
	 *             if some item could not be cloned
	 */
	public static CLLClassifier[] create( CLLClassifierParameterSet params, LogPrior prior, ScoringFunction[]... functions ) throws CloneNotSupportedException {
		int anz = 1, counter2;
		int[] current = new int[functions.length], max = new int[functions.length];
		ScoringFunction[] sf = new ScoringFunction[functions.length];
		for( int counter1 = 0; counter1 < functions.length; counter1++ ) {
			anz *= functions[counter1].length;
			max[counter1] = functions[counter1].length - 1;
		}

		CLLClassifier[] erg = new CLLClassifier[anz];
		anz = sf.length - 1;

		for( int counter1 = 0; counter1 < erg.length; counter1++ ) {
			for( counter2 = 0; counter2 < sf.length; counter2++ ) {
				sf[counter2] = functions[counter2][current[counter2]];
			}

			erg[counter1] = new CLLClassifier( params, prior, sf );

			counter2 = 0;
			while( counter2 < anz && current[counter2] == max[counter2] ) {
				current[counter2++] = 0;
			}
			current[counter2]++;
		}

		return erg;
	}

	/* (non-Javadoc)
	 * @see de.jstacs.classifier.scoringFunctionBased.ScoreClassifier#getInstanceName()
	 */
	@Override
	public String getInstanceName() {
		return super.getInstanceName() + ( prior == null || prior == DoesNothingLogPrior.defaultInstance ? ""
																										: " with " + prior.getInstanceName() );
	}
}
