/*
 * This file is part of Jstacs.
 * 
 * Jstacs is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * Jstacs is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * Jstacs. If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.io;

import java.lang.reflect.Array;
import java.lang.reflect.Method;
import java.util.HashSet;
import java.util.LinkedList;

/**
 * This class handles arrays with elements of generic type and enables the user
 * to cast and clone arrays easily.
 * 
 * @author Jens Keilwagen
 */
public final class ArrayHandler {

	/**
	 * This method returns the deepest class in the class hierarchy that is the
	 * class or a superclass of all instances in the array.
	 * 
	 * @param <T>
	 *            the type of the array elements
	 * @param o
	 *            the array
	 * 
	 * @return the superclass of all elements of the given array
	 */
	public static <T> Class getSuperClassOf( T... o ) {
		Class current;
		LinkedList<Class> classHierarchy = new LinkedList<Class>();
		HashSet<Class> hash = new HashSet<Class>();
		for( int i = 0; i < o.length; i++ ) {
			if( o[i] != null ) {
				current = o[i].getClass();
				if( classHierarchy.size() == 0 ) {
					while( current != Object.class ) {
						classHierarchy.add( current );
						hash.add( current );
						current = current.getSuperclass();
					}
					classHierarchy.add( current );
					hash.add( current );
				} else {
					while( !hash.contains( current ) ) {
						current = current.getSuperclass();
					}
					while( classHierarchy.get( 0 ) != current ) {
						hash.remove( classHierarchy.remove( 0 ) );
					}
					if( classHierarchy.size() == 1 ) {
						break;
					}
				}
			}
		}
		if( classHierarchy.size() > 0 ) {
			return classHierarchy.get( 0 );
		} else {
			return o.getClass().getComponentType();
		}
	}

	/**
	 * This method creates a new array of the superclass of all elements of the
	 * given array and copies the elements. The order of the elements is not
	 * changed.
	 * 
	 * <br>
	 * <br>
	 * 
	 * Here is an example that demonstrates what can be done:<br>
	 * <br>
	 * <code>
	 * Object[] o = { new UniformModel( alphabetContainer ), new UniformModel( alphabetContainer ) };<br>
	 * AbstractModel[] a = (AbstractModel[]) ArrayHandler.cast( o );<br>
	 * </code> <br>
	 * This should work fine, while<br>
	 * 
	 * <code>AbstractModel[] a = (AbstractModel[]) o;</code><br>
	 * 
	 * will not.
	 * 
	 * @param <T>
	 *            the type of the array elements
	 * @param o
	 *            the array
	 * 
	 * @return the casted array with the copied elements
	 * 
	 * @see ArrayHandler#getSuperClassOf(Object[])
	 * @see XMLParser#extractStorableArrayForTag(StringBuffer, String, String)
	 */
	@SuppressWarnings( "unchecked" )
	public static <T> T[] cast( T... o ) {
		T[] res = (T[])Array.newInstance( getSuperClassOf( o ), o.length );
		for( int i = 0; i < o.length; i++ ) {
			res[i] = o[i];
		}
		return res;
	}

	/**
	 * This method returns a deep copy of an array, i.e. for each element of the
	 * array the clone method will be invoked.
	 * 
	 * @param <T>
	 *            the type of the array elements that implements
	 *            {@link Cloneable}
	 * @param t
	 *            the array
	 * 
	 * @return a deep copy of the given array
	 * 
	 * @throws CloneNotSupportedException
	 *             if an element could not be cloned
	 * 
	 * @see Cloneable
	 */
	@SuppressWarnings( "unchecked" )
	public static <T extends Cloneable> T[] clone( T... t ) throws CloneNotSupportedException {
		if( t != null && t.length > 0 ) {
			Class c = t.getClass().getComponentType();
			T[] res = (T[])Array.newInstance( c, t.length );
			try {
				Method cloneMethod = c.getMethod( "clone" );
				for( int i = 0; i < t.length; i++ ) {
					res[i] = (T)cloneMethod.invoke( t[i] );
				}
			} catch ( Exception e ) {
				CloneNotSupportedException cnse = new CloneNotSupportedException( e.getMessage() );
				cnse.setStackTrace( e.getStackTrace() );
				throw cnse;
			}
			return res;
		} else {
			return null;
		}
	}
}
