package de.jstacs.scoringFunctions.mix.motifSearch;

import java.util.Arrays;

import de.jstacs.NonParsableException;
import de.jstacs.data.AlphabetContainer;
import de.jstacs.data.Sample;
import de.jstacs.data.Sequence;
import de.jstacs.data.WrongLengthException;
import de.jstacs.data.alphabets.DiscreteAlphabet;
import de.jstacs.data.sequences.IntSequence;
import de.jstacs.data.sequences.annotation.StrandedLocatedSequenceAnnotationWithLength.Strand;
import de.jstacs.io.XMLParser;
import de.jstacs.motifDiscovery.Mutable;
import de.jstacs.motifDiscovery.MutableMotifDiscoverer;
import de.jstacs.scoringFunctions.NormalizableScoringFunction;
import de.jstacs.scoringFunctions.NormalizedScoringFunction;
import de.jstacs.scoringFunctions.ScoringFunction;
import de.jstacs.scoringFunctions.homogeneous.HomogeneousScoringFunction;
import de.jstacs.scoringFunctions.mix.AbstractMixtureScoringFunction;
import de.jstacs.scoringFunctions.mix.StrandScoringFunction;
import de.jstacs.utils.DoubleList;
import de.jstacs.utils.IntList;
import de.jstacs.utils.Normalisation;

/**
 * This class handles mixtures with at least one hidden motif.
 * 
 * @author Jens Keilwagen
 */
public class HiddenMotifsMixture extends AbstractMixtureScoringFunction implements MutableMotifDiscoverer
{
	/**
	 * This constant indicates that in each sequence has one binding site of a motif instance (similar to OOPS).
	 */
	public static final boolean CONTAINS_ALWAYS_A_MOTIF = true;

	/**
	 * This constant indicates that a sequence possibly has one binding site of a motif instance (similar to ZOOPS).
	 */
	public static final boolean CONTAINS_SOMETIMES_A_MOTIF = false;

	private double[][] simpleScore;

	private double[] bgHelp;

	private int[] anz, currentPos;

	private double ess, sign;

	private int bgIndex;

	private boolean type;
	
	/**
	 * Indicates whether the old parameters or new one should be used.
	 */
	private boolean plugInBg;

	private HomogeneousScoringFunction bg;

	private static NormalizableScoringFunction[] getNormalizableScoringFunctions( int length,
			HomogeneousScoringFunction bg, NormalizableScoringFunction[] motif, DurationScoringFunction[] posPrior )
	{
		int m = motif.length;
		if( m == 0 )
		{
			throw new IllegalArgumentException(	"Please insert at least one ScoringFunction for a motif." );
		}
		NormalizableScoringFunction[] erg = new NormalizableScoringFunction[2 * m + 1];
		if( posPrior == null )
		{
			posPrior = new DurationScoringFunction[m];
		} else if ( m != posPrior.length ) {
			throw new IllegalArgumentException(	"The number of motifs and durations has to be the same." );
		}
		int i = 0, l = -1;
		AlphabetContainer seqABC = bg.getAlphabetContainer();
		if( !seqABC.isSimple() )
		{
			throw new IllegalArgumentException(
					"The AlphabetContainer of the motif and background models has to be simple." );
		}
		AlphabetContainer posABC = null;
		while( i < m )
		{
			if( !seqABC.checkConsistency( motif[i].getAlphabetContainer() ) )
			{
				throw new IllegalArgumentException( "The " + i
						+ "-th motif model is not correct. Check the AlphabetContainer." );
			}
			erg[2 * i] = motif[i];
			if( posPrior[i] != null )
			{
				if( 0 > posPrior[i].getMin() || posPrior[i].getMax() > length - motif[i].getLength() )
				{
					throw new IllegalArgumentException( "The " + i
							+ "-th model for the positional information is not correct. Check the AlphabetContainer." );
				}
				erg[2 * i + 1] = posPrior[i];
			}
			else
			{
				if( posABC == null || motif[i].getLength() != l )
				{
					posABC = new AlphabetContainer( new DiscreteAlphabet( 0, length - motif[i].getLength() ) );
				}
				erg[2 * i + 1] = new UniformDurationScoringFunction( 0, length - motif[i].getLength() );
				l = motif[i].getLength();
			}
			i++;
		}
		if( !seqABC.checkConsistency( bg.getAlphabetContainer() ) )
		{
			throw new IllegalArgumentException( "The background model is not correct. Check the AlphabetContainer." );
		}
		erg[2 * i] = bg;
		return erg;
	}
	
	/**
	 * The default significance level that is used in {@link Mutable#determineNotSignificantPositions(double, double[], double[], double[][][][], double[][][][], double)}.
	 */
	public static final double DEFAULT_SIGN = 0.3;
	

	/**
	 * This constructor creates an instance of {@link HiddenMotifsMixture} that is either an OOPS or a ZOOPS model depending on the chosen <code>type</code>.
	 * This constructor uses the {@link HiddenMotifsMixture#DEFAULT_SIGN} as significance level.
	 * 
	 * @param type the type of hidden motifs, either {@link HiddenMotifsMixture#CONTAINS_ALWAYS_A_MOTIF} or {@link HiddenMotifsMixture#CONTAINS_SOMETIMES_A_MOTIF}
	 * @param length the length of the modeled sequences (e.g. 500)
	 * @param starts the number of recommended starts
	 * @param plugIn a switch whether to use plug-in or randomly chosen parameter when using {@link ScoringFunction#initializeFunction(int, boolean, Sample[], double[][])}
	 * @param bg the {@link ScoringFunction} for the overall background (i.e. flanking sequence that does not contain a motif)
	 * @param motif the {@link ScoringFunction} for the motif
	 * @param posPrior the {@link ScoringFunction} for the position
	 * @param plugInBg a switch whether to plug in the (old = last) parameters of the background model, if <code>false</code> the background model is initialized again
	 * 
	 * @throws Exception if something went wrong (e.g. the {@link AlphabetContainer} is not simple, ...)
	 */
	public HiddenMotifsMixture( boolean type, int length, int starts, boolean plugIn, HomogeneousScoringFunction bg,
			NormalizableScoringFunction motif, DurationScoringFunction posPrior, boolean plugInBg ) throws Exception
	{
		this( type, length, starts, plugIn, bg, motif, posPrior, DEFAULT_SIGN, plugInBg );
	}

	/**
	 * This constructor creates an instance of {@link HiddenMotifsMixture} that is either an OOPS or a ZOOPS model depending on the chosen <code>type</code>.
	 * 
	 * @param type the type of hidden motifs, either {@link HiddenMotifsMixture#CONTAINS_ALWAYS_A_MOTIF} or {@link HiddenMotifsMixture#CONTAINS_SOMETIMES_A_MOTIF}
	 * @param length the length of the modeled sequences (e.g. 500)
	 * @param starts the number of recommended starts
	 * @param plugIn a switch whether to use plug-in or randomly chosen parameter when using {@link ScoringFunction#initializeFunction(int, boolean, Sample[], double[][])}
	 * @param bg the {@link ScoringFunction} for the overall background (i.e. flanking sequence that does not contain a motif)
	 * @param motif the {@link ScoringFunction} for the motif
	 * @param posPrior the {@link ScoringFunction} for the position
	 * @param sign the significance level to determine the insignificant border positions of a motif
	 * @param plugInBg a switch whether to plug in the (old = last) parameters of the background model, if <code>false</code> the background model is initialized again
	 * 
	 * @throws Exception if something went wrong (e.g. the {@link AlphabetContainer} is not simple, ...)
	 */
	public HiddenMotifsMixture( boolean type, int length, int starts, boolean plugIn, HomogeneousScoringFunction bg,
			NormalizableScoringFunction motif, DurationScoringFunction posPrior, double sign, boolean plugInBg ) throws Exception
	{
		this( type, length, starts, plugIn, bg, new NormalizableScoringFunction[]{ motif },
				new DurationScoringFunction[]{ posPrior }, sign, plugInBg );
	}

	/**
	 * This constructor creates an instance of {@link HiddenMotifsMixture} that allows to have one site of the specified motifs in a {@link Sequence}.
	 * A pair of entries of <code>motif</code> and <code>pos</code> model a specific motif.
	 * This constructor uses the {@link HiddenMotifsMixture#DEFAULT_SIGN} as significance level.
	 * 
	 * @param type the type of hidden motifs, either {@link HiddenMotifsMixture#CONTAINS_ALWAYS_A_MOTIF} or {@link HiddenMotifsMixture#CONTAINS_SOMETIMES_A_MOTIF}
	 * @param length the length of the modeled sequences (e.g. 500)
	 * @param starts the number of recommended starts
	 * @param plugIn a switch whether to use plug-in or randomly chosen parameter when using {@link ScoringFunction#initializeFunction(int, boolean, Sample[], double[][])}
	 * @param bg the {@link ScoringFunction} for the overall background (i.e. flanking sequence that does not contain a motif)
	 * @param motif the {@link ScoringFunction}s for the motif
	 * @param posPrior the {@link ScoringFunction}s for the position
	 * @param plugInBg a switch whether to plug in the (old = last) parameters of the background model, if <code>false</code> the background model is initialized again
	 * 
	 * @throws Exception if something went wrong (e.g. the {@link AlphabetContainer} is not simple, ...)
	 */
	public HiddenMotifsMixture( boolean type, int length, int starts, boolean plugIn, HomogeneousScoringFunction bg,
			NormalizableScoringFunction[] motif, DurationScoringFunction[] posPrior, boolean plugInBg )
			throws Exception
	{
		this( type, length, starts, plugIn, bg, motif, posPrior, DEFAULT_SIGN, plugInBg );
	}
	
	/**
	 * This constructor creates an instance of {@link HiddenMotifsMixture} that allows to have one site of the specified motifs in a {@link Sequence}.
	 * 
	 * @param type the type of hidden motifs, either {@link HiddenMotifsMixture#CONTAINS_ALWAYS_A_MOTIF} or {@link HiddenMotifsMixture#CONTAINS_SOMETIMES_A_MOTIF}
	 * @param length the length of the modeled sequences (e.g. 500)
	 * @param starts the number of recommended starts
	 * @param plugIn a switch whether to use plug-in or randomly chosen parameter when using {@link ScoringFunction#initializeFunction(int, boolean, Sample[], double[][])}
	 * @param bg the {@link ScoringFunction} for the overall background (i.e. flanking sequence that does not contain a motif)
	 * @param motif the {@link ScoringFunction}s for the sequence motif
	 * @param posPrior the {@link ScoringFunction}s for the position of the the sequence motifs
	 * @param sign the significance level to determine the insignificant border positions of a motif
	 * @param plugInBg a switch whether to plug in the parameters of the background model in some way, if <code>false</code> the background model is initialized uniformly
	 * 
	 * @throws Exception if something went wrong (e.g. the {@link AlphabetContainer} is not simple, ...)
	 */
	public HiddenMotifsMixture( boolean type, int length, int starts, boolean plugIn, HomogeneousScoringFunction bg,
			NormalizableScoringFunction[] motif, DurationScoringFunction[] posPrior, double sign, boolean plugInBg )
			throws Exception
	{
		super( length, starts, motif.length + (type == CONTAINS_ALWAYS_A_MOTIF ? 0 : 1), true, plugIn,
				getNormalizableScoringFunctions( length, bg, motif, posPrior ) );
		if( !alphabets.isSimple() )
		{
			throw new IllegalArgumentException( "The AlphabetContainer has to be simple." );
		}
		this.type = type;
		this.plugInBg = plugInBg;
		initObject();
		this.sign = sign;
		setHyperParametersOfBackgroundModel();
		computeLogGammaSum();
	}

	/**
	 * This is the constructor for the interface {@link de.jstacs.Storable}.
	 * Recreates a new {@link HiddenMotifsMixture} out of a
	 * {@link StringBuffer} as returned by {@link #toXML()}.
	 * 
	 * @param source
	 *            the XML representation as {@link StringBuffer}
	 * 
	 * @throws NonParsableException
	 *             if the representation could not be parsed
	 */
	public HiddenMotifsMixture( StringBuffer source ) throws NonParsableException
	{
		super( source );
		initObject();
	}

	private void initObject()
	{
		bgIndex = function.length - 1;
		bg = (HomogeneousScoringFunction) function[bgIndex];

		simpleScore = new double[(function.length-1)/2][];
		if( type == CONTAINS_ALWAYS_A_MOTIF )
		{
			ess = 0;
		}
		else
		{
			ess = bg.getEss();
		}
		for( int i = 0; i < bgIndex; i += 2 )
		{
			ess += function[i].getEss();
		}
		createSimpleScore();
		initBgHelp();
		anz = new int[componentScore.length + 1];
		currentPos = new int[1];
	}

	private void createSimpleScore()
	{
		for( int i = 0; i < bgIndex; i += 2 )
		{
			simpleScore[i / 2] = new double[(int) function[i + 1].getAlphabetContainer().getAlphabetLengthAt( 0 )];
		}
	}
	
	private void setHyperParametersOfBackgroundModel() throws Exception
	{
		int[] len = new int[length+1], l = new int[1];
		int i = 0, anz, m;
		for( ; i <= length; i++ )
		{
			len[i] = i;
		}
		double[] weight = new double[len.length];
		double w;
		boolean okay;
		DurationScoringFunction dur;
		for( i = 0; i < bgIndex; i += 2 )
		{
			m = length - function[i].getLength();
			dur = (DurationScoringFunction) function[i+1];
			
			//determine how many possibilities can be used for the current motif
			anz = 0;
			dur.reset();
			do
			{
				dur.getInternalPosition(l);
				if( l[0] <= m )
				{
					anz++;
					okay = dur.next();
				}
				else
				{
					okay = false;
				}
			}
			while( okay );
			
			//determine the weight for each possible length
			w = function[i].getEss() / (double) anz;
			
			//create the statistic
			dur.reset();
			do
			{
				dur.getInternalPosition(l);
				if( l[0] <= m )
				{
					// add the weight for the sequence before and after the motif
					weight[l[0]] += w;
					weight[m - l[0]] += w;
					okay = dur.next();
				}
				else
				{
					okay = false;
				}
			}
			while( okay );			
		}
		
		// add the weight if no motif is in the sequence
		if( type == CONTAINS_SOMETIMES_A_MOTIF )
		{
			weight[length] += bg.getEss();
		}
		
		bg.setStatisticForHyperparameters( len, weight );
	}
	
	public HiddenMotifsMixture clone() throws CloneNotSupportedException
	{
		HiddenMotifsMixture clone = (HiddenMotifsMixture) super.clone();
		clone.simpleScore = new double[simpleScore.length][];
		for( int i = 0; i < simpleScore.length; i++ )
		{
			clone.simpleScore[i] = new double[simpleScore[i].length];
		}
		clone.bg = (HomogeneousScoringFunction) clone.function[bgIndex];
		clone.bgHelp = bgHelp.clone();
		clone.anz = anz.clone();
		clone.currentPos = currentPos.clone();
		return clone;
	}

	public double getHyperparameterForHiddenParameter( int index )
	{
		return function[2 * index].getEss();
	}

	protected double getNormalizationConstantForComponent( int i )
	{
		i *= 2;
		if( i >= 0 && i < bgIndex )
		{
			int l = length - function[i].getLength();
			//System.out.println( function[i].getNormalizationConstant() + " * " + function[i + 1].getNormalizationConstant() +	 " * " + bg.getNormalizationConstant( l ));
			return function[i].getNormalizationConstant() * function[i + 1].getNormalizationConstant()
					* bg.getNormalizationConstant( l );
		}
		else if( i == bgIndex )
		{
			return bg.getNormalizationConstant();
		}
		else
		{
			throw new IndexOutOfBoundsException();
		}

	}

	public double getPartialNormalizationConstant( int parameterIndex ) throws Exception
	{
		if( isNormalized )
		{
			return 0;
		}
		else
		{
			if( norm < 0 )
			{
				precomputeNorm();
			}
			int[] ind = getIndices( parameterIndex );
			double res;
			if( ind[0] == function.length )
			{
				res = partNorm[ind[1]];
			}
			else if( ind[0] == bgIndex )
			{
				res = 0;
				for( int l, i = 0; i < bgIndex; i += 2 )
				{
					l = length - function[i].getLength();
					res += (hiddenPotential[i / 2] * function[i].getNormalizationConstant()
							* function[i + 1].getNormalizationConstant() * bg.getPartialNormalizationConstant( ind[1], l ));
				}
				if( type == CONTAINS_SOMETIMES_A_MOTIF )
				{
					res += (hiddenPotential[bgIndex / 2] * bg.getPartialNormalizationConstant( ind[1] ));
				}
			}
			else
			{
				if( ind[0] % 2 == 0 )
				{
					int l = length - function[ind[0]].getLength();
					res = hiddenPotential[ind[0] / 2] * function[ind[0]].getPartialNormalizationConstant( ind[1] )
							* function[ind[0] + 1].getNormalizationConstant() * bg.getNormalizationConstant( l );
				}
				else
				{
					int l = length - function[ind[0] - 1].getLength();
					res = hiddenPotential[ind[0] / 2] * function[ind[0] - 1].getNormalizationConstant()
							* function[ind[0]].getPartialNormalizationConstant( ind[1] ) * bg.getNormalizationConstant( l );
				}
			}
			
			//System.out.println( parameterIndex + "\t" + Arrays.toString(ind) + "\t" + res );
			
			return res;
		}
	}

	public double getEss()
	{
		return ess;
	}
	
	protected void initializeUsingPlugIn( int index, boolean freeParams, Sample[] data, double[][] weights ) throws Exception
	{
		double[] old = null;
		if( plugInBg ) {
			old = bg.getCurrentParameterValues();
		}
		bg.initializeFunction( index, freeParams, data, weights );
		initializeHiddenUniformly();	
		
		int num = getNumberOfMotifs(), a = (int) alphabets.getAlphabetLengthAt(0), s, p, l;
		double d = 0.1/(a-1), f = (1d-a*d)/(a*d), h;
		Sequence init;
		
		int i = 0, b, c, ml;
		Sequence[] seq;
		Sequence[] sites;
		Sequence[] start;
		AlphabetContainer posCon = null;
		DiscreteAlphabet pos = null;
		Sequence[] posSeq = null;
		
		double[][] siteWeights = new double[1][data[index].getNumberOfElements()];
		for( int motif = 0; motif < num; motif++ )
		{
			//determine init sequence
			((DurationScoringFunction) function[2*motif+1]).initializeUniformly();
			l = function[2*motif].getLength();
			s = r.nextInt( data[index].getNumberOfElements() );
			init = data[index].getElementAt( s );
			p = r.nextInt( init.getLength() - l + 1 );
			init = init.getSubSequence( p, l );
			//System.out.println( s + "\t" + p + "\t" + init );
			h = f * function[2*motif].getEss(); 
			function[2*motif].initializeFunction( 0, freeParams, new Sample[]{ new Sample( "", init ) }, new double[][]{{h}} );

			//prepare weights
			ml = function[2*motif].getLength();
			seq = data[index].getAllElements();
			sites = new Sequence[seq.length];

			start = new Sequence[seq.length];
			if( posCon == null || !posCon.checkConsistency( function[2*motif+1].getAlphabetContainer() ) ) {
				posCon = function[2*motif+1].getAlphabetContainer();
				pos = (DiscreteAlphabet) posCon.getAlphabetAt( 0 );
				posSeq = new Sequence[(int) pos.length()];
				for( i = 0; i < posSeq.length; i++ ) {
					posSeq[i] = new IntSequence( posCon, i );
				}
			}
			
			//deterimine best matching subsequences
			//System.out.println( function[2*motif+1].toString() );
			for( i = 0; i < seq.length; i++ ) {
				fillComponentScoreOf(motif, seq[i], 0);
				b = 0;
				for( c = 1; c < simpleScore[motif].length; c++ ) {
					if( simpleScore[motif][c] > simpleScore[motif][b] ) {
						b = c;
					}
				}
				if( start[i] == null || start[i].discreteVal(0) != b ) {
					sites[i] = seq[i].getSubSequence( Integer.parseInt( pos.getSymbolAt(b) ), ml );
					start[i] = posSeq[b];
					if( getStrandFor( motif, 0, seq[i], Integer.parseInt( pos.getSymbolAt(b) ) ) == Strand.REVERSE ) {
						sites[i] = sites[i].reverseComplement();
					}
				}
				//System.out.print( b + ", " );
				//System.out.println( simpleScore[motif][b] );
			}
			
			h /= siteWeights.length;
			if( weights == null || weights[index] == null ) {
				Arrays.fill( siteWeights[0], h );
			} else {
				for( i = 0; i < seq.length; i++ ) {
					siteWeights[0][i] = weights[index][i] * h;
				}
			}
			
			//initialize
			initializeMotif( motif, new Sample( "sites", sites ), weights == null ? null : weights[index] ); //siteWeights[0] );
			function[2*motif+1].initializeFunction(0, freeParams, new Sample[]{ new Sample( "position", start ) }, weights ); //siteWeights );
		}
		
		c = getNumberOfComponents();
		double[] count = new double[c];
		double w = 1;
		for( int n = 0; n < data[index].getNumberOfElements(); n++ )
		{
			if( weights != null && weights[index] != null ) {
				w = weights[index][n];
			}
			fillComponentScores( data[index].getElementAt(n), 0 );
			Normalisation.logSumNormalisation( componentScore );
			for( i = 0; i < c; i++ ) {
				count[i] += componentScore[i]*w;
			}
		}
		computeHiddenParameter( count );
		
		if( plugInBg ) {
			bg.setParameters( old, 0 );
		} else {
			bg.initializeUniformly( freeParams );
		}
		
		initBgHelp();
	}

	public void initializeFunctionRandomly( boolean freeParams ) throws Exception
	{
		for( int n = function.length-1, i = 0; i < n; i++ )
		{
			function[i].initializeFunctionRandomly( freeParams );
		}
		if( !plugInBg ) {
			bg.initializeFunctionRandomly( freeParams );
		}
		if( optimizeHidden ) {
			initializeHiddenPotentialRandomly();
		}
		init( freeParams );
		initBgHelp();
	}
	
	private void initBgHelp()
	{
		int n = bg.getNumberOfParameters();
		if( n == ScoringFunction.UNKNOWN )
		{
			bgHelp = new double[0];
		}
		else if( bgHelp == null || bgHelp.length != n )
		{
			bgHelp = new double[n];
		}
	}
	
	public String getInstanceName()
	{
		String erg = "hiddenMotifsMixture("
				+ (type == CONTAINS_ALWAYS_A_MOTIF ? "contains always a motif" : "contains sometimes a motif") + "; bg = "
				+ bg.getInstanceName();
		for( int i = 0; i < function.length-1; i += 2 )
		{
			erg += "; (" + function[i].getInstanceName() + ", " + function[i+1].getInstanceName() + ")";
		}
		return erg + ")";
	}

	/**
	 * This method fills an internal array with the partial scores.
	 * 
	 * @param i the index of the component
	 * @param seq the {@link Sequence}
	 * @param start the start position
	 * 
	 * @return the number of computed partial scores
	 */
	protected int fillComponentScoreOf( int i, Sequence seq, int start )
	{
		int j = 2 * i, m = function[j].getLength(), l = 0, bgOrder = bg.getMaximalMarkovOrder();
		//start & end points
		int homSt, homE;
		PositionScoringFunction pos = (PositionScoringFunction) function[j + 1];
		pos.reset();
		do
		{
			pos.getInternalPosition( currentPos );
			homSt = Math.max( 0, currentPos[0] - bgOrder );
			homE = Math.min( length, currentPos[0] + m + bgOrder );
			simpleScore[i][l++] = pos.getLogScoreForInternal() + function[j].getLogScore( seq, start + currentPos[0] )
				- bg.getLogScore( seq, start + homSt, homE - homSt )
				+ bg.getLogScore( seq, start + homSt, currentPos[0] - homSt ) // left
				+ bg.getLogScore( seq, start + currentPos[0] + m, homE - currentPos[0] - m ); //right

		} while( pos.next() );
		return l;
	}

	protected void fillComponentScores( Sequence seq, int start )
	{
		int i = 0, j;
		for( ; i < bgIndex / 2; i++ )
		{
			j = fillComponentScoreOf( i, seq, start );
			componentScore[i] = logHiddenPotential[i] + Normalisation.getLogSum( 0, j, simpleScore[i] );
		}
		if( type == CONTAINS_SOMETIMES_A_MOTIF )
		{
			componentScore[i] = logHiddenPotential[i];
		}
	}

	public double getLogScore( Sequence seq, int start )
	{
		fillComponentScores( seq, start );
		return bg.getLogScore( seq, start, length ) + Normalisation.getLogSum( componentScore );
	}

	public double getLogScoreAndPartialDerivation( Sequence seq, int start, IntList indices, DoubleList partialDer )
	{
		int i = 0, j = 0, l, m, n, counter, stop, bgOrder = bg.getMaximalMarkovOrder();
		int homSt, homE;
		anz[0] = partialDer.length();
		int[][] end = new int[4][];
		PositionScoringFunction pos;
		// for each motif (function)
		for( ; j < bgIndex; i++, j = 2 * i )
		{
			m = function[j].getLength();
			stop = length - m + 1;
			
			iList[j].clear();
			dList[j].clear();
			end[0] = new int[stop];
			iList[j + 1].clear();
			dList[j + 1].clear();
			pos = (PositionScoringFunction) function[j + 1];
			pos.reset();
			end[1] = new int[stop];
			iList[bgIndex].clear();
			dList[bgIndex].clear();
			end[2] = new int[stop];
			end[3] = new int[stop];
			
			// for each start position
			stop = 0;
			do
			{
				// get current position
				pos.getInternalPosition( currentPos );
				homSt = Math.max( 0, currentPos[0] - bgOrder );
				homE = Math.min( length, currentPos[0] + m + bgOrder );
				// compute the score
				simpleScore[i][stop] =
					pos.getLogScoreAndPartialDerivationForInternal( iList[j + 1], dList[j + 1] ) // position
					+ function[j].getLogScoreAndPartialDerivation( seq, start + currentPos[0], iList[j], dList[j] ) // motif
					- bg.getLogScoreAndPartialDerivation( seq, start + homSt, homE - homSt, iList[bgIndex], dList[bgIndex] );
				
				end[0][stop] = iList[j].length();
				end[1][stop] = iList[j + 1].length();
				end[2][stop] = iList[bgIndex].length();
				
				simpleScore[i][stop] +=
					bg.getLogScoreAndPartialDerivation( seq, start + homSt, currentPos[0] - homSt, iList[bgIndex], dList[bgIndex] )
					+ bg.getLogScoreAndPartialDerivation( seq, start + currentPos[0] + m, homE - currentPos[0] - m, iList[bgIndex], dList[bgIndex] );

				end[3][stop++] = iList[bgIndex].length();
			}while( pos.next() );

			// normalize
			componentScore[i] = logHiddenPotential[i]
					+ Normalisation.logSumNormalisation( simpleScore[i], 0, stop, simpleScore[i], 0 );

			// weight the partial derivations
			for( m = 0; m < 2; m++ )
			{
				for( l = 0, counter = 0; l < stop; l++ )
				{
					while( counter < end[m][l] )
					{
						indices.add( iList[j + m].get( counter ) + paramRef[j + m] );
						partialDer.add( dList[j + m].get( counter++ ) * simpleScore[i][l] );
					}
				}
			}
			Arrays.fill( bgHelp, 0 );
			for( l = 0, counter = 0; l < stop; l++ )
			{
				while( counter < end[2][l] )
				{
					bgHelp[iList[bgIndex].get( counter )] -= (dList[bgIndex].get( counter ) * simpleScore[i][l]);
					counter++;
				}
				while( counter < end[3][l] )
				{
					bgHelp[iList[bgIndex].get( counter )] += (dList[bgIndex].get( counter ) * simpleScore[i][l]);
					counter++;
				}
			}
			for( counter = 0; counter < bgHelp.length; counter++ )
			{
				indices.add( paramRef[bgIndex] + counter );
				partialDer.add( bgHelp[counter] );
			}
			anz[i + 1] = partialDer.length();
		}
		if( type == CONTAINS_SOMETIMES_A_MOTIF )
		{
			componentScore[bgIndex / 2] = logHiddenPotential[bgIndex / 2];
		}

		double logScore = Normalisation.logSumNormalisation( componentScore, 0, componentScore.length, componentScore,
				0 );

		// adjust if necessary
		if( componentScore.length > 1 )
		{
			for( i = 0; i < hiddenPotential.length; i++ )
			{
				partialDer.multiply( anz[i], anz[i + 1], componentScore[i] );
			}
		}

		// bg for complete sequence
		iList[bgIndex].clear();
		dList[bgIndex].clear();
		logScore += bg.getLogScoreAndPartialDerivation( seq, start, length, iList[bgIndex], dList[bgIndex] );

		for( i = 0; i < iList[bgIndex].length(); i++ )
		{
			indices.add( paramRef[bgIndex] + iList[bgIndex].get( i ) );
			partialDer.add( dList[bgIndex].get( i ) );
		}

		// hiddenLambda
		n = bgIndex + 1;
		i = paramRef[n + 1] - paramRef[n];
		for( j = 0; j < i; j++ )
		{
			indices.add( paramRef[n] + j );
			partialDer.add( componentScore[j] - (isNormalized?hiddenPotential[j]:0) );
		}

		return logScore;
	}

	public String toString()
	{
		if( norm < 0 )
		{
			precomputeNorm();
		}
		StringBuffer erg = new StringBuffer( function.length * 1000 );
		erg.append( "bg:\n" + bg.toString() + "\n" );
		if( type == CONTAINS_SOMETIMES_A_MOTIF )
		{
			erg.append( "\nno motif: " + (partNorm[bgIndex / 2] / norm) + "\t" +partNorm[bgIndex / 2] + " / " + norm + "\t" + logHiddenPotential[bgIndex/2] + "\n" );
		}
		for( int i = 0; i < bgIndex; i += 2 )
		{
			erg.append( "\nmotif " + (i / 2) + ": " );
			if( hiddenPotential.length > 1 )
			{
				erg.append( (partNorm[i / 2] / norm) +"\t" +partNorm[i / 2] + " / " + norm + "\t" + logHiddenPotential[i/2] );
			}
			erg.append( "\n" + function[i].toString() + "\n" + function[i + 1].toString() + "\n" );
		}
		return erg.toString();
	}

	protected StringBuffer getFurtherInformation()
	{
		StringBuffer erg = new StringBuffer( 200 );
		XMLParser.appendBooleanWithTags( erg, type, "type" );
		XMLParser.appendDoubleWithTags( erg, sign, "sign" );
		XMLParser.appendBooleanWithTags( erg, plugInBg, "plugInBg" );
		return erg;
	}

	protected void extractFurtherInformation( StringBuffer xml ) throws NonParsableException
	{
		type = XMLParser.extractBooleanForTag( xml, "type" );
		sign = XMLParser.extractDoubleForTag( xml, "sign" );
		plugInBg = XMLParser.extractBooleanForTag( xml, "plugInBg" );
	}

	private double[][][] getContrast()
	{
		return bg.getAllConditionalStationaryDistributions();
	}
	
	public int[] determineNotSignificantPositionsFor( int motif, Sample[] data, double[][] weights, int classIdx  )
	{
		int c = getNumberOfComponents() - 1;
		if( ( motif < c || (motif == c && type == CONTAINS_ALWAYS_A_MOTIF) ) && function[2*motif] instanceof Mutable )
		{
			double[][][][] contrast = { getContrast() };
			double[] w = {1};
			if( norm < 0 ) {
				precomputeNorm();
			}
			double samples = 0, currentWeight, p = partNorm[motif] / norm, classCount;
			for(int j,i=0;i<data.length;i++){
				if( i == classIdx ){
					currentWeight = p;
				} else {
					currentWeight = 1d-p;
				}
				if( weights == null || weights[i] == null ) {
					classCount = data[i].getNumberOfElements();
				} else {
					classCount = 0;
					for( j = 0; j < weights[i].length; j++ ) {
						classCount += weights[i][j];
					}
				}
				samples += currentWeight * classCount;
			}
			int[] res = ((Mutable)function[2*motif]).determineNotSignificantPositions( samples, w, w, contrast, contrast, sign );
			return res;
		}
		else
		{
			return new int[2];
		}
	}

	public boolean modifyMotif( int motif, double[] weightsLeft, double[] weightsRight, double[][][][] replacementLeft, double[][][][] replacementRight, int offsetLeft, int offsetRight ) throws Exception
	{
		int c = getNumberOfComponents() - 1;
		if( (motif < c || (motif == c && type == CONTAINS_ALWAYS_A_MOTIF) )
				&& function[2*motif] instanceof Mutable )
		{
			double norm_old = function[2*motif].getNormalizationConstant();
			boolean res = ((Mutable)function[2*motif]).modify( weightsLeft, weightsRight, replacementLeft, replacementRight, offsetLeft, offsetRight );
			if( res )
			{
				setHyperParametersOfBackgroundModel();
				init( freeParams );
				((DurationScoringFunction)function[2*motif+1]).modify( offsetLeft-offsetRight );
				simpleScore[motif] = new double[(int) function[2*motif + 1].getAlphabetContainer().getAlphabetLengthAt( 0 )];
				hiddenParameter[motif] += Math.log( norm_old / function[2*motif].getNormalizationConstant() );
				this.setHiddenParameters( hiddenParameter , 0 );
			}
			return res;
		}
		else
		{
			return false;
		}
	}
	
	public boolean modifyMotif( int motifIndex, int offsetLeft, int offsetRight ) throws Exception
	{
		double[][][][] contrast = { getContrast() };
		double[] w = {1};
		return modifyMotif( motifIndex, w, w, contrast, contrast, offsetLeft, offsetRight );
	}

	public void initializeMotif( int motif, Sample data, double[] weights ) throws Exception
	{
		int c = getNumberOfComponents() - 1;
		if( motif < c || (motif == c && type == CONTAINS_ALWAYS_A_MOTIF) )
		{
			function[2*motif].initializeFunction( 0, freeParams, new Sample[]{data}, (weights==null ? null : new double[][]{weights}) );
			init( freeParams );
		}
		else
		{
			throw new IndexOutOfBoundsException();
		}
	}
	
	/**
	 * This method initializes the motif with index <code>motif</code> randomly using {@link NormalizableScoringFunction#initializeFunctionRandomly(boolean)}.
	 *  
	 * @param motif the index of the motif
	 * 
	 * @throws Exception either if the index is wrong or if it is thrown by the method {@link NormalizableScoringFunction#initializeFunctionRandomly(boolean)}
	 */
	public void initializeMotifRandomly( int motif ) throws Exception
	{
		int c = getNumberOfComponents() - 1;
		if( motif < c || (motif == c && type == CONTAINS_ALWAYS_A_MOTIF) )
		{
			function[2*motif].initializeFunctionRandomly(freeParams);
			init( freeParams );
		}
		else
		{
			throw new IndexOutOfBoundsException();
		}
	}

	public int getNumberOfMotifs()
	{
		return getNumberOfComponents() - (type==CONTAINS_ALWAYS_A_MOTIF?0:1);
	}

	public int getNumberOfMotifsInComponent( int component )
	{
		int c = getNumberOfComponents() - 1;
		if( component < c || (component == c && type == CONTAINS_ALWAYS_A_MOTIF) )
		{
			return 1;
		}
		else
		{
			return 0;
		}
	}
	
	public int getIndexOfMaximalComponentFor( Sequence sequence )
	{
		return getIndexOfMaximalComponentFor( sequence, 0 );
	}

	public int getGlobalIndexOfMotifInComponent( int component, int motif )
	{
		return component;
	}

	public double[] getProfileOfScoresFor( int component, int motif, Sequence sequence, int startpos, KindOfProfile dist ) throws WrongLengthException
	{
		if( length != sequence.getLength() - startpos )
		{
			throw new WrongLengthException( "The model can not score a sequence of this length." );
		}
		int c = getNumberOfComponents() - (type==CONTAINS_SOMETIMES_A_MOTIF?1:0);
		if( motif == 0 && component < c )
		{
			//unnormalized log score
			fillComponentScoreOf( component, sequence, startpos );

			double d = 0;
			switch( dist )
			{
				case UNNORMALIZED_JOINT:
					d = logHiddenPotential[component];
				case UNNORMALIZED_CONDITIONAL:
					d += bg.getLogScore( sequence, startpos, length );
					break;				
				case NORMALIZED_CONDITIONAL:
					d = -Normalisation.getLogSum( 0, simpleScore[component].length, simpleScore[component] );
					break;
				
				default:
					throw new IndexOutOfBoundsException();
			}
			
			double[] res = new double[length - function[2*component].getLength() + 1];
			DurationScoringFunction posPrior = (DurationScoringFunction) function[2*component+1];
			posPrior.reset();
			int[] internal = new int[1];
			posPrior.getInternalPosition( internal );
			boolean b;
			for( int i = 0, j = 0; i < res.length; i++ )
			{
				if( i == internal[0] )
				{
					res[i] = simpleScore[component][j++] + d;
					b = posPrior.next();
					if( b )
					{
						posPrior.getInternalPosition(internal);
					}
					else
					{
						internal[0] = -1;
					}
				}
				else
				{
					res[i] = Double.NEGATIVE_INFINITY;
				}
			}
			return res;
		}
		else
		{
			throw new IndexOutOfBoundsException();
		}
	}

	public int getMotifLength( int motif )
	{
		int c = getNumberOfComponents() - 1;
		if( motif < c || (motif == c && type == CONTAINS_ALWAYS_A_MOTIF) )
		{
			return function[2*motif].getLength();
		}
		else
		{
			throw new IndexOutOfBoundsException();
		}
	}
	
	/**
	 * This method set the sign that is used in shift, shrink and expand.
	 * 
	 * @param newSign the new sign in [0,1]
	 */
	public void setSign( double newSign )
	{
		if( 0 <= newSign && newSign <= 1 )
		{
			sign = newSign;
		}
		else
		{
			throw new IllegalArgumentException();
		}
	}
	
	/**
	 * This method returns the sign that is used in shift, shrink and expand.
	 * 
	 * @return the sign that is used in shift, shrink and expand.
	 */
	public double getSign()
	{
		return sign;
	}
	
	/**
	 * This method allows to adjust the hidden parameter in some way.
	 * 
	 * @param data the data
	 * @param dataWeights the weights corresponding to the {@link Sequence}s in <code>data</code> 
	 */
	public void adjustHiddenParameters( Sample data, double[] dataWeights )
	{
		int c = getNumberOfComponents(), stop = c - (type == CONTAINS_ALWAYS_A_MOTIF?0:1), i = 0, n = 0, anz = data.getNumberOfElements(), l;
		
		int[][] len = new int[stop][];
		double[][] weights = new double[stop][];
		DurationScoringFunction dur;
		for( ; i < stop; i++ )
		{
			dur = (DurationScoringFunction) function[2*i+1];
			len[i] = new int[dur.getNumberOfPossibilities()];
			weights[i] = new double[len[i].length];
			dur.reset();
			l = 0;
			do
			{
				len[i][l++] = dur.internal[0];
			}
			while( dur.next() );
		}
		Arrays.fill( hiddenParameter, 0 );
		int maxIndex;
		double w = 1;
		for( ; n < anz; n++ )
		{
			if( dataWeights != null )
			{
				w = dataWeights[n];
			}
			fillComponentScores( data.getElementAt(n), 0 );
			Normalisation.logSumNormalisation( componentScore );
			for( i = 0; i < c; i++ )
			{
				hiddenParameter[i] += componentScore[i]*w;
				if( i < stop )
				{
					maxIndex = 0;
					for( l = 1; l < weights[i].length; l++ )
					{
						if( simpleScore[i][l] > simpleScore[i][maxIndex] )
						{
							maxIndex=l;
						}
					}
					weights[i][maxIndex] += componentScore[i]*w;		
				}
			}			
		}
		
		computeHiddenParameter( hiddenParameter );
		for( i = 0; i < stop; i++ )
		{
			((DurationScoringFunction) function[2*i+1]).adjust( len[i], weights[i] );
			//System.out.println( function[2*i+1] );
		}
	}
	
	public Strand getStrandFor( int component, int motif, Sequence sequence, int startpos ) throws Exception
	{	 
		int c = getNumberOfComponents() - (type==CONTAINS_SOMETIMES_A_MOTIF?1:0);
		if( motif > 0 || component >= c )
		{
			throw new IndexOutOfBoundsException();
		}
		else
		{
			ScoringFunction m = function[2*component];
			while( m instanceof NormalizedScoringFunction )
			{
				m = ((NormalizedScoringFunction)m).getFunction();
			}
			if( m instanceof StrandScoringFunction )
			{
			    return ((StrandScoringFunction)m).getIndexOfMaximalComponentFor( sequence, startpos )==0?Strand.FORWARD:Strand.REVERSE;
			}
			else
			{
			    return Strand.FORWARD;
			}
		}
	}
}
