/*
 * This file is part of Jstacs.
 * 
 * Jstacs is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * Jstacs is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * Jstacs. If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.data.sequences;

import de.jstacs.data.AlphabetContainer;
import de.jstacs.data.Sequence;
import de.jstacs.data.sequences.annotation.SequenceAnnotation;

/**
 * This is the main class for subsequences, composite sequences, ... . All these
 * sequences are defined on an existing {@link Sequence}. After creating an
 * {@link RecursiveSequence}, no {@link SequenceAnnotation} of the internally {@link Sequence}
 * is returned by {@link Sequence#getAnnotation()}, ...
 * 
 * @author Jens Keilwagen
 */
public abstract class RecursiveSequence extends Sequence {

	/**
	 * The internal sequence.
	 */
	protected Sequence content;

	/**
	 * Creates a new {@link RecursiveSequence} on the {@link Sequence}
	 * <code>seq</code> with the {@link AlphabetContainer} <code>alphabet</code>
	 * and the annotation <code>annotation</code>.
	 * 
	 * @param alphabet
	 *            the {@link AlphabetContainer}
	 * @param annotation
	 *            the annotation of the {@link RecursiveSequence}
	 * @param seq
	 *            the sequence
	 * 
	 * @see Sequence#Sequence(AlphabetContainer, SequenceAnnotation[])
	 */
	public RecursiveSequence( AlphabetContainer alphabet, SequenceAnnotation[] annotation, Sequence seq ) {
		super( alphabet, annotation );
		content = seq;
	}

	/**
	 * Creates a new {@link RecursiveSequence} on the {@link Sequence}
	 * <code>seq</code> with the {@link AlphabetContainer} <code>alphabet</code>
	 * using the annotation of the given {@link Sequence}.
	 * 
	 * @param alphabet
	 *            the {@link AlphabetContainer}
	 * @param seq
	 *            the sequence
	 */
	public RecursiveSequence( AlphabetContainer alphabet, Sequence seq ) {
		this( alphabet, null, seq );
	}
	
	/* (non-Javadoc)
	 * @see de.jstacs.data.Sequence#continuousVal(int)
	 */
	@Override
	public double continuousVal( int pos ) {
		return content.continuousVal( getIndex( pos ) );
	}

	/* (non-Javadoc)
	 * @see de.jstacs.data.Sequence#discreteVal(int)
	 */
	@Override
	public int discreteVal( int pos ) {
		return content.discreteVal( getIndex( pos ) );
	}

	/**
	 * Returns the index in the internal sequence.
	 * 
	 * @param pos
	 *            the index in the external sequence
	 * 
	 * @return the index in the internal sequence
	 */
	protected abstract int getIndex( int pos );

}
