/*
 * This file is part of Jstacs.
 * 
 * Jstacs is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * Jstacs is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * Jstacs. If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.utils;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.Modifier;
import java.net.JarURLConnection;
import java.net.URL;
import java.util.AbstractList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

import de.jstacs.InstantiableFromParameterSet;
import de.jstacs.parameters.CollectionParameter;
import de.jstacs.parameters.InstanceParameterSet;

/**
 * Utility-class with static methods to
 * <ul>
 * <li>find all sub-classes of a certain class (or interface) within the scope
 * of the current class-loader</li>
 * <li>find all sub-classes of a certain class (or interface) within the scope
 * of the current class-loader that can be instantiated, i.e. that are neither
 * interfaces nor abstract</li>
 * <li>filter a set of classes by inheritance from a super-class</li>
 * <li>obtain the class of an {@link InstanceParameterSet} that can be used to
 * instantiate a sub-class of {@link InstantiableFromParameterSet}.</li>
 * <li>obtain a {@link CollectionParameter} using all possible
 * {@link InstanceParameterSet}s (for classes that are a subclass of a specified
 * superclass) as elements</li>
 * </ul>
 * 
 * The methods may fail to find a certain sub-class, if
 * <ul>
 * <li>it was loaded by another {@link ClassLoader} than the caller</li>
 * <li>it was loaded from a physically other place than the super-class, e.g.
 * another jar-file.
 * </ul>
 * 
 * @author Jan Grau, Jens Keilwagen
 */
public class SubclassFinder {

	/**
	 * Returns all sub-classes of <code>T</code> that can be instantiated, i.e.
	 * are neither an interface nor abstract, and that are located in a package
	 * below <code>startPackage</code>.
	 * 
	 * @param <T>
	 *            The class to obtain the sub-classes for
	 * @param clazz
	 *            the {@link Class} object for T
	 * @param startPackage
	 *            the package under which to search
	 * @return the {@link Class} objects for the sub-classes
	 * @throws ClassNotFoundException
	 *             if one of the classes is present in the file system or jar
	 *             but cannot be loaded by the class loader
	 * @throws IOException
	 *             is thrown if the classes are searched for in a jar file, but
	 *             that file could not be accessed or read
	 */
	public static <T> LinkedList<Class<? extends T>> findInstantiableSubclasses( Class<T> clazz, String startPackage ) throws ClassNotFoundException,
			IOException {
		LinkedList<Class<? extends T>> list = findSubclasses( clazz, startPackage );
		LinkedList<Class<? extends T>> list2 = new LinkedList<Class<? extends T>>();
		Iterator<Class<? extends T>> it = list.iterator();
		while( it.hasNext() ) {
			Class<? extends T> c = it.next();
			if( !c.isInterface() && !Modifier.isAbstract( c.getModifiers() ) ) {
				list2.add( c );
			}
		}
		return list2;
	}

	/**
	 * Returns a {@link LinkedList} of the {@link Class} objects for all classes
	 * in <code>listToFilter</code> that are sub-classes of
	 * <code>superClass</code>.
	 * 
	 * @param <S>
	 *            the class that is used as filter
	 * @param <T>
	 *            a common super-class of all classes in
	 *            <code>listToFilter</code>
	 * @param superclass
	 *            the additional class to use as a filter criterion
	 * @param listToFilter
	 *            the list of classes
	 * @return the filtered list
	 */
	public static <S, T> LinkedList<Class<? extends S>> filterBySuperclass( Class<S> superclass, LinkedList<Class<? extends T>> listToFilter ) {

		LinkedList<Class<? extends S>> list2 = new LinkedList<Class<? extends S>>();
		Iterator<Class<? extends T>> it = listToFilter.iterator();
		while( it.hasNext() ) {
			Class<? extends T> c = it.next();
			if( superclass.isAssignableFrom( c ) ) {
				list2.add( (Class<? extends S>)c );
			}
		}
		return list2;
	}

	/**
	 * Returns a {@link LinkedList} of the classes of the
	 * {@link InstanceParameterSet}s that can be used to instantiate the
	 * sub-class of {@link InstantiableFromParameterSet} that is given by
	 * <code>clazz</code>
	 * 
	 * @param clazz
	 *            the {@link Class} object of the sub-class of
	 *            {@link InstantiableFromParameterSet}
	 * @return a {@link LinkedList} of {@link Class}es of the corresponding
	 *         {@link InstanceParameterSet}s
	 */
	public static LinkedList<Class<? extends InstanceParameterSet>> getParameterSetFor( Class<? extends InstantiableFromParameterSet> clazz ) {
		Constructor[] cons = clazz.getConstructors();
		Class[] types = null;
		LinkedList<Class<? extends InstanceParameterSet>> list = new LinkedList<Class<? extends InstanceParameterSet>>();
		for( int i = 0; i < cons.length; i++ ) {
			if( ( types = cons[i].getParameterTypes() ).length == 1 ) {
				if( InstanceParameterSet.class.isAssignableFrom( types[0] ) ) {
					list.add( types[0] );
				}
			}
		}
		return list;
	}

	/**
	 * Returns all sub-classes of <code>T</code> including interfaces and
	 * abstract classes that are located in a package below
	 * <code>startPackage</code>.
	 * 
	 * @param <T>
	 *            The class to obtain the sub-classes for
	 * @param clazz
	 *            the {@link Class} object for T
	 * @param startPackage
	 *            the package under which to search
	 * @return the {@link Class} objects for the sub-classes
	 * @throws ClassNotFoundException
	 *             if one of the classes is present in the file system or jar
	 *             but cannot be loaded by the class loader
	 * @throws IOException
	 *             is thrown if the classes are searched for in a jar file, but
	 *             that file could not be accessed or read
	 */
	public static <T> LinkedList<Class<? extends T>> findSubclasses( Class<T> clazz, String startPackage ) throws ClassNotFoundException,
			IOException {

		String name = startPackage;
		if( !name.startsWith( "/" ) ) {
			name = "/" + name;
		}
		name = name.replace( ".", "/" );

		URL url = clazz.getResource( name );

		LinkedList<Class<? extends T>> list = new LinkedList<Class<? extends T>>();

		if( url != null ) {
			File dir = new File( url.getFile() );

			if( dir.exists() ) {
				File[] files = dir.listFiles();
				for( int i = 0; i < files.length; i++ ) {
					if( files[i].isDirectory() ) {
						//System.out.println(startPackage+"."+files[i].getName());
						list.addAll( findSubclasses( clazz, startPackage + "." + files[i].getName() ) );
					} else if( files[i].isFile() && files[i].getName().endsWith( ".class" ) ) {
						add( clazz, list, startPackage + "." + files[i].getName().substring( 0, files[i].getName().lastIndexOf( "." ) ) );
					}
				}
			} else {
				JarURLConnection con = (JarURLConnection)url.openConnection();
				JarFile jar = con.getJarFile();
				String starts = con.getEntryName();
				Enumeration<JarEntry> en = jar.entries();
				while( en.hasMoreElements() ) {
					JarEntry entry = en.nextElement();
					String entryname = entry.getName();
					if( entryname.startsWith( starts ) && entryname.endsWith( ".class" ) ) {
						String classname = entryname.substring( 0, entryname.length() - 6 );
						if( classname.startsWith( "/" ) ) {
							classname.substring( 1 );
						}
						add( clazz, list, classname.replace( "/", "." ) );
					}
				}
			}
		}

		return list;
	}

	@SuppressWarnings( "unchecked" )
	private static <T> void add( Class<T> clazz, AbstractList<Class<? extends T>> list, String className ) {
		try {
			Class c = Class.forName( className );
			if( clazz.isAssignableFrom( c ) ) {
				list.add( c );
			}
		} catch ( Exception e ) {} catch ( Error e ) {}
	}

	/**
	 * This method creates an {@link CollectionParameter} that contains
	 * {@link de.jstacs.parameters.InstanceParameterSet} for each possible
	 * class. The classes are specified by
	 * {@link SubclassFinder#findInstantiableSubclasses(Class, String)} and
	 * {@link SubclassFinder#filterBySuperclass(Class, LinkedList)}.
	 * 
	 * @param <T>
	 *            The class to use the sub-classes in the
	 *            {@link CollectionParameter}
	 * @param clazz
	 *            the {@link Class} object for <code>T</code>
	 * @param startPackage
	 *            the package under which to start the search
	 * @param name
	 *            the name of the {@link CollectionParameter}
	 * @param comment
	 *            the comment for the {@link CollectionParameter}
	 * @param required
	 *            whether the {@link CollectionParameter} is required
	 * 
	 * @return a {@link CollectionParameter} that contains
	 *         {@link de.jstacs.parameters.InstanceParameterSet} for each
	 *         possible class.
	 * 
	 * @throws InstantiationException
	 *             if any {@link InstanceParameterSet} has no nullary
	 *             constructor; or if the instantiation fails for some other
	 *             reason
	 * @throws IllegalAccessException
	 *             if any {@link InstanceParameterSet} or its nullary
	 *             constructor is not accessible
	 * @throws ClassNotFoundException
	 *             if one of the classes is present in the file system or jar
	 *             but cannot be loaded by the class loader
	 * @throws IOException
	 *             if the classes are searched for in a jar file, but that file
	 *             could not be accessed or read
	 * 
	 * @see SubclassFinder#findInstantiableSubclasses(Class, String)
	 * @see SubclassFinder#filterBySuperclass(Class, LinkedList)
	 */
	public static <T> CollectionParameter getCollection( Class<T> clazz, String startPackage, String name, String comment, boolean required ) throws InstantiationException,
			IllegalAccessException,
			ClassNotFoundException,
			IOException {
		LinkedList<Class<? extends T>> classes = SubclassFinder.findInstantiableSubclasses( clazz, startPackage );
		LinkedList<Class<? extends InstantiableFromParameterSet>> filteredClasses = SubclassFinder.filterBySuperclass( InstantiableFromParameterSet.class,
				classes );
		LinkedList<InstanceParameterSet> sets = new LinkedList<InstanceParameterSet>();
		Iterator<Class<? extends InstantiableFromParameterSet>> it = filteredClasses.iterator();
		Iterator<Class<? extends InstanceParameterSet>> psIt;
		while( it.hasNext() ) {
			psIt = SubclassFinder.getParameterSetFor( it.next() ).iterator();
			while( psIt.hasNext() ) {
				sets.add( psIt.next().newInstance() );
			}
		}
		return new CollectionParameter( sets.toArray( new InstanceParameterSet[0] ), name, comment, required );
	}
}
