/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.algorithms.optimization.termination;

import de.jstacs.utils.Time;

/**
 * This class implements a {@link TerminationCondition} that allows no further iteration in an optimization if the
 * scalar product of the current and the last values of <code>x</code> will be small, i.e.,
 * {@latex.inline $(\\underline{x}_i-\\underline{x}_{i-1})^T (\\underline{x}_i-\\underline{x}_{i-1}) < \\epsilon$}.
 * 
 * @author Jens Keilwagen
 */
public class SmallStepCondition implements TerminationCondition {

	private double epsilon;
	
	/**
	 * This constructor creates an instance that allows no further iteration in an optimization if the
	 * scalar product of the current and the last values of <code>x</code> is smaller than <code>epsilon</code>.
	 * 
	 * @param epsilon the threshold for stopping, has to be non-negative
	 */
	public SmallStepCondition( double epsilon ) {
		if( epsilon < 0 ){
			throw new IllegalArgumentException( "The espilon has to be non-negative." );
		}
		this.epsilon = epsilon;
	}
	
	public boolean doNextIteration( int iteration, double f_last, double f_current, double[] gradient, double[] direction, double alpha,
			Time t ) {
		double s = 0;
		for( int counter = 0; counter < direction.length; counter++ ) {
			s += direction[counter] * direction[counter];
		}
		return s * ( alpha * alpha ) >= this.epsilon;
	}
}
