/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.scoringFunctions;

import java.util.Arrays;
import java.util.Map;
import java.util.TreeMap;

import de.jstacs.NonParsableException;
import de.jstacs.data.AlphabetContainer;
import de.jstacs.data.Sample;
import de.jstacs.data.Sequence;
import de.jstacs.data.alphabets.DiscreteAlphabet;
import de.jstacs.io.XMLParser;
import de.jstacs.utils.DoubleList;
import de.jstacs.utils.IntList;
import de.jstacs.utils.Normalisation;
import de.jstacs.utils.random.DirichletMRG;
import de.jstacs.utils.random.FastDirichletMRGParams;
import de.jtem.numericalMethods.calculus.specialFunctions.Gamma;

/**
 * This scoring function implements a cyclic Markov model of arbitrary order and periodicity for any sequence length.
 * The scoring function uses the parametrization of Meila.
 * 
 * @author Jens Keilwagen
 */
public class CMMScoringFunction extends AbstractVariableLengthScoringFunction
{
	private boolean freeParams, plugIn, optimize, optimizeFrame;

	private int order, period, starts, initFrame;
	private int[] powers;
	private double classEss, logGammaSum;
	
	// the index can be computed using the powers array
	private double[][][] params, probs, logNorm;
	private double[] sumOfHyperParams, frameLogScore, frameParams, frameProbs, partDer;
	private double logFrameNorm;
	private int[][][] counter, distCounter;
	private int[][] offset;

	/**
	 * The main constructor.
	 * 
	 * @param alphabets the alphabet container
	 * @param order the oder of the model (has to be non-negative)
	 * @param period the period
	 * @param classEss the ess of the class
	 * @param sumOfHyperParams the sum of the hyper parameter for each order (length has to be <code>order</code>+1, each entry has to be non-negative), the sum also sums over the period
	 * @param plugIn a switch which enables to used the MAP-parameters as plug-in parameters
	 * @param optimize a switch which enables to optimize or fix the parameters
	 * @param starts the number of recommended starts
	 * @param initFrame the frame which should be used for plug-in initialization, negative for random initialization
	 */
	public CMMScoringFunction( AlphabetContainer alphabets, int order, int period, double classEss, double[] sumOfHyperParams, boolean plugIn, boolean optimize, int starts, int initFrame )
	{
		super( alphabets );
		if( order < 0 )
		{
			throw new IllegalArgumentException( "The order has to be non-negative." );
		}
		this.order = order;
		if( period <= 0 )
		{
			throw new IllegalArgumentException( "The period has to be positive." );
		}
		this.period = period;		
		createArrays();
		if( classEss < 0 )
		{
			throw new IllegalArgumentException( "The ess for the class has to be non-negative." );
		}
		this.classEss = classEss;
		if( sumOfHyperParams == null )
		{
			sumOfHyperParams = new double[order+1];
		}
		else
		{
			if( sumOfHyperParams.length != order+1 )
			{
				throw new IllegalArgumentException( "Wrong dimension of the ess array." );
			}
			else
			{
				this.sumOfHyperParams = new double[order+1];
				for( int i = 0; i <= order; i++ )
				{
					if( sumOfHyperParams[i] < 0 )
					{
						throw new IllegalArgumentException( "The ess has to be non-negative. Violated at position " + i + "." );
					}
					if( i > 0 && i < order && sumOfHyperParams[i] > sumOfHyperParams[i-1] )
					{
						throw new IllegalArgumentException( "The ess for start probabilities of order " + i + " is inconsistent with the ess for the probabilities of the previous order." );
					}
					this.sumOfHyperParams[i] = sumOfHyperParams[i];
				}
			}
		}
		frameParams = new double[period];
		Arrays.fill( frameParams, -Math.log( period ) );
		Arrays.fill( frameProbs, 1d / (double) period );
		params = new double[period][order+1][];
		double uniform = 1d / (double) powers[1], logUniform = Math.log( uniform );
		for( int i, p = 0; p < period; p++ )
		{
			for( i = 0; i <= order; i++ )
			{
				params[p][i] = new double[powers[i+1]];
				probs[p][i] = new double[powers[i+1]];
				logNorm[p][i] = new double[powers[i]];
				Arrays.fill( params[p][i], logUniform );
				Arrays.fill( probs[p][i], uniform );
			}
		}
		this.plugIn = plugIn;
		this.optimize = optimize;
		this.optimizeFrame = true;
		if( starts <= 0 )
		{
			throw new IllegalArgumentException( "The number of starts has to be positive." );
		}
		this.starts = starts;
		setFreeParams( false );
		computeConstantsOfLogPrior();
		if( initFrame < period )
		{
			this.initFrame = initFrame;
		}
		else
		{
			throw new IllegalArgumentException( "Check initFrame." );
		}
	}

	/**
	 * This is the constructor for {@link de.jstacs.Storable}.
	 * 
	 * @param source the xml representation
	 * 
	 * @throws NonParsableException if the representation could not be parsed.
	 */
	public CMMScoringFunction( StringBuffer source ) throws NonParsableException
	{
		super( source );
	}

	private void createArrays()
	{
		powers = new int[order+2];
		powers[0] = 1;
		powers[1] = (int) alphabets.getAlphabetLengthAt( 0 );
		int i;
		for( i = 2; i < powers.length; i++ )
		{
			powers[i] = powers[i-1]*powers[1];
		}
		probs = new double[period][order+1][];
		logNorm = new double[period][order+1][];
		frameProbs = new double[period];
		counter = new int[period][period][powers[order+1]];
		distCounter = new int[period][period][powers[order]];
		offset = new int[period][order+2];
		frameLogScore = new double[period];
		partDer = new double[powers[1]];
	}
	
	public CMMScoringFunction clone() throws CloneNotSupportedException
	{
		CMMScoringFunction clone = (CMMScoringFunction) super.clone();
		// the powers do not have to be cloned
		clone.sumOfHyperParams = sumOfHyperParams.clone();
		
		clone.params = new double[period][order+1][];
		clone.probs = new double[period][order+1][];
		clone.logNorm = new double[period][order+1][];
		clone.offset = new int[period][];
		clone.counter = new int[period][period][];
		clone.distCounter = new int[period][period][];
		for( int o, p = 0; p < period; p++ )
		{
			for( o = 0; o <= order; o++ )
			{
				clone.params[p][o] = params[p][o].clone();
				clone.probs[p][o] = probs[p][o].clone();
				clone.logNorm[p][o] = logNorm[p][o].clone();
			}
			clone.offset[p] = offset[p].clone();
			for( o = 0; o < period; o++ )
			{
				clone.counter[p][o] = counter[p][o].clone();
				clone.distCounter[p][o] = distCounter[p][o].clone();
			}
		}
		clone.frameParams = frameParams.clone();
		clone.frameProbs = frameProbs.clone();
		clone.frameLogScore = frameLogScore.clone();
		clone.partDer = partDer.clone();
		return clone;
	}

	public String getInstanceName()
	{
		return "cMM(" + order + ", " + period + ")";
	}
	
	private void fillFrameLogScores( Sequence seq, int start, int length ) {
		int l = 0, indexOld, indexNew = 0, o = Math.min( order, length ), p;
		for( p = 0; p < period; p++ )
		{
			frameLogScore[p] = frameParams[p];
		}
		for( ; l < o; l++ )
		{
			indexOld = indexNew;
			indexNew = indexOld*powers[1] + seq.discreteVal( start++ );
			for( p = 0; p < period; p++ )
			{
				frameLogScore[p] += params[(p+l)%period][l][indexNew] - logNorm[(p+l)%period][l][indexOld];
			}
		}
		for( ; l < length; l++ )
		{
			indexOld = indexNew % powers[order];
			indexNew = indexOld * powers[1] + seq.discreteVal( start++ );
			for( p = 0; p < period; p++ )
			{
				frameLogScore[p] += params[(p+l)%period][order][indexNew] - logNorm[(p+l)%period][order][indexOld];
			}
		}
	}
	
	public double getLogScore( Sequence seq, int start, int length )
	{
		fillFrameLogScores(seq, start, length);
		return Normalisation.getLogSum( frameLogScore ) - logFrameNorm;
	}

	public double getLogScoreAndPartialDerivation( Sequence seq, int start, int length, IntList indices, DoubleList dList )
	{
		if( optimize )
		{
			int l = 0, indexOld, indexNew = 0, h, o = Math.min( order, length ), index, z, p;
			for( p = 0; p < period; p++ )
			{
				for( z = 0; z < period; z++ )
				{
					Arrays.fill( counter[p][z], 0 );
					Arrays.fill( distCounter[p][z], 0 );
				}
				frameLogScore[p] = frameParams[p];
			}
			int stop = powers[1] - (freeParams?1:0);
			
			// start probablities
			for( ; l < o; l++ )
			{
				indexOld = indexNew;
				z = indexOld * powers[1];
				indexNew = z + seq.discreteVal( start++ );
				h = z - (freeParams?indexOld:0);
				
				for( p = 0; p < period; p++ )
				{
					frameLogScore[p] += params[(p+l)%period][l][indexNew] - logNorm[(p+l)%period][l][indexOld];
					for( index = 0; index < stop; index++ )
					{
						indices.add( offset[p][l] + h + index );
						if( z + index == indexNew )
						{
							dList.add( 1 - probs[p][l][z + index] );
						}
						else
						{
							dList.add( - probs[p][l][z + index] );
						}
					}
				}
			}
			//counting the usage of transition probability parameters for the sequence
			for( ; l < length; l++ )
			{
				indexOld = indexNew % powers[order];
				indexNew = indexOld * powers[1] + seq.discreteVal( start++ );
				for( p = 0; p < period; p++ )
				{
					frameLogScore[p] += params[(p+l)%period][order][indexNew] - logNorm[(p+l)%period][order][indexOld];
					distCounter[p][(p+l)%period][indexOld]++;
					counter[p][(p+l)%period][indexNew]++;
				}
			}
			
			//computing the score
			double erg = Normalisation.logSumNormalisation( frameLogScore, 0, period, frameLogScore, 0 ) - logFrameNorm;
			
			
			//computing the gradient and the score

			//start probs
			indexOld = 0;
			for( l = 0; l < o; l++ )
			{
				for( p = 0; p < period; p++ )
				{
					indexNew = indexOld + stop;
					dList.multiply( indexOld, indexNew, frameLogScore[p] );
					indexOld = indexNew;
				}
			}
			
			boolean used;
			//transition probs
			for( l = 0; l < distCounter[0][0].length; l++ )
			{
				h = l*(powers[1]-(freeParams?1:0));
				o = l*powers[1];
				for( z = 0; z < period; z++ )
				{
					Arrays.fill( partDer, 0 );
					used = false;
					for( p = 0; p < period; p++ )
					{
						if( distCounter[p][z][l] > 0 )
						{
							used = true;
							for( index = 0; index < stop; index++ )
							{
								partDer[index] += (frameLogScore[p] *( counter[p][z][o+index] - distCounter[p][z][l]*probs[z][order][o+index] ));
							}
						}
					}
					if( used )
					{
						for( index = 0; index < stop; index++ )
						{
							indices.add( offset[z][order] + h + index  );
							dList.add( partDer[index] );
						}
					}
				}
			}
			
			if( optimizeFrame )
			{
				for( p = 0; p < period-(freeParams?1:0); p++ )
				{
					indices.add( p );
					dList.add( frameLogScore[p] - frameProbs[p] );
				}
			}
			return erg;
		}
		else
		{
			return getLogScore( seq, start, length );
		}
	}

	public int getNumberOfParameters()
	{
		return offset[period-1][order+1];
	}

	public void setParameters( double[] params, int start )
	{
		if( optimize )
		{
			int j, n, index, o, p, stop;
			if( optimizeFrame )
			{
				stop = period - (freeParams?1:0);
				logFrameNorm = 0;
				for( p = 0; p < stop; p++ )
				{
					frameParams[p] = params[start++];
					frameProbs[p] = Math.exp( frameParams[p] );
					logFrameNorm += frameProbs[p];
				}
				if( stop < period )
				{
					frameProbs[p] = Math.exp( frameParams[p] );
					logFrameNorm += frameProbs[p];
				}
				for( p = 0; p < period; p++ )
				{
					frameProbs[p] /= logFrameNorm;
				}
				logFrameNorm = Math.log( logFrameNorm );
			}
			stop = powers[1] - (freeParams?1:0);
			for( p = 0; p < period; p++ )
			{
				for( o = 0; o <= order; o++ )
				{
					for( index = n = 0; n < logNorm[p][o].length; n++ )
					{
						logNorm[p][o][n] = 0d;
						for( j = 0; j < stop; j++, start++ )
						{
							this.params[p][o][index+j] = params[start]; 
							probs[p][o][index+j] = Math.exp( this.params[p][o][index+j] );
							logNorm[p][o][n] += probs[p][o][index+j]; 
						}
						if( j < powers[1] )
						{
							probs[p][o][index+j] = Math.exp( this.params[p][o][index+j] );
							logNorm[p][o][n] += probs[p][o][index+j];
						}
						for( j = 0; j < powers[1]; j++, index++ )
						{
							this.probs[p][o][index] /= logNorm[p][o][n]; 
						}
						logNorm[p][o][n] = Math.log( logNorm[p][o][n] );
					}
				}
			}
		}
	}

	public StringBuffer toXML()
	{
		StringBuffer b = new StringBuffer( 10000 );
		XMLParser.appendIntWithTags( b, length, "length" );
		XMLParser.appendStorableWithTags( b, alphabets, "alphabets" );
		XMLParser.appendIntWithTags( b, order, "order" );
		XMLParser.appendIntWithTags( b, period, "period" );
		XMLParser.appendDoubleWithTags( b, classEss, "classEss" );
		XMLParser.appendDoubleArrayWithTags( b, sumOfHyperParams, "sumOfHyperParams" );
		XMLParser.appendDoubleArrayWithTags( b, frameParams, "frameParams" );
		for( int p = 0; p < period; p++ )
		{
			XMLParser.appendDouble2ArrayWithTagsAndAttributes( b, params[p], "params", "frame=\"" + p + "\"" );
		}
		XMLParser.appendBooleanWithTags( b, plugIn, "plugIn" );
		XMLParser.appendBooleanWithTags( b, optimize, "optimize" );
		XMLParser.appendBooleanWithTags( b, optimizeFrame, "optimizeFrame" );
		XMLParser.appendIntWithTags( b, starts, "starts" );
		XMLParser.appendBooleanWithTags( b, freeParams, "freeParams" );
		XMLParser.appendIntWithTags( b, initFrame, "initFrame" );
		XMLParser.addTags( b, getClass().getSimpleName() );
		return b;
	}

	public double[] getCurrentParameterValues()
	{
		int l = optimize?offset[period-1][order+1]:0;
		double[] erg = new double[l];
		if( optimize )
		{
			int stop, p, i = 0, j, index, o;
			if( optimizeFrame )
			{
				stop = period - (freeParams?1:0);
				for( p = 0; p < stop; p++, i++ )
				{
					erg[i] = frameParams[p];
				}
			}
			stop = powers[1] - (freeParams?1:0);
			for( p = 0; p < period; p++ )
			{
				for( o = 0; o <= order; o++ )
				{
					for( index = 0; index < params[p][o].length; index += powers[1] )
					{
						for( j = 0; j < stop; j++, i++ )
						{
							erg[i] = params[p][o][index+j];
						}
					}
				}
			}
		}
		return erg;
	}
	
	public void initializeFunction( int index, boolean freeParams, Sample[] data, double[][] weights )
	{
		if( optimize && plugIn && data != null && data[index] != null )
		{
			int anz = data[index].getNumberOfElements();
			Sequence seq;
			double w = 1;
			boolean externalWeights = weights != null && weights [index] != null;
			int rMax = initFrame >= 0 ? 1 : 3;
			double[][] frameP = new double[anz][period];
			FastDirichletMRGParams dirParams = new FastDirichletMRGParams( classEss/(double) period );
			for( int r = 0; r < rMax; r++ ) {
				for( int i = 0; i < anz; i++ ) {
					if( initFrame < 0 ) {
						if( r == 0 ) {
							DirichletMRG.DEFAULT_INSTANCE.generate( frameP[i], 0, period, dirParams );
						} else {
							fillFrameLogScores(data[index].getElementAt(i), 0, length);
							Normalisation.logSumNormalisation(frameLogScore, 0, period, frameP[i], 0);
						}
					} else {
						Arrays.fill( frameP[i], 0 );
						frameP[i][initFrame] = 1;
					}
				}
				
				//preparation
				int len, o, indexOld, indexNew, p;
				double hyper;
				for( p = 0; p < period; p++ )
				{
					for( o = 0; o <= order; o++ )
					{
						hyper = sumOfHyperParams[o] / (double) (period*params[p][o].length);
						Arrays.fill( probs[p][o], hyper );
						Arrays.fill( logNorm[p][o], hyper*powers[1] );
					}
				}
				if( optimizeFrame )
				{
					Arrays.fill( frameProbs, classEss/(double) period );
					logFrameNorm = classEss;
				}
				
				//counting
				for( int l, i = 0; i < anz; i++ )
				{
					seq = data[index].getElementAt(i);
					len = seq.getLength();
					o = Math.min( len, order );
					indexNew = 0;
					if( externalWeights )
					{
						w = weights[index][i];
					}
					
					if( optimizeFrame )
					{
						for( p = 0; p < period; p++ )
						{
							frameP[i][p] *= w;
							frameProbs[p] += frameP[i][p];
						}
						logFrameNorm += w;
					}
					
					for( l = 0 ; l < o; l++ )
					{
						indexOld = indexNew;
						indexNew = indexOld*powers[1] + seq.discreteVal( l );
						for( p = 0; p < period; p++ )
						{
							probs[(p+l)%period][l][indexNew] += frameP[i][p];
							logNorm[(p+l)%period][l][indexOld] += frameP[i][p];
						}
					}
					for( ; l < len; l++ )
					{
						indexOld = indexNew % powers[order];
						indexNew = indexOld * powers[1] + seq.discreteVal( l );
						for( p = 0; p < period; p++ )
						{
							probs[(p+l)%period][order][indexNew] += frameP[i][p];
							logNorm[(p+l)%period][order][indexOld] += frameP[i][p];
						}
					}
				}
				
				//computing freqs and parameters
				if( optimizeFrame )
				{
					for( p = 0; p < period; p++ )
					{
						frameProbs[p] /= logFrameNorm;
						frameParams[p] = Math.log( frameProbs[p] );
					}
					logFrameNorm = 0;
				}
				for( p = 0; p < period; p++ )
				{
					for( o = 0; o <= order; o++ )
					{
						for( indexOld = indexNew = 0; indexOld < logNorm[p][o].length; indexOld++ )
						{
							for( len = 0; len < powers[1]; len++, indexNew++ )
							{
								probs[p][o][indexNew] /= logNorm[p][o][indexOld];
								params[p][o][indexNew] = Math.log( probs[p][o][indexNew]);
							}
							logNorm[p][o][indexOld] = 0;
						}
					}
				}
			}
		}
		else
		{
			initializeFunctionRandomly( freeParams );
		}
		setFreeParams( freeParams );	
	}
	
	public void initializeFunctionRandomly( boolean freeParams )
	{
		if( optimize )
		{
			int o, normCounter, paramCounter, len, p;
			FastDirichletMRGParams hyper;
			double[] freq;
			if( optimizeFrame )
			{
				hyper = new FastDirichletMRGParams( classEss == 0 ? 1 : (classEss / (double) period) );
				freq = DirichletMRG.DEFAULT_INSTANCE.generate( period, hyper );
				for( p = 0; p < period; p++ )
				{
					frameProbs[p] = freq[p];
					frameParams[p] = Math.log( freq[p] );
				}
				logFrameNorm = 0;
			}
			freq = new double[powers[1]];;
			for( p = 0; p < period; p++ )
			{
				for( o = 0; o <= order; o++ )
				{
					hyper = new FastDirichletMRGParams( sumOfHyperParams[o] == 0 ? 1 : (sumOfHyperParams[o] / (double) (period*params[p][o].length)) );
					for( normCounter = paramCounter = 0; normCounter < logNorm[p][o].length; normCounter++ )
					{
						logNorm[p][o][normCounter] = 0;
						DirichletMRG.DEFAULT_INSTANCE.generate( freq, 0, powers[1], hyper );
						for( len = 0; len < powers[1]; len++, paramCounter++ )
						{
							probs[p][o][paramCounter] = freq[len];
							params[p][o][paramCounter] = Math.log( freq[len] );
						}
					}
				}
			}
			setFreeParams( freeParams );
		}
	}

	protected void fromXML( StringBuffer xml ) throws NonParsableException
	{
		StringBuffer b = XMLParser.extractForTag( xml, getClass().getSimpleName() );
		length = XMLParser.extractIntForTag( b, "length" );
		alphabets = (AlphabetContainer) XMLParser.extractStorableForTag( b, "alphabets" );
		order = XMLParser.extractIntForTag( b, "order" );
		period = XMLParser.extractIntForTag( b, "period" );
		createArrays();
		classEss = XMLParser.extractDoubleForTag( b, "classEss" );
		sumOfHyperParams = XMLParser.extractDoubleArrayForTag( b, "sumOfHyperParams" );
		frameParams = XMLParser.extractDoubleArrayForTag( b, "frameParams" );
		logFrameNorm = 0;
		int j, n, index, o, p = 0;
		params = new double[period][][];
		Map<String, String> map = new TreeMap<String, String>();
		for( ; p < period; p++ )
		{
			frameProbs[p] = Math.exp( frameParams[p] );
			logFrameNorm += frameProbs[p];
			map.clear();
			map.put( "frame", ""+p );
			params[p] = XMLParser.extractDouble2ArrayAndAttributesForTag( b, "params", null, map );
			for(o = 0; o <= order; o++ )
			{
				probs[p][o] = new double[params[p][o].length];
				logNorm[p][o] = new double[powers[o]];
				for( n = index = 0; n < logNorm[p][o].length; n++ )
				{
					logNorm[p][o][n] = 0d;
					for( j = 0; j < powers[1]; j++ )
					{
						probs[p][o][index+j] = Math.exp( params[p][o][index+j] );
						logNorm[p][o][n] += probs[p][o][index+j]; 
					}
					for( j = 0; j < powers[1]; j++, index++ )
					{
						this.probs[p][o][index] /= logNorm[p][o][n]; 
					}
					logNorm[p][o][n] = Math.log( logNorm[p][o][n] );
				}
			}
		}
		for( p = 0; p < period; p++ )
		{
			frameProbs[p] /= logFrameNorm;
		}
		logFrameNorm = Math.log( logFrameNorm );
		
		plugIn = XMLParser.extractBooleanForTag( b, "plugIn" );
		optimize = XMLParser.extractBooleanForTag( b, "optimize" );
		optimizeFrame = XMLParser.extractBooleanForTag( b, "optimizeFrame" );
		starts = XMLParser.extractIntForTag( b, "starts" );
		setFreeParams( XMLParser.extractBooleanForTag( b, "freeParams" ) );
		initFrame = XMLParser.extractIntForTag( b, "initFrame" );
		computeConstantsOfLogPrior();
	}
	
	private void setFreeParams( boolean freeParams )
	{
		this.freeParams = freeParams;
		if( optimize )
		{
			for( int o, p = 0; p < period; p++ )
			{
				if( p == 0 )
				{
					offset[0][0] = optimizeFrame?period -(freeParams?1:0):0;
				}
				else
				{
					offset[p][0] = offset[p-1][order+1];
				}
				for( o = 0; o <= order; o++ )
				{
					offset[p][o+1] = offset[p][o] + params[p][o].length - (freeParams?powers[o]:0);
				}
			}
		}
		else
		{
			offset[period-1][order+1] = 0;
		}
	}

	public int getSizeOfEventSpaceForRandomVariablesOfParameter( int index )
	{
		if( index < offset[period-1][order+1] )
		{
			return powers[1];
		}
		else
		{
			throw new IndexOutOfBoundsException();
		}
	}

	public double getLogNormalizationConstant( int length )
	{
		return 0;
	}
	
	public double getLogPartialNormalizationConstant( int parameterIndex, int length ) throws Exception
	{
		if( parameterIndex < offset[period-1][order+1] )
		{
			return Double.NEGATIVE_INFINITY;
		}
		else
		{
			throw new IndexOutOfBoundsException();
		}
	}

	public double getEss()
	{
		return classEss;
	}
	
	public String toString()
	{
		DiscreteAlphabet abc =(DiscreteAlphabet) alphabets.getAlphabetAt( 0 ); 
		int i = 0, o, p, index, l = (int)abc.length();
		StringBuffer info = new StringBuffer( (int) Math.pow(l,order) * l * period * 15 );
		
		String[] sym = new String[l];
		l--;
		for( ; i <= l; i++ )
		{
			sym[i] = abc.getSymbolAt(i);
		}
		int[] context = new int[order+1];
		for( p = 0; p < period; p++ )
		{
			info.append( "frame " + p + ": p("+p+") = " + frameProbs[p] + "\n" );
			for( i = 0; i <= l; i++ )
			{
				info.append( "\t" + sym[i] );
			}
			info.append( "\n" );
			for( o = 0; o <= order; o++ )
			{
				info.append( "P(X_" + o );
				for( i = 0; i < o; i++ )
				{
					if( i == 0 )
					{
						info.append( "|" );
					}
					else
					{
						info.append( " " );
					}
					info.append( "X_" + i );
				}
				info.append( ")\n" );
				Arrays.fill( context, 0 );
				for( index = 0; index < probs[p][o].length; )
				{
					for( i = 0; i < o; i++ )
					{
						info.append( sym[context[i]] );
					}
					for( i = 0; i <= l; i++, index++ )
					{
						info.append( "\t" + probs[p][o][index] );
					}
					info.append( "\n" );
					
					i = o-1;
					while( i >= 0 && context[i] == l )
					{
						context[i] = 0;
						i--;
					}
					if( i >= 0 )
					{
						context[i]++;
					}
				}
				info.append( "\n" );
			}
		}
		return info.toString();
		/**/
	}
	
	public double getLogPriorTerm()
	{
		if( optimize )
		{
			double val = classEss*logFrameNorm, hyper, hyperSum, hyperFrame = classEss / (double) period;
			for( int j, n, index, o, p = 0; p < period; p++ )
			{
				val += hyperFrame*frameParams[p];
				for( o = 0; o <= order; o++ )
				{
					hyper = sumOfHyperParams[o] / (double) (period*params[p][o].length);
					hyperSum = powers[1] * hyper;
					for( n = index = 0; n < logNorm[p][o].length; n++ )
					{
						val -= hyperSum * logNorm[p][o][n];
						for( j = 0; j < powers[1]; j++, index++ )
						{
							val += hyper*params[p][o][index]; 
						}
					}
				}
			}
			return val + logGammaSum;
		}
		else
		{
			return 0;
		}
	}

	private void computeConstantsOfLogPrior()
	{
		double hyper = classEss / (double) period;
		logGammaSum = Gamma.logOfGamma( classEss ) - period * Gamma.logOfGamma( hyper );
		for( int o = 0; o <= order; o++ )
		{
			hyper = sumOfHyperParams[o] / (double) (period*params[0][o].length);
			
			logGammaSum += period * logNorm[0][o].length * Gamma.logOfGamma( powers[1] * hyper )
				- period * params[0][o].length * Gamma.logOfGamma( hyper );
		}
	}
	
	public void addGradientOfLogPriorTerm( double[] grad, int start )
	{
		if( optimize )
		{
			double hyper, hyperSum;
			int stop = powers[1] - (freeParams?1:0);
			int j, index, o, p = 0;
			if( optimizeFrame )
			{
				hyper = classEss / (double) period;
				j = period - (freeParams?1:0);
				for( ; p < j; p++, start++ )
				{
					grad[start] += hyper - classEss * frameProbs[p];
				}
			}
			for( p = 0; p < period; p++ )
			{
				for( o = 0; o <= order; o++ )
				{
					hyper = sumOfHyperParams[o] / (double) (period*params[p][o].length);
					hyperSum = powers[1] * hyper;
					for( index = 0; index < params[p][o].length; index += powers[1] )
					{
						for( j = 0; j < stop; j++, start++ )
						{
							grad[start] += hyper - hyperSum*probs[p][o][index+j]; 
						}
					}
				}
			}
			//System.out.println( start );System.exit(0);
		}
	}
	
	public boolean isNormalized()
	{
		return true;
	}

	public boolean isInitialized()
	{
		return true;
	}
	
	public int getNumberOfRecommendedStarts()
	{
		return starts;
	}
	
	/**
	 * This method enables the user to choose whether the parameters should be optimized or not.
	 *  
	 * @param optimize the switch for optimization of the parameters
	 */
	public void setParameterOptimization( boolean optimize )
	{
		this.optimize = optimize;
		setFreeParams( freeParams );
	}
	
	/**
	 * This method enables the user to choose whether the frame parameters should be optimized or not.
	 *  
	 * @param optimize the switch for optimization of the frame parameters
	 */
	public void setFrameParameterOptimization( boolean optimize )
	{
		this.optimizeFrame = optimize;
		setFreeParams( freeParams );
	}
	
	/*
	public void show()
	{
		for( int p = 0; p < period; p++ )
		{
			System.out.println( p + "\t" + Arrays.toString( offset[p] ) );
		}
	}*/
	
	public void setStatisticForHyperparameters( int[] length, double[] weight ) throws Exception
	{
		if( weight.length != length.length )
		{
			throw new IllegalArgumentException( "The length of both arrays (length, weight) have to be identical." );
		}
		Arrays.fill( sumOfHyperParams, 0 );
		for( int l, i = 0; i < length.length; i++ )
		{
			if( weight[i] < 0 || length[i] < 0 )
			{
				throw new IllegalArgumentException( "check length and weight for entry " + i );
			}
			else
			{
				for( l = 0; l < length[i] && l < order; l++ )
				{
					sumOfHyperParams[l] += weight[i];
				}
				if( order < length[i] )
				{
					sumOfHyperParams[order] += (length[i]-order) * weight[i];
				}
			}
		}
		computeConstantsOfLogPrior();
	}
}
