/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.algorithms.optimization.termination;

import de.jstacs.utils.Time;

/**
 * This class implements a {@link TerminationCondition} that stops an optimization
 * if the difference of the current and the last function evaluations will be small, i.e., 
 * {@latex.inline $f(\\underline{x}_{i-1}) - f(\\underline{x}_i) < \\epsilon$}.
 * 
 * @author Jens Keilwagen
 */
public class SmallDifferenceOfFunctionEvaluationsCondition implements TerminationCondition {

	private double eps;
	
	/**
	 * This constructor creates an instance that stops the optimization if the difference of the
	 * current and the last function evaluations is smaller than <code>epsilon</code>.
	 *  
	 * @param epsilon the threshold for stopping, has to be non.negative
	 */
	public SmallDifferenceOfFunctionEvaluationsCondition( double epsilon ) {
		if( epsilon <= 0 ) {
			throw new IllegalArgumentException( "The epsilon has to be non negative.");
		}
		eps = epsilon;
	}
	
	public boolean doNextIteration( int iteration, double f_last, double f_current, double[] gradient, double[] direction, double alpha,
			Time t ) {
		return eps <= f_last-f_current;
	}
}
